package com.openfin.desktop.channel;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;

import org.json.JSONObject;

import com.openfin.desktop.Ack;
import com.openfin.desktop.AckListener;

public class ChannelClient extends ChannelBase {

	private CopyOnWriteArrayList<ChannelListener> channelListeners;
	private AbstractProtocolHandler protocolHandler;

	ChannelClient(AbstractProtocolHandler protocolHandler, EndpointIdentity endpointIdentity) {
		super(endpointIdentity);
		this.protocolHandler = protocolHandler;
		this.channelListeners = new CopyOnWriteArrayList<>();
	}

	/**
	 * Dispatch the given action to the channel provider.
	 * 
	 * @param action
	 *            Name of the action to be invoked by the channel provider.
	 * @param actionPayload
	 *            Payload to be sent along with the action.
	 * @param ackListener
	 *            AckListener for the request
	 */
	public void dispatch(String action, JSONObject actionPayload, AckListener ackListener) {
		super.dispatch(this.protocolHandler, endpointIdentity.toJSON(), action, actionPayload, ackListener);
	}

	/**
	 * Dispatch an action to a specified client.
	 *
	 * @param action Name of the action to be invoked by the client.
	 * @param actionPayload Payload to be sent along with the action.
	 * @return CompletableFuture with Ack object
	 * @see Ack
	 */
	public CompletableFuture<Ack> dispatchAsync(String action, Object actionPayload) {
		return super.dispatchAsync(this.protocolHandler, endpointIdentity.toJSON(), action, actionPayload);
	}
	
	/**
	 * Register an action to be called by ChannelProvider
	 * @param action Name of the action to be invoked by the channel provider
	 * @param listener Function representing the action to be taken on a client dispatch.
	 * @return if the action was registered successfully
	 */
	public boolean register(String action, ChannelAction listener) {
		return super.register(action, listener);
	}
	
	/**
	 * Disconnect from the channel.
	 *
	 * @return CompletableFuture with Ack object
	 * @see Ack
	 */
	public CompletableFuture<Ack> disconnect() {
		logger.debug("disconnecting {} {}", this.getChannelName(), this.getEndpointId());
		CompletableFuture<Ack> future = this.protocolHandler.getChannel().disconnect(this);
		future.thenAccept(ack -> {
			ChannelClient.this.cleanup();
		});
		return future;
	}
	
	public boolean addChannelListener(ChannelListener listener) {
		return this.channelListeners.add(listener);
	}

	public boolean removeChannelListener(ChannelListener listener) {
		return this.channelListeners.remove(listener);
	}
	
	void fireChannelConnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelConnect(event);
		}
	}

	void fireChannelDisconnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelDisconnect(event);
		}
	}

	@Override
	protected void cleanup() {
		super.cleanup();
		this.channelListeners.clear();
	}
}
