package com.openfin.desktop.channel;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import com.openfin.desktop.*;
import org.json.JSONObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChannelProvider extends ChannelBase {
	private final static Logger logger = LoggerFactory.getLogger(ChannelProvider.class.getName());

	private final Channel channel;
	private final List<ProtocolOptions> protocolOptionsList;  // supported protocols by this provider
	private final ConcurrentHashMap<String, EndpointIdentity> clientMap;
	private final ConcurrentHashMap<String, AbstractProtocolHandler> protocolHandlerMap;  // client endpoint Id -> protocol handler
	private final CopyOnWriteArrayList<ChannelProviderListener> providerListeners;

	final private static String CLIENT_DISCONNECTED_EVENT = "client-disconnected";  // fired when a channel client disconnects

	ChannelProvider(Channel channel, List<ProtocolOptions> protocolOptionsList, EndpointIdentity endpointIdentity) {
		super(endpointIdentity);
		this.channel = channel;
		this.protocolOptionsList = protocolOptionsList;
		this.clientMap = new ConcurrentHashMap<>();
		this.protocolHandlerMap = new ConcurrentHashMap<>();
		this.providerListeners = new CopyOnWriteArrayList<>();
		this.channel.addChannelListener(new ChannelListener() {
			@Override
			public void onChannelConnect(ConnectionEvent connectionEvent) {
			}

			@Override
			public void onChannelDisconnect(ConnectionEvent connectionEvent) {
				processDisconnection(connectionEvent.getEndpointId());
			}
		});

		JSONObject eventListenerPayload = new JSONObject();
		eventListenerPayload.put("topic", "channel");
		eventListenerPayload.put("type", CLIENT_DISCONNECTED_EVENT);
		this.channel.getDesktopConnection().addEventCallback(eventListenerPayload, new EventListener() {
			@Override
			public void eventReceived(ActionEvent actionEvent) {
				JSONObject event = actionEvent.getEventObject();
				if (ChannelProvider.this.channel.getName().equals(event.getString("channelName"))) {
					ChannelProvider.this.processDisconnection(event.getString("endpointId"));
				}
			}
		}, null, this);
	}
	
	/**
	 * Destroy the channel
	 * @param ackListener AckListener for the request
	 */
	public void destroy(AckListener ackListener) {
		this.channel.destroy(this, ackListener);
	}

	public void processConnection(JSONObject connectionPayload, JSONObject ackToSenderPayload) throws Exception {
		JSONObject clientIdentity = connectionPayload.getJSONObject("clientIdentity");
		String clientEndpointId = clientIdentity.optString("endpointId");
		if (Objects.nonNull(clientEndpointId)) {
			JSONObject offer = connectionPayload.optJSONObject("offer");
			ProtocolOptions protocolOptions = this.selectRequestedProtocol(offer);
			if (Objects.isNull(protocolOptions)) {
				logger.error("requested channel protocol is not supported");
				ackToSenderPayload.put("success", false);
				return;
			}
			EndpointIdentity clientEndpointIdentity = new EndpointIdentity(clientIdentity);

			this.fireClientConnectEvent(new ChannelClientConnectEvent(clientEndpointIdentity.getChannelId(), clientEndpointIdentity.getUuid(),
					clientEndpointIdentity.getName(), clientEndpointIdentity.getChannelName(),
					clientEndpointIdentity.getEndpointId(), connectionPayload.opt("payload")));

			this.clientMap.put(clientEndpointId, clientEndpointIdentity);
			this.channel.fireChannelConnectEvent(new ChannelClientConnectEvent(clientEndpointIdentity.getChannelId(), clientEndpointIdentity.getUuid(),
					clientEndpointIdentity.getName(), clientEndpointIdentity.getChannelName(),
					clientEndpointIdentity.getEndpointId(), connectionPayload.has("payload") ? connectionPayload.get("payload") : null));
			AbstractProtocolHandler protocolHandler = AbstractProtocolHandler.createProtocolHandler(protocolOptions, this.channel);
			protocolHandler.initializeProvider(this.getEndpointIdentity(), clientEndpointIdentity);
			JSONObject answer = protocolHandler.processConnectOffer(offer);
			this.protocolHandlerMap.put(clientEndpointId, protocolHandler);
			if (Objects.nonNull(answer)) {
				JSONObject ackPayload = ackToSenderPayload.getJSONObject("payload");
				ackPayload.put("answer", answer);
			} else {
				ackToSenderPayload.put("success", false);
			}
		} else {
			logger.warn("Missing client endpoint id");
		}
	}

	private ProtocolOptions selectRequestedProtocol(JSONObject offer) {
		ProtocolOptions options = Channel.CLASSIC_PROTOCOL;
		if (Objects.nonNull(offer)) {
			options = null;  // if "offer" is present, has to be supported
			// prefer RTC_PROTOCOL over CLASSIC_PROTOCOL if both are specified
			if (Objects.nonNull(AbstractProtocolHandler.getOfferProtocolByType(offer, Channel.RTC_PROTOCOL)) &&
					this.protocolOptionsList.contains(Channel.RTC_PROTOCOL)) {
				options = Channel.RTC_PROTOCOL;
			} else if (Objects.nonNull(AbstractProtocolHandler.getOfferProtocolByType(offer, Channel.CLASSIC_PROTOCOL)) &&
					this.protocolOptionsList.contains(Channel.CLASSIC_PROTOCOL)) {
				options = Channel.CLASSIC_PROTOCOL;
			}
		}
		logger.debug("selected channel protocol {}", Objects.nonNull(options)?options.getName():"");
		return options;
	}

	private void processDisconnection(String endpointId) {
		logger.debug(String.format("Client disconnected %s from channel %s", endpointId, this.channel.getName()));
		EndpointIdentity endpointIdentity = this.clientMap.remove(endpointId);
		AbstractProtocolHandler protocolHandler = this.protocolHandlerMap.remove(endpointId);
		if (Objects.nonNull(protocolHandler)) {
			protocolHandler.cleanup();
		}
		this.fireClientDisconnectEvent(new ChannelClientConnectEvent(endpointIdentity.getChannelId(), endpointIdentity.getUuid(),
				endpointIdentity.getName(), endpointIdentity.getChannelName(),
				endpointIdentity.getEndpointId(), null));
	}

	/**
	 * Publish an action and payload to every connected client.
	 * @param action Name of the action to be invoked by the channel client
	 * @param actionPayload Payload to be sent along with the action.
	 * @param ackListener AckListener for the request
	 * @deprecated Use publishAsync
	 */
	public void publish(String action, JSONObject actionPayload, AckListener ackListener) {
		for (EndpointIdentity client : this.clientMap.values()) {
			//the logic in core, if destinationId has channelId, it means the message is intended for channel provider
			//here create the new endpointIdentity that has no channelId
			EndpointIdentity destId = new EndpointIdentity(client.getChannelName(), null, client.getUuid(), client.getName(), client.getEndpointId());
			AbstractProtocolHandler protocolHandler = this.protocolHandlerMap.get(client.getEndpointId());
			if (Objects.nonNull(protocolHandler)) {
				this.dispatch(protocolHandler, destId.toJSON(), action, actionPayload, ackListener);
			} else {
				logger.warn("Missing protocol handler for publish {} {}", client.getUuid(), client.getEndpointId());
			}
		}
	}

	/**
	 *
	 * Publish an action and payload to every connected client.
	 * @param action Name of the action to be invoked by the channel client
	 * @param actionPayload Payload to be sent along with the action.
	 * @return A list of CompletableFuture with Ack object
	 * @see Ack
	 */
	public List<CompletableFuture<Ack>> publishAsync(String action, JSONObject actionPayload) {
		List<CompletableFuture<Ack>> list = new ArrayList<>();
		for (EndpointIdentity client : this.clientMap.values()) {
			//the logic in core, if destinationId has channelId, it means the message is intended for channel provider
			//here create the new endpointIdentity that has no channelId
			EndpointIdentity destId = new EndpointIdentity(client.getChannelName(), null, client.getUuid(), client.getName(), client.getEndpointId());
			AbstractProtocolHandler protocolHandler = this.protocolHandlerMap.get(client.getEndpointId());
			if (Objects.nonNull(protocolHandler)) {
				list.add(this.dispatchAsync(protocolHandler, destId.toJSON(), action, actionPayload));
			} else {
				logger.warn("Missing protocol handler for publish {} {}", client.getUuid(), client.getEndpointId());
			}
		}
		return list;
	}


	/**
	 * Dispatch an action to a specified client.
	 *
	 * @param destinationIdentity Identity of the target client.
	 * @param action Name of the action to be invoked by the client.
	 * @param actionPayload Payload to be sent along with the action.
	 * @return CompletableFuture with Ack object
	 * @see Ack
	 */
	public CompletableFuture<Ack> dispatchAsync(JSONObject destinationIdentity, String action, JSONObject actionPayload) {
		AbstractProtocolHandler protocolHandler = this.protocolHandlerMap.get(destinationIdentity.getString("endpointId"));
		if (Objects.nonNull(protocolHandler)) {
			return dispatchAsync(protocolHandler, destinationIdentity, action, actionPayload);
		} else {
			CompletableFuture<Ack> ackFuture = new CompletableFuture<>();
			ackFuture.complete(DesktopUtils.getNack(this, "Invalid destinationIdentity"));
			return ackFuture;
		}
	}

	/**
	 * Dispatch an action to a specified client.
	 *
	 * @param destinationIdentity Identity of the target client.
	 * @param action Name of the action to be invoked by the client.
	 * @param actionPayload Payload to be sent along with the action.
	 * @param ackListener AckListener for the request
	 */
	public void dispatch(JSONObject destinationIdentity, String action, JSONObject actionPayload,
							AckListener ackListener) {
		AbstractProtocolHandler protocolHandler = this.protocolHandlerMap.get(destinationIdentity.getString("endpointId"));
		if (Objects.nonNull(protocolHandler)) {
			super.dispatch(protocolHandler, destinationIdentity, action, actionPayload, ackListener);
		} else {
			DesktopUtils.errorAck(ackListener, DesktopUtils.getNack(this, "Invalid destinationIdentity"));
		}
	}

	/**
	 * Register an action to be called by ChannelClient
	 * @param action Name of the action to be invoked by the channel client
	 * @param listener Function representing the action to be taken on a client dispatch.
	 * @return if the action was registered successfully
	 */
	public boolean register(String action, ChannelAction listener) {
		return super.register(action, listener);
	}

	@Override
	protected void cleanup() {
		super.cleanup();
		this.clientMap.clear();
		this.protocolHandlerMap.clear();
	}

	/**
	 * Add a listener for channel provider events
	 * @param listener listener to add
	 * @return
	 */
	public boolean addProviderListener(ChannelProviderListener listener) {
		return this.providerListeners.add(listener);
	}

	/**
	 * Add a ChannelProviderListener
	 * @param listener listener to remove
	 * @return
	 */
	public boolean removeProviderListener(ChannelProviderListener listener) {
		return this.providerListeners.remove(listener);
	}

	protected void fireClientConnectEvent(ChannelClientConnectEvent event) throws Exception {
		for (ChannelProviderListener listener : this.providerListeners) {
			listener.onClientConnect(event);
		}
	}

	protected void fireClientDisconnectEvent(ChannelClientConnectEvent event) {
		for (ChannelProviderListener listener : this.providerListeners) {
			listener.onClientDisconnect(event);
		}
	}

}
