package com.openfin.desktop.channel.webrtc;

import com.openfin.desktop.*;
import com.openfin.desktop.channel.AbstractProtocolHandler;
import com.openfin.desktop.channel.Channel;
import com.openfin.desktop.channel.EndpointIdentity;
import com.openfin.desktop.channel.ProtocolOptions;
import dev.onvoid.webrtc.*;
import dev.onvoid.webrtc.media.MediaStream;
import org.json.JSONObject;

import java.security.spec.ECField;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import static java.util.Objects.nonNull;

public class WebRtcProtocolHandler extends AbstractProtocolHandler implements PeerConnectionObserver, EventListener, DataChannelListener {
    private String connectionId;
    private String iceClientEventType;    // event type for receiving ice candidates from channel client
    private String iceProviderEventType;  // event type for receiving ice candidates from channel provider
    private String sendIceEventType;      // even type for sending ice candidate.  iceClientEventType for channel client, iceProviderEventType for provider
    private final PeerConnectionFactory factory;
    protected RTCPeerConnection peerConnection;
    private DataChannel requestChannel;
    private DataChannel responseChannel;
    private CompletableFuture<Boolean> clientConnectFuture;  // for channel client
    private final AtomicLong messageId = new AtomicLong(0);
    private final Map<String, CompletableFuture<Ack>> responseMap;

    private static final String SDPAnswer = "answer";
    private static final String SDPOffer = "offer";
    private static final String SDPCandidate = "candidate";
    private static final String SDPTrickleReady = "trickle";

    private static final String REQUEST_CHANNEL_NAME = "request";
    private static final String RESPONSE_CHANNEL_NAME = "response";
    private static final String MESSAGE_ID_KEY = "messageId";

    public WebRtcProtocolHandler(Channel channel) {
        super(channel);
        this.factory = new PeerConnectionFactory();

        this.responseMap = new ConcurrentHashMap<>();
    }

    @Override
    public ProtocolOptions getProtocolOptions() {
        return Channel.RTC_PROTOCOL;
    }

    @Override
    public void initializeClient() throws Exception {
        this.connectionId = UUID.randomUUID().toString();
        logger.debug("Initializing client {} {}", this.channel.getName(), this.connectionId);
        this.iceProviderEventType = String.format("ice-provider-%s", this.connectionId);
        this.sendIceEventType = String.format("ice-client-%s", this.connectionId);;
        // listenForProviderIce
        this.channel.addEventListener(iceProviderEventType, this, null);
        this.createPeerConnection();
        this.createOffer();
    }

    private void createPeerConnection() {
        logger.debug("Creating peer connection {} {}", this.getChannel().getName(), Objects.nonNull(this.clientEndpointIdentity)?this.clientEndpointIdentity.getEndpointId():"");
        RTCConfiguration config = new RTCConfiguration();
        this.peerConnection = factory.createPeerConnection(config, this);

        if (!isProvider) {
            // request and response channels are created by channel client
            this.requestChannel = new DataChannel(this.peerConnection, REQUEST_CHANNEL_NAME);
            this.requestChannel.addChannelListener(this);
            this.responseChannel = new DataChannel(this.peerConnection, RESPONSE_CHANNEL_NAME);
            this.responseChannel.addChannelListener(this);
        }
    }

    private void createOffer() throws Exception {
        CreateDescObserver createObserver = new CreateDescObserver();
        SetDescObserver setObserver = new SetDescObserver();
        peerConnection.createOffer(new RTCOfferOptions(), createObserver);
        RTCSessionDescription offerDesc = createObserver.get();
        logger.debug("setting local description {} {}", this.channel.getName(), offerDesc.toString());
        peerConnection.setLocalDescription(offerDesc, setObserver);
        setObserver.get();
    }
    private JSONObject getOfferPayload() {
        var offer = this.peerConnection.getLocalDescription();
        JSONObject payload = new JSONObject();
        payload.put("type", SDPOffer);
        payload.put("sdp", offer.sdp);
        logger.debug("OFFER {}", payload.toString());
        return payload;
    }

    @Override
    protected JSONObject getSupportedOfferProtocol() {
        JSONObject prot = super.getSupportedOfferProtocol();
        prot.put("type", Channel.RTC_PROTOCOL.getName());
        prot.put("version", Channel.RTC_PROTOCOL.getVersion());
        JSONObject payload = new JSONObject();
        payload.put("rtcConnectionId", this.connectionId);
        payload.put("offer", this.getOfferPayload());
        prot.put("payload", payload);
        return prot;
    }

    /**
     * Accept answer from a channel provider
     *
     * @param answer answer object
     * @param future future for returning true if accepted
     * @throws Exception Exception
     */
    @Override
    protected void acceptAnswer(JSONObject answer, CompletableFuture<Boolean> future) throws Exception {
        logger.debug(String.format("accepting answer %s %s", this.channel.getName(), answer.toString()));
        this.clientConnectFuture = future;
        this.setRemoteDescription(new RTCSessionDescription(RTCSdpType.ANSWER, answer.getString("sdp")));
    }

    private void setRemoteDescription(RTCSessionDescription description) throws Exception {
        logger.debug("setting remote description {}", description.toString());
        SetDescObserver setObserver = new SetDescObserver();
        peerConnection.setRemoteDescription(description, setObserver);
        setObserver.get();
    }

    @Override
    protected void acceptOffer(JSONObject payload) throws Exception {
        logger.debug("accepting offer {} {}", this.channel.getName(), payload.toString());
        JSONObject offer = payload.getJSONObject("offer");
        this.connectionId = payload.getString("rtcConnectionId");
        logger.debug(String.format("receiving provider connection Id %s", this.connectionId));
        this.iceClientEventType = String.format("ice-client-%s", this.connectionId);
        this.sendIceEventType = String.format("ice-provider-%s", this.connectionId);

        this.channel.addEventListener(iceClientEventType, this, null);
        this.createPeerConnection();

        this.setRemoteDescription(new RTCSessionDescription(RTCSdpType.OFFER, offer.getString("sdp")));
        this.createAnswer();
    }

    protected JSONObject getSupportedAnswerProtocol() {
        JSONObject prot = super.getSupportedAnswerProtocol();
        prot.put("type", Channel.RTC_PROTOCOL.getName());
        prot.put("version", Channel.RTC_PROTOCOL.getVersion());
        JSONObject payload = new JSONObject();
        RTCSessionDescription desc = this.peerConnection.getLocalDescription();
        logger.debug("getting local description {}", desc.toString());
        JSONObject answerJson = new JSONObject();
        answerJson.put("type", SDPAnswer);
        answerJson.put("sdp", desc.sdp);
        payload.put("answer", answerJson);
        prot.put("payload", payload);
        return prot;
    }

    protected RTCSessionDescription createAnswer() throws Exception {
        CreateDescObserver createObserver = new CreateDescObserver();
        SetDescObserver setObserver = new SetDescObserver();
        peerConnection.createAnswer(new RTCAnswerOptions(), createObserver);
        RTCSessionDescription answerDesc = createObserver.get();
        logger.debug("setting local description {} {}", this.channel.getName(), answerDesc.toString());
        peerConnection.setLocalDescription(answerDesc, setObserver);
        setObserver.get();
        return answerDesc;
    }


    @Override
    public void sendChannelMessage(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
                            JSONObject actionPayload, AckListener ackListener) {
        String error = "sendChannelMessage not deprecated.  use sendChannelMessageAsync instead";
        logger.error(error);
        DesktopUtils.errorAck(ackListener, DesktopUtils.getNack(this, error));
    }

    @Override
    public CompletableFuture<Ack> sendChannelMessageAsync(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
                                                   Object actionPayload) {
        CompletableFuture<Ack> ackFuture = new CompletableFuture<>();

        try {
            String messageId = String.format("javartc-%d", this.getNextMessageId());
            JsonBean msg = new JsonBean();
            msg.setString("action", action);
            msg.setString(MESSAGE_ID_KEY, messageId);
            msg.put("payload", actionPayload);
            String data = msg.toString();
            logger.debug("sending message: {} {}", this.channel.getName(), data);
            this.requestChannel.send(data);
            this.responseMap.put(messageId, ackFuture);
        } catch (Exception ex) {
            logger.error("Error sending message {}", this.channel.getName(), ex);
        }
        return ackFuture;
    }

    @Override
    public void onSignalingChange(RTCSignalingState state) {
        logger.debug("onSignalingChange {}", state);
        PeerConnectionObserver.super.onSignalingChange(state);
    }

    @Override
    public void onConnectionChange(RTCPeerConnectionState state) {
        try {
            logger.debug("onConnectionChange {} {} {}", this.channel.getName(), state, nonNull(this.clientEndpointIdentity) ? this.clientEndpointIdentity.getEndpointId() : "");
            PeerConnectionObserver.super.onConnectionChange(state);
            if (state == RTCPeerConnectionState.CONNECTED) {
            }
        } catch (Exception ex) {
            logger.error("onConnectionChange", ex);
        }
    }

    @Override
    public void onIceConnectionChange(RTCIceConnectionState state) {
        logger.debug("onIceConnectionChange {} {}", this.channel.getName(), state);
        PeerConnectionObserver.super.onIceConnectionChange(state);
    }

    @Override
    public void onStandardizedIceConnectionChange(RTCIceConnectionState state) {
        logger.debug("onStandardizedIceConnectionChange {}", state);
        PeerConnectionObserver.super.onStandardizedIceConnectionChange(state);
    }

    @Override
    public void onIceConnectionReceivingChange(boolean receiving) {
        logger.debug("onIceConnectionReceivingChange {}", receiving);
        PeerConnectionObserver.super.onIceConnectionReceivingChange(receiving);
    }

    @Override
    public void onIceGatheringChange(RTCIceGatheringState state) {
        logger.debug("onIceGatheringChange {}", state);
        PeerConnectionObserver.super.onIceGatheringChange(state);
    }

    @Override
    public void onIceCandidate(RTCIceCandidate rtcIceCandidate) {
        try {
            logger.debug("onIceCandidate {} {}", this.channel.getName(), rtcIceCandidate.toString());
            JSONObject candidateJson = new JSONObject();
            candidateJson.put(SDPCandidate, rtcIceCandidate.sdp);
            candidateJson.put("sdpMid", rtcIceCandidate.sdpMid);
            candidateJson.put("sdpMLineIndex", rtcIceCandidate.sdpMLineIndex);
            if (nonNull(rtcIceCandidate.serverUrl)) {
                candidateJson.put("serverUrl", rtcIceCandidate.serverUrl);
            }
            JSONObject payload = new JSONObject();
            payload.put("candidate", candidateJson);
            this.channel.raiseEvent(this.sendIceEventType, payload).thenAccept(ack -> {
                if (!ack.isSuccessful()) {
                    logger.error(String.format("Error sending client candidate %s", ack.getReason()));
                }
            });
        } catch (Exception ex) {
            logger.error("Error sending client candidate",ex);
        }
    }

    @Override
    public void onIceCandidateError(RTCPeerConnectionIceErrorEvent event) {
        logger.info("onIceCandidateError {} {}", this.channel.getName(), event.toString());
        PeerConnectionObserver.super.onIceCandidateError(event);
    }

    @Override
    public void onIceCandidatesRemoved(RTCIceCandidate[] candidates) {
        logger.debug("onIceCandidatesRemoved");
        PeerConnectionObserver.super.onIceCandidatesRemoved(candidates);
    }

    @Override
    public void onAddStream(MediaStream stream) {
        PeerConnectionObserver.super.onAddStream(stream);
    }

    @Override
    public void onRemoveStream(MediaStream stream) {
        PeerConnectionObserver.super.onRemoveStream(stream);
    }

    @Override
    public void onDataChannel(RTCDataChannel dataChannel) {
        try {
            logger.debug("onDataChannel {} {}", this.channel.getName(), dataChannel.getLabel());
            PeerConnectionObserver.super.onDataChannel(dataChannel);
            if (REQUEST_CHANNEL_NAME.equals(dataChannel.getLabel())) {
                this.requestChannel = new DataChannel(dataChannel);
                this.requestChannel.addChannelListener(this);
            } else if (RESPONSE_CHANNEL_NAME.equals(dataChannel.getLabel())) {
                this.responseChannel = new DataChannel(dataChannel);
                this.responseChannel.addChannelListener(this);
            }
        } catch (Exception ex) {
            logger.error("onDataChannel", ex);
        }
    }

    @Override
    public void onRenegotiationNeeded() {
        logger.debug("onRenegotiationNeeded");
        PeerConnectionObserver.super.onRenegotiationNeeded();
    }

    @Override
    public void onAddTrack(RTCRtpReceiver receiver, MediaStream[] mediaStreams) {
        PeerConnectionObserver.super.onAddTrack(receiver, mediaStreams);
    }

    @Override
    public void onRemoveTrack(RTCRtpReceiver receiver) {
        PeerConnectionObserver.super.onRemoveTrack(receiver);
    }

    @Override
    public void onTrack(RTCRtpTransceiver transceiver) {
        PeerConnectionObserver.super.onTrack(transceiver);
    }

    @Override
    public void eventReceived(ActionEvent actionEvent) {
        if (actionEvent.getEventObject().has("candidate")) {
            try {
                JSONObject candidateJson = actionEvent.getEventObject().getJSONObject("candidate");
                logger.debug(String.format("Adding remote candidate %s %s", this.channel.getName(), candidateJson.toString()));
                RTCIceCandidate candidate = new RTCIceCandidate(candidateJson.getString("sdpMid"),
                        candidateJson.getInt("sdpMLineIndex"),
                        candidateJson.getString("candidate"),
                        candidateJson.optString("serverUrl"));
                this.peerConnection.addIceCandidate(candidate);
            } catch (Exception ex) {
                logger.error("Error accepting remote ice candidate", ex);
            }
        }
    }

    @Override
    public void onStateChange(DataChannel source, State state) {
        try {
            logger.debug("onStateChange {} {} {} {}", this.channel.getName(), nonNull(this.clientEndpointIdentity) ? this.clientEndpointIdentity.getEndpointId() : "", source.getName(), state);
            if (nonNull(this.requestChannel) && nonNull(this.responseChannel)) {
                if (this.requestChannel.getState() == State.OPEN && this.responseChannel.getState() == State.OPEN) {
                    if (Objects.nonNull(this.clientConnectFuture)) {
                        logger.debug(String.format("webrtc handler connected %s %s", this.channel.getName(), this.connectionId));
                        this.clientConnectFuture.complete(true);
                        this.clientConnectFuture = null;
                    }
                }
            }
        } catch (Exception ex) {
            logger.error("onStateChange", ex);
        }
    }

    @Override
    public void onMessage(DataChannel source, String message) {
        try {
            JSONObject json = new JSONObject(message);
            logger.debug("received message in {} {}: {}", this.channel.getName(), source.getName(), message);
            if (source.getName().equals(REQUEST_CHANNEL_NAME)) {
                this.processChannelRequest(json);
            } else if (source.getName().equals(RESPONSE_CHANNEL_NAME)) {
                this.processChannelResponse(json);
            } else {
                logger.error("received message from invalid data channel {}", source.getName());
            }
        } catch (Exception ex) {
            logger.error("Error processing {}", message);
        }
    }

    private void processChannelRequest(JSONObject request) throws Exception {
        String messageId = request.getString(MESSAGE_ID_KEY);
        JSONObject resp = new JSONObject();
        resp.put("messageId", messageId);
        try {
            String action = request.getString("action");
            Object actionPayload = request.opt("payload");
            EndpointIdentity targetEndpoint = this.isProvider?this.endpointIdentity:this.clientEndpointIdentity;
            Object actionResult = channel.invokeAction(targetEndpoint, action, actionPayload, this.clientEndpointIdentity.toJSON());
            resp.put("payload", actionResult);
            resp.put("success", true);
        } catch (Exception ex) {
            logger.error("Error processing channel request", ex);
            resp.put("success", false);
            resp.put("error", ex.getMessage());
        }
        this.responseChannel.send(resp.toString());
    }
    private void processChannelResponse(JSONObject response) throws Exception{
        String respMsgId = response.optString(MESSAGE_ID_KEY, "");
        CompletableFuture<Ack> ackCompletableFuture = this.responseMap.get(respMsgId);
        if (Objects.nonNull(ackCompletableFuture)) {
            this.responseMap.remove(respMsgId);
            boolean success = response.optBoolean("success", false);
            JSONObject error = response.optJSONObject("error");
            JSONObject payload = response.optJSONObject("payload");
            JSONObject ackObj = new JSONObject();
            ackObj.put("success", success);
            ackObj.put("reason", error);
            JSONObject data = new JSONObject();
            data.put("result", payload);
            ackObj.put("data", data);
            ackCompletableFuture.complete(new Ack(ackObj, this));
        } else {
            logger.error("received response with invalid message ID {}", respMsgId);
        }
    }

    @Override
    protected void cleanup() {
        super.cleanup();
        try {
            if (Objects.nonNull(this.iceProviderEventType)) {
                this.channel.removeEventListener(iceProviderEventType, this, null);
                this.iceProviderEventType = null;
            }
            if (Objects.nonNull(this.iceClientEventType)) {
                this.channel.removeEventListener(iceClientEventType, this, null);
                this.iceClientEventType = null;
            }
            if (Objects.nonNull(this.requestChannel)) {
                this.requestChannel.removeChannelListener(this);
                this.requestChannel.close();
            }
            if (Objects.nonNull(this.responseChannel)) {
                this.responseChannel.removeChannelListener(this);
                this.responseChannel.close();
            }
            if (Objects.nonNull(this.peerConnection)) {
                this.peerConnection.close();
            }
        } catch (Exception ex) {
            logger.error("Error cleaing up", ex);
        } finally {
            this.requestChannel = null;
            this.responseChannel = null;
            this.peerConnection = null;
        }
    }

    private long getNextMessageId() {
        return this.messageId.getAndIncrement();
    }
}
