package com.openfin.desktop.interop;

import java.util.concurrent.CompletionStage;

import com.openfin.desktop.ClientIdentity;
import com.openfin.desktop.Identity;

/**
 * Interop Client
 * @author Anthony
 *
 */
public interface InteropClient {
	
	/**
	 * Sets a context for the context group of the current entity.
	 * @param context New context to set.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> setContext(Context context);
	
	/**
	 * Adds a context listener for incoming context.
	 * @param contextListener The listener for incoming context.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> addContextListener(ContextListener contextListener);
	
	/**
	 * Adds a context listener for incoming context of specified context type.
	 * @param contextType The context type to listen to.
	 * @param contextListener The listener for incoming context.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> addContextListener(String contextType, ContextListener contextListener);
	
	/**
	 * Removes the context listener.
	 * @param contextListener The listener to be removed.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> removeContextListener(ContextListener contextListener);
	
	/**
	 * Remove the context listener of specified context type.
	 * @param contextType The context type that the listener is listened to.
	 * @param contextListener The listener to be removed.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> removeContextListener(String contextType, ContextListener contextListener);
	
	/**
	 * Gets the Interop-Broker-defined context groups available for an entity to join.
	 * @return A new CompletionStage for the the array of the context group info.
	 */
	public CompletionStage<ContextGroupInfo[]> getContextGroups();
	
	/**
	 * Joins this InteropClient to the context group.
	 * @param contextGroupId The id of context group.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> joinContextGroup(String contextGroupId);
	
	/**
	 * Joins specified InteropClient to the context group.
	 * @param contextGroupId The id of context group.
	 * @param target The identity of the target InteropClient.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> joinContextGroup(String contextGroupId, Identity target);

	/**
	 * Removes this InteropClient from the context group it currently joined.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> removeFromContextGroup();
	
	/**
	 * Removes specified InteropClient from the context group it currently joined.
	 * @param target The identity of the target InteropClient.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> removeFromContextGroup(Identity target);
	
	/**
	 * Gets all clients for a context group.
	 * @param contextGroupId  The id of context group.
	 * @return A new CompletionStage for the the array of all clients in the context group.
	 */
	public CompletionStage<ClientIdentity[]> getAllClientsInContextGroup(String contextGroupId);
	
	/**
	 * Gets display info for a context group.
	 * @param contextGroupId The id of context group.
	 * @return A new CompletionStage for the context group info.
	 */
	public CompletionStage<ContextGroupInfo> getInfoForContextGroup(String contextGroupId);


	/**
	 * Sends an intent to the Interop Broker to resolve.
	 * @param intent New intent to send.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> fireIntent(Intent intent);

	/**
	 * Register an intent listener for incoming intent.
	 * @param intentName name of the intent
	 * @param intentListener The listener for incoming intent.
	 * @return A new CompletionStage for the task.
	 */
	public CompletionStage<Void> registerIntentListener(String intentName, IntentListener intentListener);

}
