package com.openfin.desktop.notifications.events;

import org.json.JSONObject;

import com.openfin.desktop.notifications.CustomData;
import com.openfin.desktop.notifications.NotificationActionResult;
import com.openfin.desktop.notifications.NotificationSource;
import com.openfin.desktop.notifications.NotificationSourceDesktop;
import com.openfin.desktop.notifications.NotificationSourceFeed;

/**
 * Event fired when an action is raised for a notification due to a specified
 * trigger.
 * 
 * @author Anthony
 *
 */
public class NotificationActionEvent extends NotificationEvent {

	private NotificationSource source;
	private NotificationActionResult result;

	public NotificationActionEvent(JSONObject eventPayload) {
		super(eventPayload);
	}

	/**
	 * Indicates what triggered this action.
	 * 
	 * Trigger can be one of the following:
	 * 
	 * <p>
	 * "close": The notification was closed, either by user interaction,
	 * programmatically by an application, or by the notification expiring.
	 * </p>
	 * 
	 * <p>
	 * "control": The user interacted with one of the controls within the
	 * notification. This currently means a button click, but other control types
	 * will be added in future releases.
	 * </p>
	 * 
	 * <p>
	 * "expire": The notification expired.
	 * </p>
	 * 
	 * <p>
	 * "programmatic": The action was triggered programmatically by an application.
	 * Not currently supported - will be implemented in a future release.
	 * </p>
	 * 
	 * <p>
	 * "select": The user clicked the body of the notification itself. Any clicks of
	 * the notification that don't hit a control or the close button will fire an
	 * event with the 'select' action trigger.
	 * </p>
	 * 
	 * @return The trigger of this action.
	 */
	public String getTrigger() {
		return this.getString("trigger");
	}

	/**
	 * Notifications can be created by both desktop applications and as push
	 * notifications from a notification feed.
	 * 
	 * This method allows the application handling the action to identify where
	 * this notification originated.
	 * 
	 * @return The source of the notification.
	 */
	public NotificationSource getSource() {
		if (this.source == null) {
			JSONObject sourceJson = this.getJsonValue("source");
			if (sourceJson != null) {
				String type = sourceJson.getString("type");
				if (type != null && "desktop".equals(type)) {
					source = new NotificationSourceDesktop(sourceJson);
				}
				else if (type != null && "feed".equals(type)) {
					source = new NotificationSourceFeed(sourceJson);
				}
			}
		}
		return this.source;
	}

	/**
	 * Get application-defined metadata that this event is passing back to the application.
	 * @return Application-defined metadata.
	 */
	public NotificationActionResult getResult() {
		if (this.result == null && this.json.has("result") && !this.json.isNull("result")) {
			this.result = new NotificationActionResult(this.json.getJSONObject("result"));
		}
		return this.result;
	}
}
