package com.openfin.desktop.snapshot;

import com.openfin.desktop.Ack;
import com.openfin.desktop.AckListener;
import com.openfin.desktop.DesktopConnection;
import com.openfin.desktop.channel.ChannelClient;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 * SnapshotSourceClient connects to an SnapshotSourceProvider, which allows getting and applying snapshots.
 */
public class SnapshotSourceClient {
    private final static Logger logger = LoggerFactory.getLogger(SnapshotSourceClient.class.getName());

    private final DesktopConnection desktopConnection;
    private final String uuid;
    private ChannelClient channelClient;

    /**
     * Constructor of SnapshotSourceClient
     * @param desktopConnection an instance of DesktopConnection
     * @param uuid uuid of snapshot provider
     */
    public SnapshotSourceClient(DesktopConnection desktopConnection, String uuid) {
        this.desktopConnection = desktopConnection;
        this.uuid = uuid;
    }

    /**
     * Gets current snapshot from the snapshot provider.
     *
     * @return Ack for the request
     * @see Ack
     */
    public CompletableFuture<JSONObject> getSnapshotAsync() {
        return this.channelClient.dispatchAsync("get-snapshot", JSONObject.NULL).thenApply(ack -> {
            if (ack.isSuccessful()){
                return ((JSONObject)ack.getData()).getJSONObject("result");
            } else {
                throw new RuntimeException("error getting snapshot, reason: " + ack.getReason());
            }
        });
    }

    /**
     * Applies specified snapshot object to the snapshot provider.
     *
     * @param snapshot The snapshot object
     * @return Ack for the request
     */
    public CompletableFuture<Ack> applySnapshotAsync(JSONObject snapshot) {
        return this.channelClient.dispatchAsync("apply-snapshot", snapshot);
    }

    CompletionStage<SnapshotSourceClient> initChannelClientAsync() {
        final CompletableFuture<SnapshotSourceClient> future = new CompletableFuture<>();
        String cname = String.format("snapshot-source-provider-%s", uuid);
        logger.debug(String.format("initializing snapshot client %s", cname));
        CompletionStage<ChannelClient> stage =  desktopConnection.getChannel(cname).connectAsync();
        stage.thenAccept(client -> {
           SnapshotSourceClient.this.channelClient = client;
           future.complete(SnapshotSourceClient.this);
        });
        return future;
    }

}
