package com.openfin.desktop.test;

import com.openfin.desktop.*;
import com.openfin.desktop.channel.*;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;

public class Test implements DesktopStateListener {

    private static final Logger logger = LoggerFactory.getLogger(Test.class.getName());
    private static final CountDownLatch latch = new CountDownLatch(1);
    private static final String CHANNEL_NAME = "ChannelExample";

    private DesktopConnection desktopConnection;

    private static String channelType;  // client or provider, if not set, both

    public Test() {
        try {
            StringBuilder sb = new StringBuilder("ChannelExample");
            if (channelType != null) {
                sb.append(channelType);
            }
            desktopConnection = new DesktopConnection(sb.toString());
            String desktopVersion = java.lang.System.getProperty("com.openfin.demo.runtime.version", "24.96.66.10");
            RuntimeConfiguration configuration = new RuntimeConfiguration();
            configuration.setRuntimeVersion(desktopVersion);
            desktopConnection.connect(configuration, this, 60);
        } catch (Exception ex) {
            logger.error("Error launching Runtime", ex);
        }
    }

    /**
     * Create a provider that supports "getValue", "increment" and "incrementBy n" actions
     */
    public void createChannelProvider() {
        final Channel channel = desktopConnection.getChannel(CHANNEL_NAME);

        channel.addChannelListener(new ChannelListener() {
            @Override
            public void onChannelConnect(ConnectionEvent connectionEvent) {
                logger.info(String.format("provider receives channel connect event from %s ", connectionEvent.getUuid()));
            }

            @Override
            public void onChannelDisconnect(ConnectionEvent connectionEvent) {
                logger.info(String.format("provider receives channel disconnect event from %s ", connectionEvent.getUuid()));
            }
        });

        channel.createAsync(Arrays.asList(Channel.RTC_PROTOCOL, Channel.CLASSIC_PROTOCOL)).thenAccept(channelProvider -> {
            AtomicInteger x = new AtomicInteger(0);

            channelProvider.register("getValue", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    logger.info(String.format("provider processing action %s", action));
                    JSONObject obj = new JSONObject();
                    obj.put("value", x.get());
                    return obj;
                }
            });

            channelProvider.register("increment", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    logger.info(String.format("provider processing action %s, payload=%s", action, payload.toString()));
                    JSONObject obj = new JSONObject();
                    obj.put("value", x.incrementAndGet());
                    channelProvider.publish("event", obj, null);
                    return obj;
                }
            });
            channelProvider.register("incrementBy", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    logger.info(String.format("provider processing action %s, payload=%s", action, payload.toString()));
                    int delta = ((JSONObject) payload).getInt("delta");
                    JSONObject obj = new JSONObject();
                    obj.put("value", x.addAndGet(delta));
                    return obj;
                }
            });
        });
    }

    /**
     * Create a channel client that invokes "getValue", "increment" and "incrementBy n" actions
     */
    public void createChannelClient() {
        final Channel channel = desktopConnection.getChannel(CHANNEL_NAME);

        channel.connectAsync(null, Channel.RTC_PROTOCOL).thenAccept(channelClient -> {

            channelClient.addChannelListener(new ChannelListener() {
                @Override
                public void onChannelConnect(ConnectionEvent connectionEvent) {
                    logger.info(String.format("connected to channel {}} ", connectionEvent.getChannelName()));
                }

                @Override
                public void onChannelDisconnect(ConnectionEvent connectionEvent) {
                    logger.info(String.format("disconnected from channel {}} ", connectionEvent.getChannelName()));
                }
            });
            // register a channel event
            channelClient.register("event", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    logger.info("channel event {}", action);
                    return null;
                }
            });

            //connected to provider, invoke actions provided by the provider.
            //get current value
            channelClient.dispatchAsync("getValue", null).thenAccept(ack -> {
                logger.info("current value={}", ack.getJsonObject().toString());
                logger.info("current value={}", ack.getJsonObject().getJSONObject("data").getJSONObject("result").getInt("value"));
                JSONObject value = new JSONObject();
                value.put("value", 100);
                channelClient.dispatchAsync("increment", value).thenAccept(ack1 -> {
                    logger.info("current value2={}", ack1.getJsonObject().toString());
                    channelClient.disconnect().thenAccept(ack2 -> {
                        if (ack2.isSuccessful()) {
                            logger.info("channel client is disconnected");
                        }
                    });
                });
            });

        });

    }

    @Override
    public void onReady() {
        if ("provider".equals(channelType) || channelType == null) {
            createChannelProvider();
        }
        if ("client".equals(channelType) || channelType == null) {
            createChannelClient();
        }
    }

    @Override
    public void onClose(String error) {
        logger.info("onClose, value={}", error);
        latch.countDown();
    }

    @Override
    public void onError(String reason) {
        logger.info("onError, value={}", reason);
    }

    @Override
    public void onMessage(String message) {

    }

    @Override
    public void onOutgoingMessage(String message) {

    }

    public static void main(String[] args) {
        if (args.length > 0) {
            channelType = args[0];
        }
        try {
            new Test();
            latch.await();
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        java.lang.System.exit(0);
    }
}