package com.openfin.desktop.win32;

import com.openfin.desktop.*;
import com.sun.jna.Callback;
import com.sun.jna.Native;
import com.sun.jna.Platform;
import com.sun.jna.Pointer;
import com.sun.jna.platform.win32.*;
import com.sun.jna.win32.StdCallLibrary;
import com.sun.jna.win32.W32APIOptions;
import org.json.JSONObject;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * Helper class for hooking into Windows messages
 *
 * Created by wche on 3/11/15.
 *
 */
public class WinMessageHelper {
    // Window messges
    public static final int WM_CAPTURECHANGED = 0x0215;
    public static final int WM_DESTROY = 0x0002;
    public static final int WM_ENTERSIZEMOVE = 0x0231;
    public static final int WM_ERASEBKGND = 0x0014;
    public static final int WM_EXITSIZEMOVE = 0x0232;
    public static final int WM_KEYDOWN = 0x0100;
    public static final int WM_KEYUP = 0x0101;
    public static final int WM_KILLFOCUS = 0x0008;
    public static final int WM_LBUTTONDOWN = 0x0201;
    public static final int WM_LBUTTONUP = 0x0202;
    public static final int WM_MOUSEMOVE = 0x0200;
    public static final int WM_MOVE = 0x0003;
    public static final int WM_MOVING = 0x0216;
    public static final int WM_NCLBUTTONDBLCLK = 0x00A3;
    public static final int WM_NCLBUTTONUP = 0x00A2;
    public static final int WM_NOTIFY = 0x4E;
    public static final int WM_SETFOCUS = 0x0007;
    public static final int WM_SIZE = 0x0005;
    public static final int WM_SIZING = 0x0214;
    public static final int WM_SYSCOMMAND = 0x0112;
    public static final int WM_WINDOWPOSCHANGED = 0x0047;
    public static final int WM_WINDOWPOSCHANGING = 0x0046;
    public static final int WM_PAINT = 0x000F;
    // System command messages
    public static final int SC_CLOSE = 0xF060;
    public static final int SC_MAXIMIZE = 0xF030;
    public static final int SC_MINIMIZE = 0xF020;
    public static final int SC_RESTORE = 0xF120;
    // SetWindowPosition
    public static final int SWP_HIDEWINDOW = 0x0080;
    public static final int SWP_NOMOVE = 0x0002;
    public static final int SWP_NOSIZE = 0x0001;
    public static final int SWP_SHOWWINDOW = 0x0040;
    private final static Logger logger = LoggerFactory.getLogger(WinMessageHelper.class.getName());

    public static CustomUser32 customUser32;
    private static Map<WinDef.HWND, HookProcInfo> callbackMap = Collections.synchronizedMap(new HashMap<WinDef.HWND, HookProcInfo>());

    static {
        try {
            customUser32 = (CustomUser32) Native.loadLibrary("user32", CustomUser32.class,
                    W32APIOptions.DEFAULT_OPTIONS);
        } catch (Exception e) {
            logger.error("Error initializing user32 lib", e);
        }
    }

    /**
     * Makes a connection between messages on a specified window handle
     * and the callback to be called when messages are received.
     *
     * @param hwnd HWND of the window
     * @param callback callback for each message
     *
     */
    public static void hookWndProc(WinDef.HWND hwnd, WindowProcCallback callback) {
        logger.debug("hookWndProc " + hwnd);
        if (callbackMap.get(hwnd) == null) {
            CustomWindowProc customWindowProc = new CustomWindowProc() {
                @Override
                public WinDef.LRESULT callback(WinDef.HWND hwnd, int uMsg, WinDef.WPARAM wParam, Pointer lParam) {
                    logger.trace("callback uMsg " + uMsg);
                    HookProcInfo hpi = callbackMap.get(hwnd);
                    if (hpi != null) {
                        boolean handled = hpi.winProcCallback.callback(hwnd, uMsg, wParam, lParam);
                        if (!handled) {
                            return customUser32.CallWindowProc(hpi.prevWndProc, hwnd, uMsg, wParam, lParam);
                        } else {
                            logger.debug("uMsg handled " + uMsg);
                            return new WinDef.LRESULT(1);
                        }
                    } else {
                        logger.debug("custom windowproc missing. calling default");
                        return customUser32.DefWindowProc(hwnd, uMsg, wParam, lParam);
                    }
                }
            };

            Pointer prev;
            if (Platform.is64Bit()) {
                logger.debug("installing 64b WNDPROC " + customWindowProc);
                prev = customUser32.SetWindowLongPtr(hwnd, User32.GWL_WNDPROC, customWindowProc);
            } else {
                logger.debug("installing WNDPROC " + customWindowProc);
                prev = customUser32.SetWindowLong(hwnd, User32.GWL_WNDPROC, customWindowProc);
            }
            logger.debug("saving WNDPROC " + prev);

            HookProcInfo info = new HookProcInfo();
            info.hwnd = hwnd;
            info.prevWndProc = prev;
            info.winProcCallback = callback;
            info.customWindowProc = customWindowProc;
            callbackMap.put(hwnd, info);
        } else {
            logger.error("hookWndProc custom winProc already installed " + hwnd);
        }
    }

    /**
     * Removes the WindowProc and restores previous one
     *
     * @param hwnd HWND of the window
     */
    public static void unhookWndProc(WinDef.HWND hwnd) {
        logger.debug("unhookWndProc" + hwnd);
        HookProcInfo hpi = callbackMap.get(hwnd);
        if (hpi != null) {
            if (Platform.is64Bit()) {
                logger.debug("installing 64b WNDPROC " + hpi.prevWndProc);
                User32.INSTANCE.SetWindowLongPtr(hwnd, User32.GWL_WNDPROC, hpi.prevWndProc);
            } else {
                logger.debug("installing WNDPROC " + hpi.prevWndProc);
                customUser32.SetWindowLong(hwnd, User32.GWL_WNDPROC, hpi.prevWndProc);
            }
            callbackMap.remove(hwnd);
        }
    }

    /**
     * Support different signature of callback in WinUser.WindowProc
     */
    private static interface CustomWindowProc extends Callback {
        WinDef.LRESULT callback(WinDef.HWND hwnd, int uMsg, WinDef.WPARAM wParam, Pointer lParam);
    }

    /**
     * Support different signature of User32
     */
    public static interface CustomUser32 extends StdCallLibrary {
        public int GWL_HWNDPARENT = -8;
        public Pointer SetWindowLong(WinDef.HWND hwnd, int index, CustomWindowProc newProc);
        public Pointer SetWindowLong(WinDef.HWND hWnd, int nIndex, Pointer dwNewPtr);
        public Pointer SetWindowLongPtr(WinDef.HWND hwnd, int index, CustomWindowProc newProc);
        public WinDef.LRESULT CallWindowProc(Pointer proc, WinDef.HWND hWnd, int uMsg, WinDef.WPARAM uParam, Pointer lParam);
        public WinDef.LRESULT DefWindowProc(WinDef.HWND hWnd, int Msg, WinDef.WPARAM wParam, Pointer lParam);

        public boolean GetCursorPos(WinDef.POINT point);

        public boolean ChangeWindowMessageFilterEx(WinDef.HWND hWnd, int Msg, WinDef.DWORD action, Pointer lParam);
    }

    /**
     *  This class remember old WindowProc so it can be restored
     */
    private static class HookProcInfo {
        public WinDef.HWND hwnd;
        public Pointer prevWndProc;
        public WindowProcCallback winProcCallback;
        public CustomWindowProc customWindowProc;
    }

    public static void main(String[] argv) {
        try {
//            java.lang.System.out.println(Advapi32Util.registryGetStringValue(HKEY_LOCAL_MACHINE, "SOFTWARE\\Policies\\OpenFin\\AutoSelectCertificateForUrls"));

//            WinReg.HKEYByReference phkKey = Advapi32Util.registryGetKey(WinReg.HKEY_CURRENT_USER, "SOFTWARE\\OpenFin\\RVM\\Settings", WinNT.KEY_READ | WinNT.KEY_WOW64_32KEY);
//            String[] value = Advapi32Util.registryGetKeys(phkKey.getValue(), "cleanUnusedRuntimes");
//            java.lang.System.out.println(value);
//            java.lang.System.out.println(Advapi32Util.registryGetStringValue(HKEY_CURRENT_USER, "SOFTWARE\\OpenFin\\RVM", "cleanUnusedRuntimes"));

            final String cpuRegistryRoot = "HARDWARE\\DESCRIPTION\\System\\CentralProcessor";
            String[] processorIds = Advapi32Util.registryGetKeys(WinReg.HKEY_LOCAL_MACHINE, cpuRegistryRoot);
            String processorId = processorIds[0];
            String cpuRegistryPath = cpuRegistryRoot + "\\" + processorId;
            java.lang.System.out.println(Advapi32Util.registryGetStringValue(WinReg.HKEY_LOCAL_MACHINE, cpuRegistryPath, "ProcessorNameString").trim());

            String settings = "SOFTWARE\\OpenFin\\RVM\\Settings";
            java.lang.System.out.println(Advapi32Util.registryGetIntValue(WinReg.HKEY_CURRENT_USER, settings, "cleanUnusedRuntimes"));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
