package com.openfin.desktop.channel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import com.openfin.desktop.*;

import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.accessibility.AccessibleKeyBinding;

/**
 * The Channel object allows an OpenFin application to create a channel as a ChannelProvider, or connect to a channel as a ChannelClient.
 * @author Anthony
 *
 */
public class Channel {
	private final static Logger logger = LoggerFactory.getLogger(Channel.class.getName());

	private final DesktopConnection desktopConnection;
	private final String name;

	private ChannelProvider provider;
	private final ConcurrentHashMap<String, ChannelClient> clientMap;
	private final CopyOnWriteArrayList<ChannelListener> channelListeners;
	private final CopyOnWriteArrayList<CompletableFuture<ChannelClient>> pendingConnections;

	private final static String CONNECTED_EVENT = "connected";
	private final static String DISCONNECTED_EVENT = "disconnected";
	final private static String PROVIDER_NOT_READY_REASON = "internal-nack";  // set when channel provider is not active during connecting

	public Channel(String name, DesktopConnection desktopConnection) {
		this.name = name;
		this.desktopConnection = desktopConnection;
		this.clientMap = new ConcurrentHashMap<>();
		this.channelListeners = new CopyOnWriteArrayList<>();
		this.pendingConnections = new CopyOnWriteArrayList<>();
		try {
			this.addEventListener(CONNECTED_EVENT, e->{
				String eventChannelName = e.getEventObject().getString("channelName");
				if (eventChannelName.equals(this.name)) {
					logger.debug("channel[{}] connected", this.name);
					JSONObject providerIdentity = e.getEventObject();
					EndpointIdentity ei = new EndpointIdentity(providerIdentity);
					this.fireChannelConnectEvent(new ConnectionEvent(ei.getChannelId(), ei.getUuid(), ei.getName(), ei.getChannelName(), ei.getEndpointId()));
					//check if any pending connections
					ArrayList<CompletableFuture<ChannelClient>> connectionFutures = new ArrayList<>(this.pendingConnections);
					this.pendingConnections.clear();
					connectionFutures.forEach(connectionFuture->{
						this.connectAsync().thenAccept(client->{
							connectionFuture.complete(client);
						});
					});
				}
			}, null);
			this.addEventListener(DISCONNECTED_EVENT, e->{
				String eventChannelName = e.getEventObject().getString("channelName");
				if (eventChannelName.equals(this.name)) {
					logger.debug("channel[{}] disconnected", this.name);
					JSONObject providerIdentity = e.getEventObject();
					EndpointIdentity ei = new EndpointIdentity(providerIdentity);
					this.fireChannelDisconnectEvent(new ConnectionEvent(ei.getChannelId(), ei.getUuid(), ei.getName(), ei.getChannelName(), ei.getEndpointId()));
				}
			}, null);
		}
		catch (DesktopException e) {
			logger.error("unable to subscribe to channel events", e);
		}
	}

	/**
	 * Get the name of the channel
	 * @return name of the channel
	 */
	public String getName() {
		return this.name;
	}

	public DesktopConnection getDesktopConnection() {
		return this.desktopConnection;
	}
	
	public CompletionStage<ChannelProvider> createAsync() {
		JSONObject payload = new JSONObject();
		payload.put("channelName", this.name);
		return desktopConnection.sendActionAsync("create-channel", payload, this).thenApply(ack->{
			if (ack.isSuccessful()) {
				JSONObject providerIdentity = (JSONObject) ack.getData();
				EndpointIdentity providerEndpointIdentity = new EndpointIdentity(providerIdentity);
				this.provider =  new ChannelProvider(this, providerEndpointIdentity);
				return this.provider;
			}
			else {
				throw new RuntimeException("error creating channel provider, reason: " + ack.getReason());
			}
		});
	}

	/**
	 * Create a new channel.
	 * @param callback The callback that receives the wrapped {@link ChannelProvider} object
	 */
	public void create(AsyncCallback<ChannelProvider> callback) {
		this.createAsync().thenAccept(provider ->{
			callback.onSuccess(provider);
		});
	}

	void destroy(ChannelBase provider, AckListener ackListener) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", provider.getChannelName());
		desktopConnection.sendAction("destroy-channel", payload, new AckListener() {

			@Override
			public void onSuccess(Ack ack) {
				Channel.this.provider = null;
				if (ackListener != null) {
					ackListener.onSuccess(ack);
				}
			}

			@Override
			public void onError(Ack ack) {
				if (ackListener != null) {
					ackListener.onError(ack);
				}
			}
			
		}, this);
	}

	/**
	 * Connect to a channel provider
	 *
	 * @return A new CompletionStage for the client client
	 */
	public CompletionStage<ChannelClient> connectAsync() {
		return this.connectAsync(null, null);
	}

	/**
	 * Connect to a channel provider with a payload passed to the provider
	 *
	 * @param connectPayload connection payload
	 * @return A new CompletionStage for the channel client
	 */
	public CompletionStage<ChannelClient> connectAsync(Object connectPayload) {
		return this.connectAsync(null, connectPayload);
	}

	/**
	 * Connect to a channel provider with a payload passed to the provider
	 *
	 * @param noWait if true, the request will fail if the provider is not running
	 * @param connectPayload connection payload
	 * @return A new CompletionStage for the client client
	 */
	public CompletionStage<ChannelClient> connectAsync(Boolean noWait, Object connectPayload) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", Channel.this.name);
		if (connectPayload != null) {
			payload.put("payload", connectPayload);
		}
		CompletableFuture<ChannelClient> connectionFuture = new CompletableFuture<>();
		
		desktopConnection.sendActionAsync("connect-to-channel", payload, this).thenAccept(ack->{
			if (ack.isSuccessful()) {
				JSONObject providerIdentity = (JSONObject) ack.getData();
				EndpointIdentity endpointIdentity = new EndpointIdentity(providerIdentity);
				ChannelClient client = new ChannelClient(Channel.this, endpointIdentity);
				this.clientMap.put(endpointIdentity.getEndpointId(), client);
				connectionFuture.complete(client);
			}
			else {
				if (!PROVIDER_NOT_READY_REASON.equals(ack.getReason())) {
					connectionFuture.completeExceptionally(new RuntimeException("error connecting to channel " + this.name + ", reason: " + ack.getReason()));
				} else if (noWait != null && noWait.booleanValue()) {
					connectionFuture.completeExceptionally(new RuntimeException("error connecting to channel " + this.name + ", reason: " + ack.getReason()));
				}
				else {
					//connect to the provider when it's ready.
					this.pendingConnections.add(connectionFuture);
				}
			}
		});
		return connectionFuture;
	}
	
	/**
	 * Connect to the channel.
	 * @param callback The callback that receives the wrapped {@link ChannelClient} object
	 */
	public void connect(AsyncCallback<ChannelClient> callback) {
		this.connectAsync().thenAccept(client->{
			callback.onSuccess(client);
		});
	}

	void disconnect(ChannelBase client, AckListener ackListener) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", client.getChannelName());
		desktopConnection.sendAction("disconnect-from-channel", payload, new AckListener() {
			@Override
			public void onSuccess(Ack ack) {
				if (ackListener != null) {
					ackListener.onSuccess(ack);
				}
//				fireChannelDisconnectEvent(client.getChannelId(), client.getUuid(), client.getName(),
//						client.getChannelName(), client.getEndpointId());
			}

			@Override
			public void onError(Ack ack) {
				ackListener.onError(ack);
			}
		}, this);
	}

	void sendChannelMessage(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
			JSONObject actionPayload, AckListener ackListener) {
		JSONObject payload = new JSONObject(destionationIdentity,
				new String[] { "name", "uuid", "channelId", "channelName", "endpointId" });
		payload.put("providerIdentity", providerIdentity);
		payload.put("action", action);
		payload.put("payload", actionPayload);
		desktopConnection.sendAction("send-channel-message", payload, ackListener,this);
	}

	CompletableFuture<Ack> sendChannelMessageAsync(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
			Object actionPayload) {
		JSONObject payload = new JSONObject(destionationIdentity,
				new String[] { "name", "uuid", "channelId", "channelName", "endpointId" });
		payload.put("providerIdentity", providerIdentity);
		payload.put("action", action);
		payload.put("payload", actionPayload);
		return desktopConnection.sendActionAsync("send-channel-message", payload, this);
	}

	/**
	 * Check if the channel has provider created
	 * @return if the channel provider exists.
	 */
	public boolean hasProvider() {
		return this.provider != null;
	}

	/**
	 * Check if the channel has client connected
	 * @return true if a channel client exists
	 */
	public boolean hasClient() {
		return this.clientMap.keySet().size() > 0;
	}

	public Object invokeAction(EndpointIdentity targetIdentity, String action, Object actionPayload,
			JSONObject senderIdentity) {
		ChannelBase target = null;
		Object result = null;
		if (targetIdentity.getEndpointId() != null) { //specified ChannelClient.
			//target should be channel client.
			ChannelClient channelClient = this.clientMap.get(targetIdentity.getEndpointId());
			if (channelClient != null && channelClient.hasRegisteredAction(action)) {
				result = channelClient.invokeAction(action, actionPayload, senderIdentity);
			}
		}
		else { 
			if (this.provider != null && this.provider.hasRegisteredAction(action)) {
				result = provider.invokeAction(action, actionPayload, senderIdentity);
			}
			else {
				Collection<ChannelClient> clients = this.clientMap.values();
				JSONArray results = new JSONArray();
				for (ChannelClient client :  clients) {
					if (client.hasRegisteredAction(action)) {
						Object cr = client.invokeAction(action, actionPayload, senderIdentity);
						if (cr != null) {
							results.put(cr);
						}
					}
				}
				result = results.length() == 1 ? results.get(0) : results;
			}
		}
		return result;
	}

	public boolean addChannelListener(ChannelListener listener) {
		return this.channelListeners.add(listener);
	}

	public boolean removeChannelListener(ChannelListener listener) {
		return this.channelListeners.remove(listener);
	}

	protected void fireChannelConnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelConnect(event);
		}
		this.clientMap.values().forEach(channelClient->{
			channelClient.fireChannelConnectEvent(event);
		});
	}

	protected void fireChannelDisconnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelDisconnect(event);
		}
		this.clientMap.values().forEach(channelClient->{
			channelClient.fireChannelDisconnectEvent(event);
		});
	}

	public void processConnection(JSONObject payload) throws Exception {
		JSONObject clientIdentity = payload.getJSONObject("clientIdentity");
		this.provider.processConnection(clientIdentity, payload);
	}

	/**
	 * Registers an event listener on the specified event
	 * <pre>
	 *     Supported system event types are:
	 * </pre>
	 *
	 * @param subscriptionObject A JSON object containing subscription information such as the topic and type
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws DesktopException if this method fails to add event listener specified
	 * @see EventListener
	 * @see AckListener
	 */
	protected void addEventListener(JSONObject subscriptionObject,
									EventListener listener,
									AckListener callback) throws DesktopException {
		this.desktopConnection.addEventCallback(subscriptionObject, listener, callback, this);
	}

	/**
	 * Registers an event listener on the specified event
	 * <pre>
	 *     Supported system event types are:
	 * </pre>
	 *
	 * @param type Type of the event
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws  DesktopException if this method fails to add event listener specified
	 * @see EventListener
	 * @see AckListener
	 */
	public void addEventListener(String type, EventListener listener, AckListener callback) throws DesktopException {
		try {
			JSONObject eventListenerPayload = new JSONObject();
			eventListenerPayload.put("topic", "channel");
			eventListenerPayload.put("type", type);
			addEventListener(eventListenerPayload, listener, callback);
		} catch (Exception e) {
			logger.error("Error adding event listener", e);
			throw new DesktopException(e);
		}
	}

	/**
	 * Removes a previously registered event listener from the specified event
	 *
	 * @param type Type of the event
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws  DesktopException if this method fails to remove event listener specified
	 * @see DesktopException
	 * @see EventListener
	 * @see AckListener
	 */
	public void removeEventListener(String type, EventListener listener, AckListener callback) throws DesktopException{
		try {
			JSONObject eventListenerPayload = new JSONObject();
			eventListenerPayload.put("topic", "channel");
			eventListenerPayload.put("type", type);
			this.desktopConnection.removeEventCallback(eventListenerPayload, listener, callback, this);
		} catch (Exception e) {
			logger.error("Error removing event listener", e);
			throw new DesktopException(e);
		}
	}
}
