package com.openfin.desktop;

import java.awt.Rectangle;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class JsonBean {
	
	protected JSONObject json;
	
	public JsonBean() {
		this.json = new JSONObject();
	}
	
	public JsonBean(JSONObject json) {
		this.json = json;
	}

    /**
     * Gets the underlying JSONObject
     * @return JSONObject
     */
	public JSONObject getJson() {
		return this.json;
	}
	
    /**
     * Gets a copy of underlying JSONObject
     * @return JSONObject
     */
	public final JSONObject getJsonCopy() {
		return new JSONObject(this.getJson().toString());
	}
	
    /**
     * Helper method to extract data field from underlying JSONObject
     *
     * @param property name of the field
     * @return String value of a field
     */
	public String getString(String property) {
		return json.optString(property, null);
    }

    /**
     * Helper method to extract data field from underlying JSONObject
     * @param property name of the field
     * @return Integer value of a field
     */
	public int getIntegerValue(String property) {
        int value = 0;
        try {
            value =  json.getInt(property);
        } catch (JSONException e) {
        }
        return value;
    }

    /**
     * Helper method to extract data field from underlying JSONObject
     * @param property name of the field
     * @return Double value of a field
     */
	public double getDoubleValue(String property) {
        return this.getDoubleValue(property, 0);
    }
	
	public double getDoubleValue(String property, double defaultValue) {
		return this.json.optDouble(property, defaultValue);
	}

    /**
     * Helper method to extract data field from underlying JSONObject
     * @param property name of the field
     * @return Boolean value of a field
     */
	public boolean getBooleanValue(String property) {
        return getBooleanValue(property, false);
    }

    /**
     * Helper method to extract data field from underlying JSONObject
     * @param property name of the field
     * @param defaultValue default value if property is missing
     * @return Boolean value of a field
     */
	public boolean getBooleanValue(String property, boolean defaultValue) {
    	return json.optBoolean(property, defaultValue);
    }

    /**
     * Helper method to extract data field from underlying JSONObject
     * @param property name of the field
     * @return JSONObject value of a field
     */
	public JSONObject getJsonValue(String property) {
        return  json.optJSONObject(property);
    }

    /**
     * Set value for a property
     * @param key name of the property
     * @param value value
     * @return current object
     * @throws JSONException from JSONObject.put
     */
	public JsonBean put(String key, Object value) throws JSONException {
        this.json.put(key, value);
        return this;
     }
     
     public Object get(String key) {
    	 return this.json.get(key);
     }
	
	@Override
	public String toString() {
		return this.json.toString(2);
	}
	
	public Boolean getBoolean(String name) {
		return this.json.has(name) ? this.json.getBoolean(name) : null;
	}
	
	public void setBoolean(String name, Boolean value) {
		if (value == null) {
			this.json.remove(name);
		}
		else {
			this.json.put(name, value);
		}
	}
	
	public void setString(String name, String value) {
		this.json.put(name, value == null ? JSONObject.NULL : value);
	}
	
	public <T> void setArray(String name, Collection<T> value) {
		this.json.put(name, value);
	}
	
	public JSONArray getArray(String name) {
		if (this.json.has(name)) {
			return this.json.getJSONArray(name);
		}
		else {
			return null;
		}
	}
	
	@SuppressWarnings("unchecked")
	public <T> List<T> getList(String name) {
		JSONArray array = this.getArray(name);
		if (array != null) {
			int arrayLength = array.length();
			List<T> list = new ArrayList<>(arrayLength);
			for (int i=0; i<arrayLength; i++) {
				Object obj = array.get(i);
				list.add((T) obj);
			}
			return list;
		}
		else {
			return null;
		}
	}
	
	public <T> void setJsonArray(String name, List<T> values) {
		if (values == null) {
			this.json.remove(name);
		}
		else {
			JSONArray array = new JSONArray();
			values.forEach(obj->{
				if (obj instanceof JsonBean) {
					JsonBean jsonBean = (JsonBean) obj;
					array.put(jsonBean.getJsonCopy());
				}
				else {
					array.put(obj);
				}
			});
			this.json.put(name, array);
		}
	}
	
	public <T extends JsonBean> List<T> getJsonBeanList(String name, Class<T> clazz) {
		JSONArray array = this.getArray(name);
		if (array != null) {
			int arrayLength = array.length();
			List<T> list = new ArrayList<>(arrayLength);
			for (int i=0; i<arrayLength; i++) {
				JSONObject obj = array.getJSONObject(i);
				T newBean = this.getJsonBean(obj, clazz);
				if (newBean != null) {
					list.add(newBean);
				}
			}
			return list;
		}
		else {
			return null;
		}
	}
	
	public Integer getInteger(String name) {
		return this.json.has(name) ? this.json.getInt(name) : null;
	}
	
	public void setInteger(String name, Integer value) {
		this.json.put(name, value == null ? JSONObject.NULL : value);
	}
	
	public void setLong(String name, Long value) {
		this.json.put(name, value == null ? JSONObject.NULL : value);
	}

	public <T> T getJsonBean(String name, Class<T> clazz) {
		return this.getJsonBean(this.json.getJSONObject(name), clazz);
	}
	public <T> T getJsonBean(JSONObject obj, Class<T> clazz) {
		try {
			return clazz.getDeclaredConstructor(JSONObject.class).newInstance(obj);
			
		}
		catch (InstantiationException | IllegalAccessException | IllegalArgumentException
				| InvocationTargetException | NoSuchMethodException | SecurityException e) {
			e.printStackTrace();
		}
		finally {
			
		}
		return null;
	}
	
	public <T extends JsonBean> void setJsonBean(String name, T jsonBean) {
		if (jsonBean == null) {
			this.json.remove(name);
		}
		else {
			this.json.put(name, jsonBean.getJson());
		}
	}
	
	public void setRectangle(String name, Rectangle rect) {
		if (rect == null) {
			this.json.remove(name);
		}
		else {
			JSONObject jsonRect = new JSONObject();
			jsonRect.put("x", rect.x);
			jsonRect.put("y", rect.y);
			jsonRect.put("width", rect.width);
			jsonRect.put("height", rect.height);
			this.json.put(name, jsonRect);
		}
	}
	
	public Rectangle getRectangle(String name) {
		if (this.json.has(name)) {
			JSONObject rect = this.json.getJSONObject(name);
			return new Rectangle(rect.getInt("x"), rect.getInt("y"), rect.getInt("width"), rect.getInt("height"));
		}
		else {
			return null;
		}
	}
	
}
