package com.openfin.desktop;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 *
 * Helper class for Populating values from the ACK of a window 'group-changed' payload
 *
 * @author wche
 * @since 9/4/14
 */
public class WindowGroupChangeEvent {
    private String memberOf;
    private String name;
    private String reason;
    private List<Window> sourceGroup;
    private String sourceWindowAppUuid;
    private String sourceWindowName;
    private List<Window> targetGroup;
    private String targetWindowAppUuid;
    private String targetWindowName;
    private String uuid;
    private DesktopConnection desktopConnection;

    /**
     * Constructor
     *
     * @param payload JSONObject payload
     * @param desktopConnection desktop connection
     *
     */
    public WindowGroupChangeEvent(Ack payload, DesktopConnection desktopConnection) throws JSONException {
        this.desktopConnection = desktopConnection;
        JSONObject data = (payload != null && payload.getJsonObject() != null ? payload.getJsonObject() : new JSONObject());

        this.memberOf = JsonUtils.getStringValue(data, "memberOf", "nothing");
        this.name = JsonUtils.getStringValue(data, "name", "");
        this.reason = JsonUtils.getStringValue(data, "reason", "");
        this.sourceWindowAppUuid = JsonUtils.getStringValue(data, "sourceWindowAppUuid", "");
        this.sourceWindowName = JsonUtils.getStringValue(data, "sourceWindowName", "");
        this.targetWindowAppUuid = JsonUtils.getStringValue(data, "targetWindowAppUuid", "");
        this.targetWindowName = JsonUtils.getStringValue(data, "targetWindowName", "");
        this.uuid = JsonUtils.getStringValue(data, "uuid", "");

        this.sourceGroup = new ArrayList<Window>();

        // If the JSON array is present
        JSONArray sourceGroupWindows =  JsonUtils.getJsonArray(data, "sourceGroup", null);
        if (sourceGroupWindows != null) {
            for (int i = 0; i < sourceGroupWindows.length(); i++) {
                JSONObject entry = sourceGroupWindows.getJSONObject(i);
                this.sourceGroup.add(Window.wrap(JsonUtils.getStringValue(entry, "appUuid", ""),
                                        JsonUtils.getStringValue(entry, "windowName", ""),
                                        desktopConnection));
            }
        }

        targetGroup = new ArrayList<Window>();
        // If the JSON array is present
        JSONArray targetGroupWindows = JsonUtils.getJsonArray(data, "targetGroup", null);
        if (targetGroupWindows != null)
        {
            for (int i = 0; i < sourceGroupWindows.length(); i++) {
                JSONObject entry = sourceGroupWindows.getJSONObject(i);
                targetGroup.add(Window.wrap(JsonUtils.getStringValue(entry, "appUuid", ""),
                        JsonUtils.getStringValue(entry, "windowName", ""),
                        desktopConnection));
            }
        }
    }

    /**
     *
     * Which group array the window that the event listener was
     * registered on is included in.
     * Possible values:
     *  'source'  The window is included in sourceGroup
     *  'target'  The window is included in targetGroup
     *  'nothing' The window is not included in sourceGroup nor targetGroup,
     *
     * @return value of memberOf
     *
     */
    public String getMemberOf() {
        return memberOf;
    }

    /**
     * Name of the window
     *
     * @return value of name
     */
    public String getName() {
        return name;
    }

    /**
     *
     * The reason this event was triggered.
     * Possible values:
     *  'leave'   A window has left the group due to a leave or merge with group.
     *  'join'    A window has joined the group.
     *  'merge'   Two groups have been merged together.
     *  'disband' There are no other windows in the group
     *
     * @return value of reason
     */
    public String getReason() {
        return reason;
    }

    /**
     *
     * All the windows in the group the sourceWindow originated from
     *
     * @return list of Windows
     *
     */
    public List<Window> getSourceGroup() {
        return sourceGroup;
    }

    /**
     *
     * The UUID of the application the sourceWindow belongs to.
     * The source window is the window in which (merge/join/leave)group(s) was called.
     *
     * @return value of sourceWindowAppUuid
     *
     */
    public String getSourceWindowAppUuid() {
        return sourceWindowAppUuid;
    }

    /**
     *
     * The name of the sourcewindow. The source window is
     * the window in which (merge/join/leave)group(s) was called.
     *
     * @return value of sourceWindowName
     */
    public String getSourceWindowName() {
        return sourceWindowName;
    }

    /**
     *
     * All the windows in the group the targetWindow orginated from
     *
     * @return list of Windows
     */
    public List<Window> getTargetGroup() {
        return targetGroup;
    }

    /**
     *
     * The UUID of the application the targetWindow belongs to.
     * The target window is the window that was passed into (merge/join)group(s).
     *
     * @return value of targetWindowAppUuid
     */
    public String getTargetWindowAppUuid() {
        return targetWindowAppUuid;
    }

    /**
     *
     * The name of the targetWindow.
     * The target window is the window that was passed into (merge/join)group(s)
     *
     * @return value of targetWindowName
     */
    public String getTargetWindowName() {
        return targetWindowName;
    }

    /**
     *
     * The UUID of the application the window belongs to
     *
     * @return application UUID
     */
    public String getUuid() {
        return uuid;
    }
}
