package com.openfin.desktop.net;

import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.StatusCode;
import org.eclipse.jetty.websocket.api.annotations.*;
import org.eclipse.jetty.websocket.client.ClientUpgradeRequest;
import org.eclipse.jetty.websocket.client.WebSocketClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.util.concurrent.Future;

/**
 * A WebSocket connection to OpenFin Runtime
 *
 * Created by wche on 9/21/2016.
 */
@WebSocket(maxTextMessageSize = 64 * 1024)
public class WebSocketConnection {
    private static Logger logger = LoggerFactory.getLogger(WebSocketConnection.class.getName());

    private URI url = null;
    private WebSocketEventHandler eventHandler = null;
    @SuppressWarnings("unused")
    private Session session;
    private Future<Session> connectFuture;

    public WebSocketConnection(URI url) throws WebSocketException {
        this.url = url;
    }

    public void setEventHandler(WebSocketEventHandler eventHandler) {
        this.eventHandler = eventHandler;
    }


    public WebSocketEventHandler getEventHandler() {
        return this.eventHandler;
    }

    public boolean isConnected() {
        return this.session != null;
    }

    public void connect() throws WebSocketException {
        try {
            if (session != null) {
                throw new WebSocketException("already connected");
            }
            if (this.connectFuture != null && !this.connectFuture.isDone()) {
                logger.debug("Waiting to be connected");
            } else {
                ClientUpgradeRequest request = new ClientUpgradeRequest();
                this.connectFuture = WebSocketClientFactory.getWebSocketClient().connect(this, this.url, request);
            }
        } catch (Exception ex) {
                throw new WebSocketException("error while connecting: " + ex.getMessage(), ex);
        }
    }

    public synchronized void send(String data) throws WebSocketException {
        if (session == null) {
            throw new WebSocketException("error while sending text data: not connected");
        }
        try {
            session.getRemote().sendString(data);
        } catch (Exception ioe) {
            throw new WebSocketException("error while sending text data: " + data, ioe);
        }
    }

    public void close() throws WebSocketException {
        this.close("Requested by OpenFin app");
    }

    public synchronized void close(String reason) throws WebSocketException {
        try {
            if (session != null) {
                logger.debug(String.format("close %s",reason));
                session.close(StatusCode.NORMAL, reason);
            }
        } catch (Exception ex) {
            logger.error("Error closing session", ex);
        }
    }

    @OnWebSocketClose
    public void onClose(int statusCode, String reason) {
        logger.debug(String.format("Connection closed: %d - %s", statusCode, reason));
        this.session = null;
        try {
            eventHandler.onClose(statusCode, reason);
        } catch (Exception ex) {
            logger.error("Error processing close event", ex);
        }
    }

    @OnWebSocketConnect
    public void onConnect(Session session) {
        session.setIdleTimeout(0);
        logger.debug(String.format("Connected: %s timeout: %d", session.getLocalAddress(), session.getIdleTimeout()));
        this.session = session;
        this.eventHandler.onOpen();
    }

    @OnWebSocketMessage
    public void onMessage(String msg) {
        this.eventHandler.onMessage(new WebSocketMessage(msg));
    }

    @OnWebSocketError
    public void onError(Throwable cause) {
        logger.error("onError", cause);
        try {
            eventHandler.onError(cause);
        } catch (Exception ex) {
            logger.error("Error processing error event", ex);
        }
    }
    
    public void setMaxMessageSize(int size) {
    	this.session.getPolicy().setMaxTextMessageSize(size);
    }

}
