package com.openfin.desktop.notifications;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantLock;

import com.openfin.desktop.*;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openfin.desktop.channel.Channel;
import com.openfin.desktop.channel.ChannelClient;
import com.openfin.desktop.notifications.events.NotificationActionEvent;
import com.openfin.desktop.notifications.events.NotificationClosedEvent;
import com.openfin.desktop.notifications.events.NotificationCreatedEvent;
import com.openfin.desktop.notifications.events.NotificationEvent;

/**
 * This class initializes and manages notifications.
 * 
 * @author Anthony
 *
 */
public class Notifications {

	private final static Logger logger = LoggerFactory.getLogger(Notifications.class);

	public final static String EVENT_TYPE_ACTION = "notification-action";
	public final static String EVENT_TYPE_CREATED = "notification-created";
	public final static String EVENT_TYPE_CLOSED = "notification-closed";

	private final static String NotificationServiceChannelName = "of-notifications-service-v1";
	private static String NotificationsServiceManifestUrl = "https://cdn.openfin.co/services/openfin/notifications/app.json";


	private DesktopConnection desktopConnection;
	private CompletionStage<ChannelClient> channelClient;
	private ReentrantLock channelClientLock;
	private ConcurrentHashMap<String, CopyOnWriteArrayList<NotificationEventListener>> eventListenerMap;

	public Notifications(DesktopConnection desktopConnection) {
		this.desktopConnection = desktopConnection;
		this.channelClientLock = new ReentrantLock();
		this.eventListenerMap = new ConcurrentHashMap<>();
		this.eventListenerMap.put(EVENT_TYPE_ACTION, new CopyOnWriteArrayList<NotificationEventListener>());
		this.eventListenerMap.put(EVENT_TYPE_CREATED, new CopyOnWriteArrayList<NotificationEventListener>());
		this.eventListenerMap.put(EVENT_TYPE_CLOSED, new CopyOnWriteArrayList<NotificationEventListener>());
	}

	private void invokeEventListeners(JSONObject payload) {
		String eventType = payload.getString("type");
		CopyOnWriteArrayList<NotificationEventListener> listeners = this.eventListenerMap.get(eventType);
		for (NotificationEventListener listener : listeners) {

			NotificationEvent event = null;
			if (EVENT_TYPE_ACTION.equals(eventType)) {
				event = new NotificationActionEvent(payload);
			}
			else if (EVENT_TYPE_CREATED.equals(eventType)) {
				event = new NotificationCreatedEvent(payload);
			}
			else if (EVENT_TYPE_CLOSED.equals(eventType)) {
				event = new NotificationClosedEvent(payload);
			}

			if (event != null) {
				listener.onEvent(event);
			}
			else {
				// error
			}
		}
	}

	private CompletionStage<ChannelClient> getChannelClient() {
		if (channelClient == null) {
			channelClientLock.lock();
			if (this.channelClient == null) {
				try {
					String launchError = launchNotificationCenter().get();
					if (launchError == null) {
						Channel channel = this.desktopConnection.getChannel(NotificationServiceChannelName);
						this.channelClient = channel.connectAsync().thenApply(client -> {
							client.register("event", (action, payload, senderIdentity) -> {
								this.invokeEventListeners((JSONObject) payload);
								return null;
							});
							client.dispatchAsync("add-event-listener", Notifications.EVENT_TYPE_ACTION);
							return client;
						});
						logger.debug("notification channel client created");
					} else {
						logger.error(String.format("Error getting notification client  %s", launchError));
					}
				} catch (Exception ex) {
					logger.error(String.format("Error getting notification client  %s", ex.toString()));
				}
				this.channelClientLock.unlock();
			}
		}
		return this.channelClient;
	}

	private CompletableFuture<String> launchNotificationCenter() {
		CompletableFuture<String> future = new CompletableFuture();
		try {
			OpenFinRuntime r = new OpenFinRuntime(desktopConnection);
			r.launchManifest("fins://system-apps/notification-center", null, new AckListener() {
				@Override
				public void onSuccess(Ack ack) {
					future.complete(ack.getReason());
				}

				@Override
				public void onError(Ack ack) {
					logger.error(String.format("Error starting  %s %s", NotificationsServiceManifestUrl, ack.getReason()));
					future.complete(ack.getReason());
				}
			});
		} catch (Exception ex) {
			logger.error(String.format("Error loadManifest  %s %s", NotificationsServiceManifestUrl, ex.toString()));
			future.complete(ex.getMessage());
		}
		return future;
	}


	/**
	 * Creates a new notification.
	 * 
	 * The notification will appear in the Notification Center and as a toast if the
	 * Center is not visible.
	 * 
	 * If a notification is created with an id of an already existing notification,
	 * the existing notification will be recreated with the new content.
	 * 
	 * @param options
	 *            Notification configuration options.
	 * @return new CompletionStage for the fully-hydrated NotificationOptions.
	 */
	public CompletionStage<NotificationOptions> create(NotificationOptions options) {
		return this.getChannelClient().thenCompose(channelClient -> {
			return channelClient.dispatchAsync("create-notification", options.getJson()).thenApply(ack -> {
				return new NotificationOptions(((JSONObject) ack.getData()).getJSONObject("result"));
			});
		});
	}

	/**
	 * Clears a specific notification from the Notification Center.
	 * 
	 * @param id
	 *            ID of the notification to clear.
	 * @return new CompletionStage for the status, true if the notification was
	 *         successfully cleared. false if the notification was not cleared,
	 *         without errors.
	 */
	public CompletionStage<Boolean> clear(String id) {
		return this.getChannelClient().thenCompose(channelClient -> {
			JSONObject payload = new JSONObject();
			payload.put("id", id);
			return channelClient.dispatchAsync("clear-notification", payload).thenApply(ack -> {
				return ack.isSuccessful();
			});
		});
	}

	/**
	 * Clears all Notifications which were created by the calling application,
	 * including child windows.
	 * 
	 * @return new CompletionStage for the the number of successfully cleared
	 *         Notifications.
	 */
	public CompletionStage<Integer> clearAll() {
		return this.getChannelClient().thenCompose(channelClient -> {
			return channelClient.dispatchAsync("clear-app-notifications", new JSONObject()).thenApply(ack -> {
				return ((JSONObject) ack.getData()).getInt("result");
			});
		});
	}

	/**
	 * Retrieves all Notifications which were created by the calling application,
	 * including child windows.
	 * 
	 * @return new CompletionStage for all NotificationOptions.
	 */
	public CompletionStage<List<NotificationOptions>> getAll() {
		return this.getChannelClient().thenCompose(channelClient -> {
			return channelClient.dispatchAsync("fetch-app-notifications", new JSONObject()).thenApply(ack -> {
				JSONArray resultJson = ((JSONObject) ack.getData()).getJSONArray("result");
				int resultCnt = resultJson.length();
				ArrayList<NotificationOptions> result = new ArrayList<NotificationOptions>(resultCnt);

				for (int i = 0; i < resultCnt; i++) {
					result.add(new NotificationOptions(resultJson.getJSONObject(i)));
				}

				return result;
			});
		});
	}

	/**
	 * Toggles the visibility of the Notification Center.
	 * 
	 * @return new CompletionStage when the command is delivered.
	 */
	public CompletionStage<Void> toggleNotificationCenter() {
		return this.getChannelClient().thenCompose(channelClient -> {
			return channelClient.dispatchAsync("toggle-notification-center", new JSONObject()).thenAccept(ack -> {
			});
		});
	}

	/**
	 * Add a listener to handle specified notification events.
	 * @param eventType EVENT_TYPE_ACTION, EVENT_TYPE_CREATED or EVENT_TYPE_CLOSED.
	 * @param listener the event listener to be added.
	 * @return true if the listener was added as a result of the call.
	 */
	public boolean addEventListener(String eventType, NotificationEventListener listener) {
		return this.eventListenerMap.get(eventType).add(listener);
	}

	/**
	 * Removes a listener previously added with addEventListener.
	 * @param eventType EVENT_TYPE_ACTION, EVENT_TYPE_CREATED or EVENT_TYPE_CLOSED.
	 * @param listener the event listener to be removed.
	 * @return true if the listener was removed as a result of the call.
	 */
	public boolean removeEventListener(String eventType, NotificationEventListener listener) {
		return this.eventListenerMap.get(eventType).remove(listener);
	}

	/**
	 * Get notification service provider status.
	 * @return new CompletionStage for the ProviderStatus.
	 */
	public CompletionStage<ProviderStatus> getProviderStatus() {
		return this.getChannelClient().thenCompose(channelClient -> {
			return channelClient.dispatchAsync("get-provider-status", new JSONObject()).thenApply(ack -> {
				return new ProviderStatus(((JSONObject) ack.getData()).getJSONObject("result"));
			});
		});
	}

}
