package com.openfin.desktop.snapshot;

import com.openfin.desktop.DesktopConnection;
import com.openfin.desktop.channel.ChannelAction;
import com.openfin.desktop.channel.ChannelProvider;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

/**
 *  SnapshotSource provides methods to init and register the SnapshotSourceProvider or to create a SnapshotSourceClient that connects
 *  to a snapshot source provider.
 */
public class SnapshotSource {
    private final static Logger logger = LoggerFactory.getLogger(SnapshotSource.class.getName());

    private final DesktopConnection desktopConnection;

    /**
     * Constructor of SnapshotSource
     *
     * @param desktopConnection an instance of DesktopConnection
     */
    public SnapshotSource(DesktopConnection desktopConnection) {
        this.desktopConnection = desktopConnection;
    }

    /**
     * Initializes the SnapshotSourceProvider with the getSnapshot and applySnapshot methods defined.
     *
     * @param uuid The identifier of the snapshot source.
     * @param provider The snapshot source provider
     * @return CompletionStage with the provider passed as the 2nd parameter
     */
    public CompletionStage<SnapshotSourceProvider> initSnapshotSourceProviderAsync(String uuid, SnapshotSourceProvider provider) {
        final CompletableFuture<SnapshotSourceProvider> future = new CompletableFuture<>();
        String cname = String.format("snapshot-source-provider-%s", uuid);
        logger.debug(String.format("initializing snapshot provider %s", cname));
        this.desktopConnection.getChannel(cname).createAsync().thenAccept(channelProvider -> {
            channelProvider.register("get-snapshot", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    return provider.getSnapshot();
                }
            });
            channelProvider.register("apply-snapshot", new ChannelAction() {
                @Override
                public JSONObject invoke(String action, Object payload, JSONObject senderIdentity) {
                    provider.applySnapshot((JSONObject) payload);
                    return null;
                }
            });
            future.complete(provider);
        });
        return future;
    }

    /**
     * Asynchronously returns a SnapshotSourceClient object that connects to the specified snapshot source provider.
     *
     * @param uuid The identifier of the snapshot source.
     * @return CompletionStage for an instance of SnapshotSourceClient when complete
     */
    public CompletionStage<SnapshotSourceClient> createSnapshotSourceClientAsync(String uuid) {
        final SnapshotSourceClient client = new SnapshotSourceClient(this.desktopConnection, uuid);
        return client.initChannelClientAsync();
    }

}
