/*
 * Decompiled with CFR 0.152.
 */
package com.openfin.desktop;

import com.openfin.desktop.Ack;
import com.openfin.desktop.AckListener;
import com.openfin.desktop.ActionEvent;
import com.openfin.desktop.DesktopConnection;
import com.openfin.desktop.DesktopException;
import com.openfin.desktop.EventListener;
import com.openfin.desktop.JsonUtils;
import com.openfin.desktop.Window;
import com.openfin.desktop.WindowBounds;
import com.openfin.desktop.WindowMember;
import com.openfin.desktop.win32.ExternalWindowObserver;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DockingManager {
    private static final Logger logger = LoggerFactory.getLogger((String)DockingManager.class.getName());
    private DesktopConnection desktopConnection;
    private int snappingDistance = 20;
    private Map<String, WindowMember> memberMap = new HashMap<String, WindowMember>();
    private List<WindowMember> dockCandidates = new ArrayList<WindowMember>();
    private BoundsChangingListener boundsChangingListener;
    private BoundsChangeListener boundsChangeListener;
    private String javaWindowParentUuid;

    public DockingManager(DesktopConnection desktopConnection, String javaWindowParentUuid) throws Exception {
        if (!desktopConnection.isConnected()) {
            logger.error("DesktopConnection is not connected");
            throw new DesktopException("DesktopConnection is not connected");
        }
        this.desktopConnection = desktopConnection;
        this.javaWindowParentUuid = javaWindowParentUuid;
        this.boundsChangeListener = new BoundsChangeListener();
        this.boundsChangingListener = new BoundsChangingListener();
        this.initRequestHandlers();
    }

    private void initRequestHandlers() throws Exception {
        this.desktopConnection.getInterApplicationBus().subscribe("*", "undock-window", (uuid, topic, data) -> this.processUndockRequest((JSONObject)data));
        this.desktopConnection.getInterApplicationBus().subscribe("*", "register-docking-window", (uuid, topic, data) -> this.processRegisterRequest((JSONObject)data));
        this.desktopConnection.getInterApplicationBus().subscribe("*", "unregister-docking-window", (uuid, topic, data) -> this.processUnregisterRequest((JSONObject)data));
    }

    public synchronized void registerWindow(Window window) {
        String newKey = WindowMember.getMemberKey(window);
        if (this.memberMap.get(newKey) == null) {
            this.addWindow(window);
        } else {
            logger.error(String.format("%s already registered", newKey));
        }
    }

    public void registerJavaWindow(final String javaWindowName, java.awt.Window window, final AckListener ackListener) {
        final CountDownLatch latch = new CountDownLatch(1);
        try {
            ExternalWindowObserver externalWindowObserver = new ExternalWindowObserver(this.desktopConnection.getPort().intValue(), this.javaWindowParentUuid, javaWindowName, window, new AckListener(){

                public void onSuccess(Ack ack) {
                    if (ack.isSuccessful()) {
                        Window ofWindow = Window.wrap((String)DockingManager.this.javaWindowParentUuid, (String)javaWindowName, (DesktopConnection)DockingManager.this.desktopConnection);
                        DockingManager.this.registerWindow(ofWindow);
                    }
                    latch.countDown();
                }

                public void onError(Ack ack) {
                    logger.error("Error registering java window ", (Object)ack.getReason());
                    latch.countDown();
                    if (ackListener != null) {
                        ackListener.onError(ack);
                    }
                }
            });
            latch.await(10L, TimeUnit.SECONDS);
            if (latch.getCount() == 0L) {
                WindowMember member = this.memberMap.get(WindowMember.getMemberKey(this.javaWindowParentUuid, javaWindowName));
                if (member != null) {
                    member.setExternalWindowObserver(externalWindowObserver);
                    this.ackSuccess(ackListener);
                } else {
                    this.ackError(ackListener, "Error registering Java window");
                }
            } else {
                this.ackError(ackListener, "Error registering Java window");
            }
        }
        catch (Exception e) {
            logger.error("Error registering external window", (Throwable)e);
            latch.countDown();
            this.ackError(ackListener, e.getMessage());
        }
    }

    private void ackSuccess(AckListener ackListener) {
        if (ackListener != null) {
            ackListener.onSuccess(new Ack(new JSONObject(), (Object)this));
        }
    }

    private void ackError(AckListener ackListener, String reason) {
        if (ackListener != null) {
            JSONObject obj = new JSONObject();
            obj.put("reason", (Object)reason);
            ackListener.onError(new Ack(obj, (Object)this));
        }
    }

    private void addWindow(Window window) {
        final WindowMember member = new WindowMember(window, this.desktopConnection);
        this.memberMap.put(member.getKey(), member);
        this.addBoundsChangeListener("disabled-frame-bounds-changing", member, this.boundsChangingListener);
        this.addBoundsChangeListener("disabled-frame-bounds-changed", member, this.boundsChangeListener);
        window.disableFrame(new AckListener(){

            public void onSuccess(Ack ack) {
                logger.debug(String.format("Frame disabled %s", member.getKey()));
            }

            public void onError(Ack ack) {
                logger.error(String.format("onError disable Frame disabled %s %s", member.getKey(), ack.getReason()));
            }
        });
        window.getBounds(bounds -> member.setBounds((WindowBounds)bounds), new AckListener(){

            public void onSuccess(Ack ack) {
            }

            public void onError(Ack ack) {
                logger.error(String.format("onError getBounds %s", ack.getReason()));
            }
        });
    }

    public synchronized void unregisterWindow(String applicationUuid, String windowName) {
        WindowMember member = this.memberMap.get(WindowMember.getMemberKey(applicationUuid, windowName));
        if (member != null) {
            this.removeWindow(member);
        } else {
            logger.error(String.format("Window not registered %s", WindowMember.getMemberKey(applicationUuid, windowName)));
        }
    }

    private void removeWindow(WindowMember member) {
        logger.debug(String.format("Removing %s", member.getKey()));
        this.memberMap.remove(member.getKey());
        member.getWindow().removeEventListener("disabled-frame-bounds-changing", (EventListener)this.boundsChangingListener, null);
        member.getWindow().removeEventListener("disabled-frame-bounds-changed", (EventListener)this.boundsChangeListener, null);
        member.getWindow().enableFrame(null);
        if (member.getExternalWindowObserver() != null) {
            try {
                member.getExternalWindowObserver().dispose();
            }
            catch (Exception e) {
                logger.error(String.format("Error disposing ExternalWindowObserver %s", member.getKey()), (Throwable)e);
            }
        }
    }

    public void dispose() {
        logger.debug("calling dispose");
        HashSet<WindowMember> members = new HashSet<WindowMember>(this.memberMap.values());
        members.forEach(m -> this.removeWindow((WindowMember)m));
    }

    private void addBoundsChangeListener(final String eventName, final WindowMember member, EventListener eventListener) {
        member.getWindow().addEventListener(eventName, eventListener, new AckListener(){

            public void onSuccess(Ack ack) {
                if (!ack.isSuccessful()) {
                    logger.error(String.format("Failed to add event listener %s to window %s", eventName, member.getKey()));
                }
            }

            public void onError(Ack ack) {
                logger.error(String.format("Failed to add event listener %s to window %s", eventName, member.getKey()));
            }
        });
    }

    private WindowBounds parseBounds(JSONObject data) {
        WindowBounds bounds = new WindowBounds(JsonUtils.getIntegerValue((JSONObject)data, (String)"top", null), JsonUtils.getIntegerValue((JSONObject)data, (String)"left", null), JsonUtils.getIntegerValue((JSONObject)data, (String)"width", null), JsonUtils.getIntegerValue((JSONObject)data, (String)"height", null));
        return bounds;
    }

    private WindowMember parseMember(JSONObject data) {
        WindowMember member = null;
        String appUuid = data.getString("uuid");
        String windowName = data.getString("name");
        if (appUuid != null && windowName != null) {
            member = this.memberMap.get(WindowMember.getMemberKey(appUuid, windowName));
        }
        return member;
    }

    private void onWindowMoving(WindowMember movingMember, WindowBounds bounds) {
        WindowBounds movingBounds = bounds;
        if (!movingMember.isDocked()) {
            this.dockCandidates.clear();
            for (WindowMember another : this.memberMap.values()) {
                WindowBounds snapBounds;
                if (another.equals(movingMember) || (snapBounds = this.shouldSnap(movingMember, movingBounds, another)) == null) continue;
                this.dockCandidates.add(movingMember);
                this.dockCandidates.add(another);
                logger.debug(String.format("Snapping %s", movingMember.getKey()));
                movingBounds = snapBounds;
                break;
            }
        } else {
            logger.debug("Bounds changing already docked " + movingMember.getKey());
        }
        movingMember.updateBounds(movingBounds);
    }

    private void onWindowMoved(WindowMember movingMember, WindowBounds bounds) {
        if (this.dockCandidates.size() == 2) {
            WindowMember movingCandidate = this.dockCandidates.get(0);
            WindowMember targetCandidate = this.dockCandidates.get(1);
            if (movingCandidate.equals(movingCandidate)) {
                targetCandidate.dock(movingCandidate);
            } else {
                logger.error(String.format("Docking candidate mismatch %s %s", movingMember.getKey(), movingCandidate.getKey()));
            }
            this.dockCandidates.clear();
        } else {
            movingMember.updateBounds(bounds);
        }
    }

    private WindowBounds shouldSnap(WindowMember movingMember, WindowBounds movingBounds, WindowMember anchorMember) {
        logger.debug(String.format("Checking shouldDock %s to %s", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
        WindowBounds newBounds = null;
        if (!movingMember.isDocked() || !anchorMember.isDocked()) {
            WindowBounds anchorBounds = anchorMember.getBounds();
            if (movingBounds != null && anchorBounds != null) {
                int bottom1 = movingBounds.getTop() + movingBounds.getHeight();
                int bottom2 = anchorBounds.getTop() + anchorBounds.getHeight();
                int right1 = movingBounds.getLeft() + movingBounds.getWidth();
                int right2 = anchorBounds.getLeft() + anchorBounds.getWidth();
                if (movingBounds.getLeft() < right2 && anchorBounds.getLeft() < right1) {
                    if (Math.abs(movingBounds.getTop() - bottom2) < this.snappingDistance) {
                        newBounds = new WindowBounds(movingBounds.getTop(), movingBounds.getLeft(), movingBounds.getWidth(), movingBounds.getHeight());
                        newBounds.setTop(Integer.valueOf(bottom2));
                        logger.debug(String.format("Detecting bottom-top docking %s to %s", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                    } else if (Math.abs(anchorBounds.getTop() - bottom1) < this.snappingDistance) {
                        newBounds = new WindowBounds(movingBounds.getTop(), movingBounds.getLeft(), movingBounds.getWidth(), movingBounds.getHeight());
                        newBounds.setTop(Integer.valueOf(anchorBounds.getTop() - movingBounds.getHeight()));
                        logger.debug(String.format("Detecting top-bottom docking %s to %s", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                    } else {
                        logger.debug(String.format("shouldDock %s to %s too far top-bottom", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                    }
                } else if (movingBounds.getTop() < bottom2 && anchorBounds.getTop() < bottom1) {
                    if (Math.abs(movingBounds.getLeft() - right2) < this.snappingDistance) {
                        newBounds = new WindowBounds(movingBounds.getTop(), movingBounds.getLeft(), movingBounds.getWidth(), movingBounds.getHeight());
                        newBounds.setLeft(Integer.valueOf(right2));
                        logger.debug(String.format("Detecting right-left docking %s to %s", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                    } else if (Math.abs(right1 - anchorBounds.getLeft()) < this.snappingDistance) {
                        newBounds = new WindowBounds(movingBounds.getTop(), movingBounds.getLeft(), movingBounds.getWidth(), movingBounds.getHeight());
                        newBounds.setLeft(Integer.valueOf(anchorBounds.getLeft() - movingBounds.getWidth()));
                        logger.debug(String.format("Detecting left-right docking %s to %s", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                    }
                } else {
                    logger.debug(String.format("shouldDock %s to %s not overlapping", movingMember.getWindow().getName(), anchorMember.getWindow().getName()));
                }
            }
        }
        return newBounds;
    }

    private void processUndockRequest(JSONObject request) {
        String appUuid = request.getString("applicationUuid");
        String windowName = request.getString("windowName");
        if (appUuid != null && windowName != null) {
            WindowMember member = this.memberMap.get(WindowMember.getMemberKey(appUuid, windowName));
            if (member != null) {
                if (member.isDocked()) {
                    member.undock();
                }
            } else {
                logger.error(String.format("Window not registered %s %s", appUuid, windowName));
            }
        } else {
            logger.error(String.format("Invalid request to duck window %s", request.toString()));
        }
    }

    private void processRegisterRequest(JSONObject request) {
        String appUuid = request.getString("applicationUuid");
        String windowName = request.getString("windowName");
        if (appUuid != null && windowName != null) {
            Window window = Window.wrap((String)appUuid, (String)windowName, (DesktopConnection)this.desktopConnection);
            this.registerWindow(window);
        } else {
            logger.error(String.format("Invalid request to duck window %s", request.toString()));
        }
    }

    private void processUnregisterRequest(JSONObject request) {
        String appUuid = request.getString("applicationUuid");
        String windowName = request.getString("windowName");
        if (appUuid != null && windowName != null) {
            this.unregisterWindow(appUuid, windowName);
        } else {
            logger.error(String.format("Invalid request to unregister window %s", request.toString()));
        }
    }

    class BoundsChangingListener
    implements EventListener {
        BoundsChangingListener() {
        }

        public void eventReceived(ActionEvent actionEvent) {
            JSONObject data = actionEvent.getEventObject();
            WindowMember member = DockingManager.this.parseMember(data);
            if (member != null) {
                WindowBounds bounds = DockingManager.this.parseBounds(data);
                DockingManager.this.onWindowMoving(member, bounds);
            } else {
                logger.error(String.format("Window not registered %s", data.toString()));
            }
        }
    }

    class BoundsChangeListener
    implements EventListener {
        BoundsChangeListener() {
        }

        public void eventReceived(ActionEvent actionEvent) {
            JSONObject data = actionEvent.getEventObject();
            WindowMember member = DockingManager.this.parseMember(data);
            if (member != null) {
                WindowBounds bounds = DockingManager.this.parseBounds(data);
                DockingManager.this.onWindowMoved(member, bounds);
            } else {
                logger.error(String.format("Window not registered %s", data.toString()));
            }
        }
    }
}

