/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.adapter;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.abiquo.event.model.enumerations.EntityAction;
import com.google.common.base.Enums;
import com.google.common.base.Optional;

/**
 * XmlAdapter implementation to generate the XML representation of a Map<String, String>. It is used
 * to un/marshal the constraints map of an hypervisor type.
 * 
 * @see HypervisorTypeDto#getConstraints()
 */
public class MapAdapter extends XmlAdapter<Element, Optional<Map<EntityAction.KEYS, String>>>
{
    private DocumentBuilder documentBuilder;

    private static final String ROOT_ELEMENT_NAME = "details";

    private DocumentBuilder getDocumentBuilder() throws Exception
    {
        // Lazy load the DocumentBuilder as it is not used for unmarshalling.
        if (null == documentBuilder)
        {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            documentBuilder = dbf.newDocumentBuilder();
        }
        return documentBuilder;
    }

    @Override
    public Element marshal(final Optional<Map<EntityAction.KEYS, String>> map) throws Exception
    {
        DocumentBuilder db = getDocumentBuilder();
        Document document = db.newDocument();
        Element rootElement = document.createElement(ROOT_ELEMENT_NAME);
        if (map.isPresent())
        {
            marshalMap(map.get(), rootElement, document);
        }
        document.appendChild(rootElement);
        return rootElement;
    }

    /**
     * Convert the given map in a child element of the given root element. Each key is a child
     * element with his value as text content.
     * 
     * @param map the map to marshal
     * @param rootElement the root element
     * @param document the XML document that we are building
     * @return the given root element with the new child element added
     */
    private static Element marshalMap(final Map<EntityAction.KEYS, String> map,
        final Element rootElement, final Document document)
    {
        for (Entry<EntityAction.KEYS, String> entry : map.entrySet())
        {
            Element element = document.createElement(entry.getKey().name().toLowerCase());
            Object value = entry.getValue();
            element.setTextContent(value == null ? "" : value.toString());
            rootElement.appendChild(element);
        }

        return rootElement;
    }

    @Override
    public Optional<Map<EntityAction.KEYS, String>> unmarshal(final Element element)
        throws Exception
    {
        if (null == element)
        {
            return null;
        }

        return unmarshalMap(element.getFirstChild(), new HashMap<EntityAction.KEYS, String>());
    }

    /**
     * Recursive function to traverse the given XML node and fill the given map
     * 
     * @param node the node to traverse
     * @param map the map to fill
     * @return the given map
     */
    private Optional<Map<EntityAction.KEYS, String>> unmarshalMap(final Node node,
        final Map<EntityAction.KEYS, String> map)
    {
        if (node.getNodeType() == Node.ELEMENT_NODE && node.getFirstChild() == null)
        {
            // <node_element />
            map.put(Enums.getIfPresent(EntityAction.KEYS.class, node.getNodeName())
                .or(EntityAction.KEYS.NOTIFICATION), null);
        }
        else if (node.getNodeType() == Node.ELEMENT_NODE
            && node.getFirstChild().getNodeType() == Node.TEXT_NODE)
        {
            // <node_element>text_node</node_element>
            map.put(Enums.getIfPresent(EntityAction.KEYS.class, node.getNodeName())
                .or(EntityAction.KEYS.NOTIFICATION), node.getFirstChild().getNodeValue());
        }

        if (node.getNextSibling() != null)
        {
            // <node_element>text_node</node_element><next_sibling>...
            unmarshalMap(node.getNextSibling(), map);
        }
        return Optional.fromNullable(map);
    }
}
