/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.interfaces;

import java.io.Serializable;
import java.util.Optional;

import com.abiquo.event.model.Event;
import com.abiquo.event.model.details.ErrorDetails;
import com.abiquo.event.model.details.EventDetails;
import com.abiquo.event.model.enumerations.EntityAction.Action;
import com.abiquo.event.model.enumerations.ErrorScope;
import com.abiquo.event.model.interfaces.AbiquoError;

/**
 * Implementations of this interface are meant to trace {@link Event} objects abroad.
 * 
 * @author <a href="mailto:susana.acedo@abiquo.com">Susana Acedo</a>
 * @author <a href="mailto:sergi.castro@abiquo.com">Sergi Castro</a>
 * @author <a href="mailto:serafin.sedano@abiquo.com">Serafin Sedano</a>
 */
public interface Traceable<ID extends Serializable>
{
    /**
     * Straight publication of the {@link Event}.<br>
     * If any validation is performed depends on actual implementation.
     * 
     * @param event to publish.
     */
    void log(Event event);

    /**
     * Publication of {@link Event} whose {@link Event#getEntityIdentifier()} does not apply, is not
     * known in the production context or is inferred from the context.
     * <p>
     * Whether is possible to infer the {@link Event#getEntityIdentifier()} from the context or how
     * it is done relies on the implementation.
     * <p>
     * <p>
     * In the case of the creation the preferred method is {@link #info(Action, ID)} if <b>no</b>
     * {@link EventDetails} is needed. {@link #info(Action, ID, EventDetails)} otherwise.
     * </p>
     * 
     * @see #info(Action, ID)
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     */
    void info(Action< ? > action, Optional<Integer> enterpriseId);

    /**
     * Create an {@link Event}, this implies the addition and formatting the message, add this
     * details.
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param details to add to the {@link Event}.
     */
    void info(Action< ? > action, Optional<Integer> enterpriseId, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param entity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     */
    void info(Action< ? > action, ID entity, Optional<Integer> enterpriseId);

    /**
     * Generates and traces a {@link Event} which is performed over the entity uri. This method is
     * in charge for composing the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param entityUri which receive the action.
     * @param enterpriseId enterprise id of the entity.
     */
    void info(Action< ? > action, String entityUri, Optional<Integer> enterpriseId);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID} add this details.
     * This method is in charge for composing the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param entity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param details extra information to add to the {@link Event}.
     */
    void info(Action< ? > action, ID entity, Optional<Integer> enterpriseId, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over the entity uri. This method is
     * in charge for composing the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param entityUri which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param details extra information to add to the {@link Event}.
     */
    void info(Action< ? > action, String entityUri, Optional<Integer> enterpriseId,
        EventDetails details);

    /**
     * Generates and traces a {@link Event} add this details. This method is in charge for composing
     * the message with the messageArgs if any.
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     */
    void warn(Action< ? > action, Optional<Integer> enterpriseId);

    /**
     * Generates and traces a {@link Event} add this details. This method is in charge for composing
     * the message with the messageArgs if any.
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param details extra information to add to the {@link Event}.
     */
    void warn(Action< ? > action, Optional<Integer> enterpriseId, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     */
    void warn(Action< ? > action, ID identity, Optional<Integer> enterpriseId);

    /**
     * Generates and traces a {@link Event} add this details. This method is in charge for composing
     * the message with the messageArgs if any.
     * <p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}. <b>Typical use case is a creation of an entity<b>.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param details extra information to add to the {@link Event}.
     */
    void warn(Action< ? > action, ID identity, Optional<Integer> enterpriseId,
        EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param scope which belongs the {@link Event}.
     */
    void error(Action< ? > action, Optional<Integer> enterpriseId, AbiquoError abiquoError,
        ErrorScope scope);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param e exception that was arouse.
     * @param scope which belongs the {@link Event}.
     */
    void error(Action< ? > action, Optional<Integer> enterpriseId, AbiquoError abiquoError,
        Throwable e, ErrorScope scope);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param scope which belongs the {@link Event}.
     */
    void error(Action< ? > action, ID identity, Optional<Integer> enterpriseId,
        AbiquoError abiquoError, ErrorScope scope);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param e exception that was arouse.
     * @param scope which belongs the {@link Event}.
     */
    void error(Action< ? > action, ID identity, Optional<Integer> enterpriseId,
        AbiquoError abiquoError, Throwable e, ErrorScope scope);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param scope which belongs the {@link Event}.
     * @param details extra information to add to the {@link Event}.
     */
    void error(Action< ? > action, Optional<Integer> enterpriseId, AbiquoError abiquoError,
        ErrorScope scope, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param e exception that was arouse.
     * @param scope which belongs the {@link Event}.
     * @param details extra information to add to the {@link Event}.
     */
    void error(Action< ? > action, Optional<Integer> enterpriseId, AbiquoError abiquoError,
        Throwable e, ErrorScope scope, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param scope which belongs the {@link Event}.
     * @param details extra information to add to the {@link Event}.
     */
    void error(Action< ? > action, ID identity, Optional<Integer> enterpriseId,
        AbiquoError abiquoError, ErrorScope scope, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * 
     * @param action action on an entity.
     * @param identity which receive the action.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param e exception that was arouse.
     * @param scope which belongs the {@link Event}.
     * @param details extra information to add to the {@link Event}.
     */
    void error(Action< ? > action, ID identity, Optional<Integer> enterpriseId,
        AbiquoError abiquoError, Throwable e, ErrorScope scope, EventDetails details);

    /**
     * Generates and traces a {@link Event} which is performed over a {@link ID}. This method is in
     * charge for composing the message with the messageArgs if any.
     * <p>
     * This method generates a {@link ErrorDetails} with the parameters.
     * </p>
     * This method is useful to make sure that the correct entity is added as
     * {@link Event#getEntityIdentifier()}.
     * 
     * @param action action on an entity.
     * @param enterpriseId enterprise id of the entity.
     * @param abiquoError abiquo error with code and message.
     * @param scope which belongs the {@link Event}.
     * @param details extra information to add to the {@link Event}.
     * @param entityUri which receive the action.
     */
    void error(Action< ? > action, Optional<Integer> enterpriseId, AbiquoError apiError,
        ErrorScope scope, EventDetails details, String entityUri);

}
