/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model;

import java.io.Serializable;
import java.util.TimeZone;

import javax.validation.constraints.NotNull;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import com.abiquo.event.adapter.OptionalMapAdapter;
import com.abiquo.event.model.EventAdapter.OptionalStringAdapter;
import com.abiquo.event.model.details.EventDetails;
import com.abiquo.event.model.enumerations.EntityAction.Action;
import com.abiquo.event.model.enumerations.Severity;
import com.abiquo.event.model.enumerations.Source;
import com.google.common.base.Objects;
import com.google.common.base.Optional;

/**
 * Event performed by abiquo.
 * 
 * @author <a href="mailto:sergi.castro@abiquo.com">Sergi Castro</a>
 * @author <a href="mailto:aprete@abiquo.com">Alessia Prete</a>
 */
@XmlRootElement(name = "event")
@XmlType(propOrder = {"timestamp", "user", "enterprise", "severity", "source", "action", "type",
"entityIdentifier", "details"})
public class Event implements Serializable
{

    /**
     * 
     */
    private static final long serialVersionUID = -5535795080620821917L;

    private long timestamp;

    private String timezone;

    private String user;

    private String enterprise;

    private Severity severity;

    private Source source;

    private String action;

    private String type;

    private Optional<String> entityIdentifier;

    private Optional< ? extends EventDetails> details;

    private Event(final String type, final String action, final String user,
        final String enterprise, final Source source, final Severity severity)
    {
        this.timestamp = System.currentTimeMillis();
        this.timezone = TimeZone.getDefault().getID();
        this.type = type;
        this.action = action;
        this.source = source;
        this.severity = severity;
        this.user = user;
        this.enterprise = enterprise;
    }

    /**
     * Not Use. Added for serialization purpose.
     */
    @Deprecated
    public Event()
    {
        this.timestamp = 0;
        this.timezone = "";
        this.type = "";
        this.action = "";
        this.source = null;
        this.severity = null;
        this.user = "";
        this.enterprise = "";
    }

    /**
     * @return WHEN the event has been performed.
     */
    @XmlElement(name = "timestamp")
    @NotNull
    public long getTimestamp()
    {
        return timestamp;
    }

    /**
     * @return time zone applied to the timestamp
     */
    @XmlElement(name = "timezone")
    @NotNull
    public String getTimezone()
    {
        return timezone;
    }

    /**
     * @return WHO has performed the event.
     */
    @XmlElement(name = "user")
    @NotNull
    public String getUser()
    {
        return user;
    }

    /**
     * @return The severity level of the performed event.
     */
    @XmlElement(name = "severity")
    @NotNull
    public Severity getSeverity()
    {
        return severity;
    }

    /**
     * @return The enterprise of the entity WHOM action has affected.
     */
    @XmlElement(name = "enterprise")
    @NotNull
    public String getEnterprise()
    {
        return enterprise;
    }

    /**
     * @return WHERE the event has been performed.
     */
    @XmlElement(name = "source")
    @NotNull
    public Source getSource()
    {
        return source;
    }

    /**
     * @return WHAT action has performed the event.
     */
    @XmlElement(name = "action")
    @NotNull
    public String getAction()
    {
        return action;
    }

    /**
     * @return The entity type WHOM action has affected.
     */
    @XmlElement(name = "type")
    @NotNull
    public String getType()
    {
        return type;
    }

    /**
     * @return An {@link Optional<String>} of an entity identifier WHOM action has affected.
     */
    @XmlJavaTypeAdapter(OptionalStringAdapter.class)
    public Optional<String> getEntityIdentifier()
    {
        return entityIdentifier;
    }

    /**
     * @return An {@link Optional<EventDetails>} with details about the performed event.
     */
    @XmlAnyElement
    @XmlJavaTypeAdapter(OptionalMapAdapter.class)
    public Optional< ? extends EventDetails> getDetails()
    {
        return details;
    }

    // Setter added for serialization purpose
    private void setTimestamp(final long timestamp)
    {
        this.timestamp = timestamp;
    }

    private void setTimezone(final String timezone)
    {
        this.timezone = timezone;
    }

    private void setUser(final String user)
    {
        this.user = user;
    }

    private void setEnterprise(final String enterprise)
    {
        this.enterprise = enterprise;
    }

    private void setSeverity(final Severity severity)
    {
        this.severity = severity;
    }

    private void setSource(final Source source)
    {
        this.source = source;
    }

    private void setAction(final String action)
    {
        this.action = action;
    }

    private void setType(final String type)
    {
        this.type = type;
    }

    private void setDetails(final Optional< ? extends EventDetails> details)
    {
        this.details = details;
    }

    public static EventBuilder builder(final Action< ? > entityAction, final String user,
        final String enterprise, final Source source, final Severity severity)
    {
        return new EventBuilder(entityAction, user, enterprise, source, severity);
    }

    /**
     * Builder helper for the event.
     */
    public static class EventBuilder
    {
        private Event event;

        /**
         * Initializes the builder with the event required properties.
         * 
         * @param entityAction
         * @param user
         * @param source
         * @param severity
         */
        private EventBuilder(final Action< ? > entityAction, final String user,
            final String enterprise, final Source source, final Severity severity)
        {
            String type = entityAction.entity();
            String action = entityAction.action();
            this.event = new Event(type, action, user, enterprise, source, severity);
            this.entityIdentifier(null);
            this.details(null);
        }

        /**
         * @return The event completed
         */
        public Event build()
        {
            return this.event;
        }

        /**
         * Sets an identifier of the event's entity.
         * 
         * @param value an identifier of the event's entity
         * @return the builder
         */
        public EventBuilder entityIdentifier(final String value)
        {
            this.event.entityIdentifier = Optional.fromNullable(value);
            return this;
        }

        /**
         * Sets a details object.
         * 
         * @param value a details object
         * @return the builder
         */
        public <E extends EventDetails> EventBuilder details(final E value)
        {
            this.event.details = Optional.fromNullable(value);
            return this;
        }
    }

    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = 1;
        result = prime * result + (action == null ? 0 : action.hashCode());
        result = prime * result + (details == null ? 0 : details.hashCode());
        result = prime * result + (entityIdentifier == null ? 0 : entityIdentifier.hashCode());
        result = prime * result + (severity == null ? 0 : severity.hashCode());
        result = prime * result + (source == null ? 0 : source.hashCode());
        result = prime * result + (int) (timestamp ^ timestamp >>> 32);
        result = prime * result + (type == null ? 0 : type.hashCode());
        result = prime * result + (user == null ? 0 : user.hashCode());
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        Event other = (Event) obj;
        if (action == null)
        {
            if (other.action != null)
            {
                return false;
            }
        }
        else if (!action.equals(other.action))
        {
            return false;
        }
        if (details == null)
        {
            if (other.details != null)
            {
                return false;
            }
        }
        else if (!details.equals(other.details))
        {
            return false;
        }
        if (entityIdentifier == null)
        {
            if (other.entityIdentifier != null)
            {
                return false;
            }
        }
        else if (!entityIdentifier.equals(other.entityIdentifier))
        {
            return false;
        }
        if (severity != other.severity)
        {
            return false;
        }
        if (source != other.source)
        {
            return false;
        }
        if (timestamp != other.timestamp)
        {
            return false;
        }
        if (type == null)
        {
            if (other.type != null)
            {
                return false;
            }
        }
        else if (!type.equals(other.type))
        {
            return false;
        }
        if (user == null)
        {
            if (other.user != null)
            {
                return false;
            }
        }
        else if (!user.equals(other.user))
        {
            return false;
        }
        return true;
    }

    @Override
    public String toString()
    {
        return Objects.toStringHelper(this).omitNullValues() //
            .add("severity", severity) //
            .add("type", type) //
            .add("action", action) //
            .add("timestamp", timestamp) //
            .add("user", user) //
            .add("enterprise", enterprise) //
            .add("entityIdentifier", entityIdentifier.orNull()) //
            .add("source", source) //
            .toString();
    }
}
