/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model;

import static java.lang.String.valueOf;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlValue;
import javax.xml.bind.annotation.adapters.XmlAdapter;

import com.abiquo.event.model.details.ErrorDetails;
import com.abiquo.event.model.details.EventDetails;
import com.abiquo.event.model.details.MessageDetails;
import com.abiquo.event.model.details.MoveDetails;
import com.abiquo.event.model.details.ReconfigureDetails;
import com.abiquo.event.model.interfaces.AbiquoKey;
import com.google.common.base.Optional;
import com.google.common.base.Strings;

public class EventAdapter
{

    public static class OptionalStringAdapter extends XmlAdapter<String, Optional<String>>
    {

        @Override
        public String marshal(final Optional<String> optional) throws Exception
        {
            return optional.or("");
        }

        @Override
        public Optional<String> unmarshal(final String optional) throws Exception
        {

            return Optional.fromNullable(Strings.emptyToNull(optional));
        }

    }

    public static class OptionalMapAdapter
        extends XmlAdapter<MapType, Optional< ? extends EventDetails>>
    {
        @Override
        public MapType marshal(final Optional< ? extends EventDetails> details)
        {
            Map<AbiquoKey, Object> map =
                details.isPresent() ? details.get() : new HashMap<AbiquoKey, Object>();
            MapType mapType = new MapType();
            for (java.util.Map.Entry<AbiquoKey, Object> entry : map.entrySet())
            {
                MapEntry mapEntry = new MapEntry();
                mapEntry.key = entry.getKey().name();
                mapEntry.value = entry.getValue();
                mapType.entryList.add(mapEntry);
            }
            return mapType;
        }

        @Override
        public Optional< ? extends EventDetails> unmarshal(final MapType type) throws Exception
        {
            if (type.entryList.isEmpty())
            {
                return Optional.absent();
            }

            return getDetails(type);
        }

        /**
         * Methods to return a {@link Optional<EventDetails>} with details about the performed
         * event.
         * 
         * @param type
         * @return
         */
        private Optional< ? extends EventDetails> getDetails(final MapType type)
        {
            boolean isError = Boolean.FALSE;
            boolean isMove = Boolean.FALSE;
            boolean isReconfigure = Boolean.FALSE;
            String scope = null;
            String code = null;
            String message = null;
            String targetUri = null;

            for (MapEntry thisEntry : type.entryList)
            {
                String key = thisEntry.key;
                if (key.equals(ErrorDetails.KEYS.SCOPE.name()))
                {
                    isError = Boolean.TRUE;
                    scope = valueOf(thisEntry.value);
                }
                else if (key.equals(ErrorDetails.KEYS.MESSAGE.name()))
                {
                    message = valueOf(thisEntry.value);
                }
                else if (key.equals(ErrorDetails.KEYS.CODE.name()))
                {
                    code = valueOf(thisEntry.value);
                }
                else if (key.equals(MoveDetails.KEYS.TARGETURI.name()))
                {
                    isMove = Boolean.TRUE;
                    targetUri = valueOf(thisEntry.value);
                }
                else if (key.equals(ReconfigureDetails.keys.CPU.name())
                    || key.equals(ReconfigureDetails.keys.RAM.name()))
                {
                    isReconfigure = Boolean.TRUE;
                }

            }
            if (isError)
            {
                return getOptional(type, ErrorDetails.builder(code, message, scope));
            }
            else if (isMove)
            {
                return getOptional(type, MoveDetails.builder(targetUri));
            }
            else if (isReconfigure)
            {
                return getOptional(type, ReconfigureDetails.builder());
            }

            return getOptional(type, MessageDetails.builder());
        }

        private Optional< ? extends EventDetails> getOptional(final MapType type,
            final com.abiquo.event.model.details.EventDetails.Builder< ? , ? extends EventDetails> b)
        {
            for (final MapEntry entry : type.entryList)
            {
                AbiquoKey key = new AbiquoKey()
                {
                    @Override
                    public String name()
                    {
                        return entry.key;
                    }
                };
                if (entry.value instanceof Collection)
                {
                    b.put(key, (Collection) entry.value);
                }
                else
                {
                    b.put(key, valueOf(entry.value));
                }
            }
            return Optional.of(b.build());
        }

    }

    private static class MapType
    {
        @XmlElement(name = "detail")
        public List<MapEntry> entryList = new ArrayList<MapEntry>();
    }

    private static class MapEntry
    {
        @XmlAttribute
        public String key;

        @XmlValue
        public Object value;

    }

}
