/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model.details;

import static com.google.common.collect.Iterables.transform;
import static java.lang.String.valueOf;

import java.util.List;

import javax.xml.bind.annotation.XmlRootElement;

import com.abiquo.event.model.enumerations.ErrorScope;
import com.abiquo.event.model.interfaces.AbiquoError;
import com.abiquo.event.model.interfaces.AbiquoKey;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;

/**
 * Decorator for map with the details of the performed event when the severity event is
 * {@link com.abiquo.event.model.enumerations.Severity#ERROR}.
 * <p>
 * An ErrorDetails always will report about the error code, error message and error scope of the
 * event, and may report the stracktrace of a thrown Exception.
 * </p>
 * 
 * @author <a href="mailto:sergi.castro@abiquo.com">Sergi Castro</a>
 */
@XmlRootElement
public class ErrorDetails extends EventDetails
{
    /**
     * Enumeration of all keys for this map
     */
    public static enum KEYS implements AbiquoKey
    {
        CODE, MESSAGE, SCOPE, STACKTRACE, ERROR_MESSAGE
    }

    /**
     * Creates the forwarding map using a copy of the given {@link ImmutableMap<AbiquoKey, String>}
     * map.
     * <p>
     * Must be a non-visible constructor to force the developer to use the {@link Builder}.
     * </p>
     * 
     * @param map the immutable map to create the event details
     */
    protected ErrorDetails(final ImmutableMap<AbiquoKey, Object> map)
    {
        super(map);
    }

    /**
     * @return the code of an abiquo api error
     */
    public String getCode()
    {
        return valueOf(get(KEYS.CODE));
    }

    /**
     * @return the message an abiquo api error
     */
    public String getMessage()
    {
        return valueOf(get(KEYS.MESSAGE));
    }

    /**
     * @return the scope affected by the error
     */
    public String getScope()
    {
        return valueOf(get(KEYS.SCOPE));
    }

    /**
     * @return the stacktrace generated by the error
     */
    public String getStacktrace()
    {
        return valueOf(get(KEYS.STACKTRACE));
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with the required parameters
     * {@link AbiquoError} error and {@link ErrorScope} scope.
     * 
     * @param error the {@link AbiquoError} which has promoted the event
     * @param scope the {@link ErrorScope} which has affected by the error of the event
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Builder< ? , ErrorDetails> builder(final AbiquoError error,
        final ErrorScope scope)
    {
        return new Builder(error, scope);
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with the required parameters
     * {@link AbiquoError} errors and {@link ErrorScope} scope.
     * 
     * @param errors the list of {@link AbiquoError} which have promoted the event
     * @param scope the {@link ErrorScope} which has affected by the error of the event
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Builder< ? , ErrorDetails> builder(final List<AbiquoError> errors,
        final ErrorScope scope)
    {
        return new Builder(errors, scope);
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with the required parameters
     * {@link AbiquoError} errors and {@link ErrorScope} scope and the optional parameter stacktrace
     * of the {@link Throwable}.
     * 
     * @param errors the {@link AbiquoError} which have promoted the event
     * @param scope the {@link ErrorScope} which has affected by the error of the event
     * @param stacktrace of the {@link Throwable} which has caused the error
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Builder< ? , ErrorDetails> builder(final List<AbiquoError> errors,
        final ErrorScope scope, final Throwable stacktrace)
    {
        return new Builder(errors, scope, stacktrace);
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with the required parameters
     * {@link AbiquoError} error and {@link ErrorScope} scope and the optional parameter stacktrace
     * of the {@link Throwable}.
     * 
     * @param error the {@link AbiquoError} which has promoted the event
     * @param scope the {@link ErrorScope} which has affected by the error of the event
     * @param stacktrace of the {@link Throwable} which has caused the error
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Builder< ? , ErrorDetails> builder(final AbiquoError error,
        final ErrorScope scope, final Throwable stacktrace)
    {
        return new Builder(error, scope, stacktrace);
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with no parameters. This is intended
     * package visibility for use by the {@link EventDetailsDeserializer}.
     * 
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    static Builder< ? , ErrorDetails> builder()
    {
        return new Builder();
    }

    /**
     * Returns a builder instance for {@link ErrorDetails} with the required parameters
     * 
     * @param code the code of an abiquo api error
     * @param message the message an abiquo api error
     * @param scope the stacktrace generated by the error
     * @return a builder instance for {@link ErrorDetails}
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public static Builder< ? , ErrorDetails> builder(final String code, final String message,
        final String scope)
    {
        return new Builder(code, scope, message);
    }

    /**
     * Builder helper for {@link ErrorDetails}.
     * 
     * @param <B> Any instance type of the hierarchy of {@link Builder}, required to allow the
     *            hierarchy in the builders.
     * @param <E> Any class extending of {@link ErrorDetails} which will be generated by the
     *            builder.
     */
    public static class Builder<B extends Builder<B, E>, E extends ErrorDetails>
        extends EventDetails.Builder<B, E>
    {
        /**
         * @see com.abiquo.events.model.details.EventDetails.DetailsBuilder#build()
         */
        @SuppressWarnings("unchecked")
        @Override
        public E build()
        {
            return (E) new ErrorDetails(this.mapBuilder.build());
        }

        /**
         * @see com.abiquo.model.details.EventDetails.Builder#self()
         */
        @SuppressWarnings("unchecked")
        @Override
        protected B self()
        {
            return (B) this;
        }

        /**
         * Initializes the builder with the required parameters {@link AbiquoError} error and
         * {@link ErrorScope} scope.
         * 
         * @param error the {@link AbiquoError} which has promoted the event
         * @param scope the {@link ErrorScope} which has affected by the error of the event
         */
        protected Builder(final AbiquoError error, final ErrorScope scope)
        {
            super();
            this.mapBuilder.put(KEYS.CODE, error.getCode());
            this.mapBuilder.put(KEYS.MESSAGE, error.getMessage());
            this.mapBuilder.put(KEYS.SCOPE, scope.name());
        }

        /**
         * Initializes the builder with the required parameters {@link AbiquoError} errors and
         * {@link ErrorScope} scope.
         * 
         * @param errors the {@link AbiquoError} which have promoted the event
         * @param scope the {@link ErrorScope} which has affected by the error of the event
         */
        protected Builder(final List<AbiquoError> errors, final ErrorScope scope)
        {
            super();
            this.mapBuilder.put(KEYS.CODE,
                Joiner.on(',').join(transform(errors, new Function<AbiquoError, String>()
                {
                    @Override
                    public String apply(final AbiquoError input)
                    {
                        return input.getCode();
                    }

                })));
            this.mapBuilder.put(KEYS.MESSAGE,
                Joiner.on('\n').join(transform(errors, new Function<AbiquoError, String>()
                {
                    @Override
                    public String apply(final AbiquoError input)
                    {
                        return input.getMessage();
                    }

                })));
            this.mapBuilder.put(KEYS.SCOPE, scope.name());
        }

        /**
         * Initializes the builder with the required parameters {@link AbiquoError} error and
         * {@link ErrorScope} scope and the optional parameter stacktrace of the {@link Throwable}.
         * 
         * @param error the {@link AbiquoError} which has promoted the event
         * @param scope the {@link ErrorScope} which has affected by the error of the event
         * @param stacktrace the stacktrace of the {@link Throwable} which has caused the error
         */
        protected Builder(final AbiquoError error, final ErrorScope scope,
            final Throwable throwable)
        {
            this(error, scope);
            this.mapBuilder.put(KEYS.STACKTRACE, getFormattedStracktrace(throwable));
            this.mapBuilder.put(KEYS.ERROR_MESSAGE,
                throwable.getMessage() != null ? throwable.getMessage() : "");
        }

        /**
         * Initializes the builder with the required parameters {@link AbiquoError} errors and
         * {@link ErrorScope} scope and the optional parameter stacktrace of the {@link Throwable}.
         * 
         * @param errors the {@link AbiquoError} which have promoted the event
         * @param scope the {@link ErrorScope} which has affected by the error of the event
         * @param stacktrace the stacktrace of the {@link Throwable} which has caused the error
         */
        protected Builder(final List<AbiquoError> errors, final ErrorScope scope,
            final Throwable throwable)
        {
            this(errors, scope);
            this.mapBuilder.put(KEYS.STACKTRACE, getFormattedStracktrace(throwable));
            this.mapBuilder.put(KEYS.ERROR_MESSAGE,
                throwable.getMessage() != null ? throwable.getMessage() : "");
        }

        /**
         * Sets a stacktrace generated by the {@link Throwable} which has caused the error.
         * 
         * @param value a stacktrace generated by the {@link Throwable} which has caused the error
         * @return the current instance of the {@link Builder}
         */
        public B stacktrace(final String stacktrace)
        {
            this.mapBuilder.put(KEYS.STACKTRACE, stacktrace);
            return self();
        }

        public B code(final String code)
        {
            this.mapBuilder.put(KEYS.CODE, code);
            return self();
        }

        public B error(final AbiquoError error)
        {
            this.mapBuilder.put(KEYS.CODE, error.getCode());
            this.mapBuilder.put(KEYS.MESSAGE, error.getMessage());
            return self();
        }

        public B message(final String message)
        {
            this.mapBuilder.put(KEYS.MESSAGE, message);
            return self();
        }

        public B scope(final String scope)
        {
            this.mapBuilder.put(KEYS.SCOPE, scope);
            return self();
        }

        /**
         * Initializes the builder with the required parameters
         * 
         * @param code the code of an abiquo api error
         * @param message the message an abiquo api error
         * @param scope the stacktrace generated by the error
         */
        public Builder(final String code, final String scope, final String message)
        {
            super();
            this.mapBuilder.put(KEYS.CODE, code);
            this.mapBuilder.put(KEYS.MESSAGE, message);
            this.mapBuilder.put(KEYS.SCOPE, scope);
        }

        /**
         * Initializes the builder with the required parameters
         * 
         * @param code the code of an abiquo api error
         * @param message the message an abiquo api error
         * @param scope the stacktrace generated by the error
         */
        public Builder()
        {
            super();
        }

    }

    /**
     * Extracts the stack trace elements and formatted them into a single string. If no stack trace
     * a empty string is returned.
     * 
     * @param throwable with a stacktrace or not.
     * @return Formatted stack trace if present in throwable empty otherwise.
     */

    public static String getFormattedStracktrace(final Throwable throwable)
    {

        StringBuilder stack = new StringBuilder("");
        if (throwable.getStackTrace() != null)
        {
            for (StackTraceElement e : throwable.getStackTrace())
            {
                stack.append(e.toString());
            }
        }

        return stack.toString();
    }

}
