/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model.details;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import com.abiquo.event.model.interfaces.AbiquoKey;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.google.common.collect.ForwardingMap;
import com.google.common.collect.ImmutableMap;

/**
 * Decorator for map with the details of the performed event.
 * 
 * @author <a href="mailto:sergi.castro@abiquo.com">Sergi Castro</a>
 */
@JsonDeserialize(using = EventDetailsDeserializer.class)
@JsonSerialize(using = EventDetailsSerializer.class)
public abstract class EventDetails extends ForwardingMap<AbiquoKey, Object> implements Serializable
{
    private static final long serialVersionUID = 1907281138673790063L;

    /**
     * The delegate of the forwarding map.
     */
    private final ImmutableMap<AbiquoKey, Object> delegate;

    /**
     * Creates the forwarding map using a copy of the given {@link ImmutableMap<AbiquoKey, String>}
     * map.
     * <p>
     * Must be a non-visible constructor to force the developer to use the {@link Builder}.
     * </p>
     * 
     * @param map the immutable map to create the event details
     */
    protected EventDetails(final ImmutableMap<AbiquoKey, Object> map)
    {
        this.delegate = ImmutableMap.copyOf(map);
    }

    /**
     * @see com.google.common.collect.ForwardingMap#delegate()
     */
    @Override
    protected Map<AbiquoKey, Object> delegate()
    {
        return delegate;
    }

    /**
     * <b>Resurrected method, seems necessary.</b>
     * <p>
     * From the beginning of the times we have a problem with {@link EventDetailsDeserializer} and
     * the keys of the map does not match. So we need a map instance transforming all
     * {@link AbiquoKey} keys to {@link String} in order to be available to properly retrieves the
     * values.
     * <p>
     * Fixing that this method should be removed.
     * <p>
     * JIRA issue: {@linkplain http://jira.abiquo.com/browse/ABICLOUDPREMIUM-7734}
     * 
     * @param map
     * @return
     */
    public Map<String, Object> getTransportMap()
    {
        Map<String, Object> newMap = new HashMap<String, Object>();
        for (Map.Entry<AbiquoKey, Object> entry : delegate().entrySet())
        {
            newMap.put(entry.getKey().name(), entry.getValue());
        }
        return newMap;
    }

    /**
     * Builder helper for extending classes of {@link EventDetails}.
     * 
     * @param <B> Any instance type of the hierarchy of {@link Builder}, required to allow the
     *            hierarchy in the builders.
     * @param <E> Any class extending of {@link EventDetails} which will be generated by the
     *            builder.
     */
    public static abstract class Builder<B extends Builder<B, E>, E extends EventDetails>
    {
        /**
         * The builder of the result immutable map.
         */
        protected ImmutableMap.Builder<AbiquoKey, Object> mapBuilder;

        /**
         * Initializes the builder.
         */
        protected Builder()
        {
            this.mapBuilder = ImmutableMap.builder();
        }

        /**
         * Returns an instance of the specific builder.
         * 
         * @return an instance of the specific builder
         */
        protected abstract B self();

        /**
         * Builds the {@link E} event details.
         * 
         * @return the event details built
         */
        public abstract E build();

        /**
         * Adds the given message arguments.
         * 
         * @param messageArgs the given message arguments
         * @return the builder instance
         */
        public B messageArgs(final Map< ? extends AbiquoKey, ? extends Object> messageArgs)
        {
            self().mapBuilder.putAll(messageArgs);
            return self();
        }

        /**
         * Associates {@code key} with {@code value} in the built map. Duplicate keys are not
         * allowed, and will cause {@link #build} to fail.
         * 
         * @param key
         * @param value
         * @return the builder instance
         */
        public B put(final AbiquoKey key, final String value)
        {
            self().mapBuilder.put(key, value);
            return self();
        }

        /**
         * Adds the given message arguments (as collection).
         * 
         * @param messageArgs the given message arguments
         * @return the builder instance
         */
        public B messageArgsCollection(
            final Map< ? extends AbiquoKey, Collection<String>> messageArgs)
        {
            self().mapBuilder.putAll(messageArgs);
            return self();
        }

        /**
         * Associates {@code key} with {@code value} (as collection) in the built map. Duplicate
         * keys are not allowed, and will cause {@link #build} to fail.
         * 
         * @param key
         * @param value
         * @return the builder instance
         */
        public B put(final AbiquoKey key, final Collection<String> value)
        {
            self().mapBuilder.put(key, value);
            return self();
        }

    }

    @Override
    public int hashCode()
    {
        final int prime = 31;
        int result = super.hashCode();
        result = prime * result + (delegate == null ? 0 : delegate.hashCode());
        return result;
    }

    @Override
    public boolean equals(final Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (!super.equals(obj))
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        EventDetails other = (EventDetails) obj;
        if (delegate == null)
        {
            if (other.delegate != null)
            {
                return false;
            }
        }
        else if (!delegate.equals(other.delegate))
        {
            return false;
        }
        return true;
    }

}
