/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model.details;

import static com.google.common.collect.Iterators.transform;
import static com.google.common.collect.Lists.newArrayList;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Objects;

import com.abiquo.event.model.details.EventDetails.Builder;
import com.abiquo.event.model.interfaces.AbiquoKey;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.ObjectCodec;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

/**
 * Custom {@link JsonDeserializer} for {@link EventDetails} hierarchy.
 * 
 * @author <a href="mailto:serafin.sedano@abiquo.com">Serafin Sedano</a>
 */
public class EventDetailsDeserializer extends JsonDeserializer<EventDetails>
{

    @Override
    public EventDetails deserialize(final JsonParser jp, final DeserializationContext ctxt)
        throws IOException, JsonProcessingException
    {
        ObjectCodec oc = jp.getCodec();
        JsonNode node = oc.readTree(jp);
        return getDetails(node);
    }

    private static EventDetails getDetails(final JsonNode node)
    {
        Builder< ? extends Builder< ? , ? extends EventDetails>, ? extends EventDetails> builder =
            getDetailsBuilder(ImmutableList.<String> copyOf(node.fieldNames()));

        com.google.common.collect.ImmutableMap.Builder<AbiquoKey, Object> mapBuilder =
            createMap(node.fields());

        return builder.messageArgs(mapBuilder.build()).build();
    }

    private final static Function<JsonNode, String> jsonNodeAsText =
        new Function<JsonNode, String>()
        {
            @Override
            public String apply(final JsonNode input)
            {
                return input.asText();
            }
        };

    private static com.google.common.collect.ImmutableMap.Builder<AbiquoKey, Object> createMap(
        final Iterator<Entry<String, JsonNode>> fields)
    {
        com.google.common.collect.ImmutableMap.Builder<AbiquoKey, Object> mapBuilder =
            ImmutableMap.builder();
        while (fields.hasNext())
        {
            final Entry<String, JsonNode> entry = fields.next();

            mapBuilder.put(new AbiquoKey()
            {
                private final String name = entry.getKey();

                @Override
                public String name()
                {
                    return name;
                }

                @Override
                public String toString()
                {
                    return name();
                }

                /**
                 * This class is meant to be equivalent to a enum. Thats why this is overriden.
                 */
                @Override
                public boolean equals(final Object obj)
                {
                    if (obj instanceof AbiquoKey)
                    {
                        return Objects.equals(name, ((AbiquoKey) obj).name());
                    }
                    return false;
                }
            }, entry.getValue().isArray() ? //
                newArrayList(transform(entry.getValue().elements(), jsonNodeAsText)) : //
                entry.getValue().asText());
        }
        return mapBuilder;
    }

    private static EventDetails.Builder< ? extends Builder< ? , ? extends EventDetails>, ? extends EventDetails> getDetailsBuilder(
        final List<String> keys)
    {
        if (keys.contains(ErrorDetails.KEYS.SCOPE.name())
            || keys.contains(ErrorDetails.KEYS.MESSAGE.name())
            || keys.contains(ErrorDetails.KEYS.CODE.name()))
        {
            return ErrorDetails.builder();
        }
        else if (keys.contains(MoveDetails.KEYS.TARGETURI.name()))
        {
            return MoveDetails.builder();
        }
        else if (keys.contains(ReconfigureDetails.keys.CPU.name())
            || keys.contains(ReconfigureDetails.keys.RAM.name()))
        {
            return ReconfigureDetails.builder();
        }
        return MessageDetails.builder();
    }
}
