/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model.details;

import static java.lang.String.valueOf;

import javax.xml.bind.annotation.XmlRootElement;

import com.abiquo.event.model.interfaces.AbiquoKey;
import com.google.common.collect.ImmutableMap;

/**
 * Decorator for map with the details of the performed event when a resource has changed its uri
 * identifier.
 * <p>
 * This details always will report the new uri of the resource.
 * </p>
 * 
 * @author <a href="mailto:sergi.castro@abiquo.com">Sergi Castro</a>
 */
@XmlRootElement
public class MoveDetails extends EventDetails
{
    /**
     * Enumeration of all keys for this map
     */
    public static enum KEYS implements AbiquoKey
    {
        TARGETURI
    }

    /**
     * Creates the forwarding map using a copy of the given {@link ImmutableMap<AbiquoKey, String>}
     * map.
     * <p>
     * Must be a non-visible constructor to force the developer to use the {@link Builder}.
     * </p>
     * 
     * @param map the immutable map to create the event details
     */
    protected MoveDetails(final ImmutableMap<AbiquoKey, Object> map)
    {
        super(map);
    }

    /**
     * @return the target uri where the resource has moved
     */
    public String getTargetUri()
    {
        return valueOf(get(KEYS.TARGETURI));
    }

    /**
     * Returns a builder instance for {@link MoveDetails} with the required parameter targetUri.
     * 
     * @param targetUri the new uri of the resource
     * @return a builder instance for {@link MoveDetails}
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static Builder< ? , MoveDetails> builder(final String targetUri)
    {
        return new Builder(targetUri);
    }

    /**
     * Returns a builder instance for {@link MoveDetails} with no parameters. This is meant to be in
     * package visibility for use in {@link EventDetailsDeserializer}.
     * 
     * @return a builder instance for {@link MoveDetails}
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    static Builder< ? , MoveDetails> builder()
    {
        return new Builder();
    }

    /**
     * Builder helper for {@link MoveDetails}.
     * 
     * @param <B> Any instance type of the hierarchy of {@link Builder}, required to allow the
     *            hierarchy in the builders.
     * @param <E> Any class extending of {@link MoveDetails}
     */
    public static class Builder<C extends Builder<C, E>, E extends MoveDetails>
        extends EventDetails.Builder<C, E>
    {

        /**
         * Initializes the builder with the builder with the required parameter targetUri.
         * 
         * @param targetUri the new uri of the resource
         */
        protected Builder()
        {
            super();
        }

        /**
         * Initializes the builder with the builder with the required parameter targetUri.
         * 
         * @param targetUri the new uri of the resource
         */
        protected Builder(final String targetUri)
        {
            super();
            this.mapBuilder.put(KEYS.TARGETURI, targetUri);
        }

        public C targetUri(final String targetUri)
        {
            this.mapBuilder.put(KEYS.TARGETURI, targetUri);
            return self();
        }

        /**
         * @see com.abiquo.model.details.EventDetails.Builder#self()
         */
        @SuppressWarnings("unchecked")
        @Override
        protected C self()
        {
            return (C) this;
        }

        /**
         * @see com.abiquo.model.details.EventDetails.Builder#build()
         */
        @SuppressWarnings("unchecked")
        @Override
        public E build()
        {
            return (E) new MoveDetails(this.mapBuilder.build());
        }

    }
}
