/**
 * The Abiquo Platform
 * Cloud management application for hybrid clouds
 * Copyright (C) 2008 - Abiquo Holdings S.L.
 *
 * This application is free software; you can redistribute it and/or
 * modify it under the terms of the GNU LESSER GENERAL PUBLIC
 * LICENSE as published by the Free Software Foundation under
 * version 3 of the License
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * LESSER GENERAL PUBLIC LICENSE v.3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.abiquo.event.model.details;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.valueOf;

import javax.xml.bind.annotation.XmlRootElement;

import com.abiquo.event.model.interfaces.AbiquoKey;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

/**
 * Forwarding map to add further details of a virtual machine reconfigure. Works based on
 * <b>diff</b> style.
 * <p>
 * Example: {CPU: +1, RAM: +1024}
 * </p>
 * 
 * @author <a href="mailto:serafin.sedano@abiquo.com">Serafin Sedano</a>
 */
@XmlRootElement
public class ReconfigureDetails extends EventDetails
{
    /**
     * All data susceptible to change in a virtual machine reconfigure.
     */
    public static enum keys implements AbiquoKey
    {
        CPU, RAM, CORES_PER_SOCKET, DVD, REMOTE_ACCESS
    }

    /**
     * Value for a enabled key
     */
    public static String ENABLED_VALUE = "+";

    /**
     * Value for a disabled key
     */
    public static String DISABLED_VALUE = "-";

    /**
     * Creates the forwarding map using a copy of the given {@link ImmutableMap<AbiquoKey, String>}
     * map.
     * <p>
     * Must be a non-visible constructor to force the developer to use the {@link Builder}.
     * </p>
     * 
     * @param map the immutable map to create the event details
     */
    protected ReconfigureDetails(final ImmutableMap<AbiquoKey, Object> map)
    {
        super(map);
    }

    /**
     * @return the cpu diff of the reconfigure.
     */
    public String getCpuDiff()
    {
        return valueOf(get(keys.CPU));
    }

    /**
     * @return the ram diff of the reconfigure.
     */
    public String getRamDiff()
    {
        return valueOf(get(keys.RAM));
    }

    /**
     * @return the cores per socket diff of the reconfigure
     */
    public String getCoresPerSocketDiff()
    {
        return valueOf(get(keys.CORES_PER_SOCKET));
    }

    /**
     * @return the dvd diff of the reconfigure, {@code "+"} if enabled, {@code "-"} if disabled and
     *         {@code null} if unchanged
     */
    public String getDvdDiff()
    {
        return valueOf(get(keys.DVD));
    }

    /**
     * @return the remote access diff of the reconfigure, {@code "+"} if enabled, {@code "-"} if
     *         disabled and {@code null} if unchanged
     */
    public String getRemoteAccessDiff()
    {
        return valueOf(get(keys.REMOTE_ACCESS));
    }

    /**
     * Returns a builder instance for {@link ReconfigureDetails}.
     * 
     * @return a builder instance for {@link ReconfigureDetails}
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static Builder< ? , ReconfigureDetails> builder()
    {
        return new Builder();
    }

    /**
     * Builder helper for {@link ReconfigureDetails}.
     * 
     * @param <B> Any instance type of the hierarchy of {@link Builder}, required to allow the
     *            hierarchy in the builders.
     * @param <E> Any class extending of {@link ReconfigureDetails}
     */
    public static class Builder<B extends Builder<B, E>, E extends ReconfigureDetails>
        extends EventDetails.Builder<B, E>
    {
        /**
         * Initializes the builder.
         */
        protected Builder()
        {
            super();
            this.mapBuilder = ImmutableMap.builder();
        }

        /**
         * Sets the cpu diff of the reconfigure.
         * 
         * @param value cpu diff of the reconfigure
         * @return the current instance of the {@link Builder}
         */
        public B cpuDiff(final String cpuDiff)
        {
            this.mapBuilder.put(keys.CPU, cpuDiff);
            return self();
        }

        /**
         * Sets the ram diff of the reconfigure.
         * 
         * @param value ram diff of the reconfigure
         * @return the current instance of the {@link Builder}
         */
        public B ramDiff(final String ramDiff)
        {
            this.mapBuilder.put(keys.RAM, ramDiff);
            return self();
        }

        /**
         * Sets the cores per socket diff of the reconfigure.
         * 
         * @param coresPerSocketDiff per socket diff of the reconfigure
         * @return the current instance of the {@link Builder}
         */
        public B coresPerSocketDiff(final String coresPerSocketDiff)
        {
            this.mapBuilder.put(keys.CORES_PER_SOCKET, coresPerSocketDiff);
            return self();
        }

        /**
         * Sets the dvd diff of the reconfigure.
         * 
         * @param dvdDiff Must be not null or empty and one of these values: {@code "+"} and
         *            {@code "-"}, otherwise an {@link IllegalArgumentException} will be thrown
         * @return the current instance of the {@link Builder}
         * @throws {@link IllegalArgumentException} with and invalid parameter {@code dvdDiff}
         */
        public B dvdDiff(final String dvdDiff)
        {
            checkArgument(!Strings.isNullOrEmpty(dvdDiff), "dvdDiff can not be null or empty");
            checkArgument(ENABLED_VALUE.equals(dvdDiff) || DISABLED_VALUE.equals(dvdDiff),
                "Invalid value for dvd diff");
            this.mapBuilder.put(keys.DVD, dvdDiff);
            return self();
        }

        /**
         * Sets the remote access diff of the reconfigure.
         * 
         * @param remoteAccessDiff Must be not null or empty and one of these values: {@code "+"}
         *            and {@code "-"}, otherwise an {@link IllegalArgumentException} will be thrown
         * @return the current instance of the {@link Builder}
         * @throws {@link IllegalArgumentException} with and invalid parameter
         *             {@code remoteAccessDiff}
         */
        public B remoteAccessDiff(final String remoteAccessDiff)
        {
            checkArgument(!Strings.isNullOrEmpty(remoteAccessDiff),
                "dvdremoteAccessDiff can not be null or empty");
            checkArgument(
                ENABLED_VALUE.equals(remoteAccessDiff) || DISABLED_VALUE.equals(remoteAccessDiff),
                "Invalid value for remote access diff");
            this.mapBuilder.put(keys.REMOTE_ACCESS, remoteAccessDiff);
            return self();
        }

        /**
         * @see com.abiquo.events.model.details.EventDetails.DetailsBuilder#build()
         */
        @SuppressWarnings("unchecked")
        @Override
        public E build()
        {
            return (E) new ReconfigureDetails(this.mapBuilder.build());
        }

        /**
         * @see com.abiquo.model.details.EventDetails.Builder#self()
         */
        @SuppressWarnings("unchecked")
        @Override
        protected B self()
        {
            return (B) this;
        }
    }
}
