
package actionjava.anim.core;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import actionjava.anim.config.Manifest;
import actionjava.anim.ease.Ease;
import actionjava.anim.events.TweenEvent;
import actionjava.anim.events.TweenEventHandler;
import actionjava.anim.events.TweenEventParams;
import actionjava.anim.events.TweenEvent.TweenEventType;

public class Tween
{
	// Targets
	private Object target;
	private TweenContainer parent;
	private TweenProp[] tweenProps;

	// Flags
	private boolean initialized = false;
	private boolean active = false;
	private boolean complete = false;

	// Ease
	private Ease ease;
	private double easeType;
	private double easePower;

	// Time & Progress Related
	private double ratio;
	private double startTime;
	private double elapsedTime;
	private double duration;

	// Events
	private TweenEvent updateEvent;
	private TweenEventHandler updateListener;
	private Map<TweenEventType, TweenEventParams> params;
	private Map<TweenEventType, TweenEventHandler> listeners;

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Constructor & Initialize
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public Tween(Manifest manifest)
	{
		super();
		initialize(manifest);
	}

	private void initialize(Manifest manifest)
	{
		target = manifest.target();
		tweenProps = manifest.tweenProps();

		params = new HashMap<TweenEventType, TweenEventParams>();
		params.putAll(manifest.getEventParams());
		listeners = new HashMap<TweenEventType, TweenEventHandler>();
		listeners.putAll(manifest.getEventListeners());
		updateListener = getListenerByType(TweenEventType.UPDATE);
		updateEvent = new TweenEvent(TweenEventType.UPDATE, getParamsByType(TweenEventType.UPDATE));

		ease = manifest.ease();
		easeType = ease.getType();
		easePower = ease.getPower();

		ratio = 0;
		elapsedTime = 0;
		duration = manifest.duration();

		initializeProps();
		initialized = true;
		active = true;
		dispatchEvent(TweenEventType.START);
	}

	private void initializeProps()
	{
		TweenProp tp;
		Iterator<TweenProp> tpIterator = getIterator();
		while (tpIterator.hasNext()) {
			tp = tpIterator.next();
			// tp.setStart(tp.get());
			tp.setChange(tp.getEnd() - tp.getStart());
		}
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Functionality Before & After Render
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public double beforeRender()
	{
		if(elapsedTime >= duration) {
			complete();
		} else {
			update();
		}

		return ratio;
	}

	public void afterRender()
	{
		dispatchUpdate();
		if(complete) {
			active = false;
			parent.removeChild(this);
			dispatchEvent(TweenEventType.COMPLETE);
		}
	}

	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Calculations
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	private void update()
	{
		if(easeType != 0) {
			double r = elapsedTime / duration;
			if(easeType == 1 || (easeType == 3 && r >= 0.5)) {
				r = 1 - r;
			}
			if(easeType == 3) {
				r *= 2;
			}
			if(easePower == 1) {
				r *= r;
			} else if(easePower == 2) {
				r *= r * r;
			} else if(easePower == 3) {
				r *= r * r * r;
			} else if(easePower == 4) {
				r *= r * r * r * r;
			}
			if(easeType == 1) {
				ratio = 1 - r;
			} else if(easeType == 2) {
				ratio = (r);
			} else if(elapsedTime / duration < 0.5) {
				ratio = r / 2;
			} else {
				ratio = 1 - (r / 2);
			}
		} else {
			ratio = ease.getRatio(elapsedTime / duration);
		}
	}

	private void complete()
	{
		complete = true;
		elapsedTime = duration;
		if(active) {
			ratio = (ease.isCalcEnd()) ? ease.getRatio(1) : 1;
		}
	}

	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Events
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	private void dispatchUpdate()
	{
		if(updateListener != null) {
			updateListener.handleEvent(updateEvent);
		}
	}

	private void dispatchEvent(TweenEventType type)
	{
		TweenEventHandler listener;
		if((type != null) && (listener = getListenerByType(type)) != null) {
			listener.handleEvent(new TweenEvent(type, getParamsByType(type)));
		}
	}

	private TweenEventParams getParamsByType(TweenEventType type)
	{
		TweenEventParams eParams;
		return ((eParams = params.get(type)) != null) ? eParams : null;
	}

	private TweenEventHandler getListenerByType(TweenEventType type)
	{
		TweenEventHandler listener;
		return ((listener = listeners.get(type)) != null) ? listener : null;
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Getters & Setters
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public Object getTarget()
	{
		return target;
	}

	public TweenContainer getParent()
	{
		return parent;
	}

	public void setParent(TweenContainer parent)
	{
		this.parent = parent;
	}

	public Iterator<TweenProp> getIterator()
	{
		return new TweenPropIterator();
	}

	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Flags
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	public boolean isInitialized()
	{
		return initialized;
	}

	public boolean isActive()
	{
		return active;
	}

	public void setActive(boolean val)
	{
		active = val;
	}

	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Time
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	public double getStartTime()
	{
		return startTime;
	}

	public void setStartTime(double startTime)
	{
		this.startTime = startTime;
	}

	public double getElapsedTime()
	{
		return elapsedTime;
	}

	public void setElapsedTime(double elapsedTime)
	{
		this.elapsedTime = elapsedTime;
	}

	public double getDuration()
	{
		return duration;
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Tween Properties Iterator
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	private class TweenPropIterator implements Iterator<TweenProp>
	{
		private int pos = 0;

		@Override
		public boolean hasNext()
		{
			return (pos < tweenProps.length) ? true : false;
		}

		@Override
		public TweenProp next()
		{
			TweenProp prop = tweenProps[pos];
			pos++;
			return prop;
		}

		@Override
		public void remove()
		{
			// IGNORE
		}
	}
}
