
package com.actionjava.mvn.plugins.assets.parsers;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import com.actionjava.mvn.plugins.assets.data.Instance;
import com.actionjava.mvn.plugins.assets.data.Shape;
import com.actionjava.mvn.plugins.assets.data.Stage;
import com.actionjava.mvn.plugins.assets.data.Symbol;
import com.actionjava.mvn.plugins.assets.data.instr.Instruction;
import com.actionjava.mvn.plugins.assets.io.AssetReader;
import com.actionjava.mvn.plugins.assets.io.AssetWriter;
import com.actionjava.mvn.plugins.assets.vo.AssetSettings;

public class AssetParser extends AbstractParser
{
	private static final String REGEX_MANIFEST_FLAG_1 = "^//(\\s*)library properties:(\\s*)$";
	private static final String REGEX_MANIFEST_FLAG_2 = "^lib\\.properties = \\{(\\s*)$";
	private static final String REGEX_MANIFEST_END = "^};(\\s*)$";

	private static final String REGEX_STAGE_FLAG = "^//(\\s*)stage content:(\\s*)$";
	private static final String REGEX_SYMBOL_FLAG = "^//(\\s*)symbols:(\\s*)$";
	private static final String REGEX_BEGIN = "^(lib.[\\w] = function() {";
	private static final String REGEX_END = "^\\}\\)\\.prototype(.*);$";
	private static final String REGEX_BOUNDS = "^p\\.nominalBounds(.*);$";
	private static final String REGEX_DOC_END_1 = "^\\}\\)\\(lib = lib\\|\\|\\{\\}, images = images\\|\\|\\{\\}, createjs = createjs\\|\\|\\{\\}(.*);$";
	private static final String REGEX_DOC_END_2 = "^var lib, images, createjs;$";

	private AssetReader reader;
	private AssetWriter writer;

	private ManifestParser manifestParser;
	private StageParser stageParser;
	private SymbolParser symbolParser;

	private Stage stage;
	private List<Instance> instances;
	private List<Shape> shapes;
	private List<Symbol> symbols;

	private AssetSettings settings;
	private File baseDirectory;

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Constructor & initialize
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public AssetParser(AssetSettings settings, File baseDirectory)
	{
		super();
		initialize(settings, baseDirectory);
	}

	private void initialize(AssetSettings assetSettings, File baseDirectory)
	{
		settings = assetSettings;
		this.baseDirectory = baseDirectory;

		initializeIO();
		initializeParsers();
		initializeInstances();
	}

	private void initializeIO()
	{
		File assetJS;
		try {
			assetJS = new File(settings.getAssetImportDirectory().getAbsoluteFile() + "/" + settings.getName() + ".js");
		} catch (Exception e) {
			e.printStackTrace();
			return;
		}

		reader = new AssetReader(assetJS);
		writer = new AssetWriter(settings.getAssetExportDirectory());
	}

	private void initializeParsers()
	{
		manifestParser = new ManifestParser(baseDirectory);
		symbolParser = new SymbolParser(manifestParser, settings.getAssetPackage());
		stageParser = new StageParser(symbolParser);
	}

	private void initializeInstances()
	{
		stage = new Stage(settings.getName(), settings.getName());
		shapes = new ArrayList<Shape>();
		symbols = new ArrayList<Symbol>();
		instances = new ArrayList<Instance>();
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Public functionality
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public void read()
	{
		reader.read();
	}

	public void parse()
	{
		String[] lines = reader.getAssetLines();
		String[] manifestRaw = getRawManifestData(lines);
		String[] stageRaw = getRawStageData(lines);
		String[] symbolsRaw = getRawSymbolData(lines);

		manifestParser.parse(manifestRaw);
		this.symbols = Arrays.asList(symbolParser.parse(symbolsRaw));
		stageParser.parse(stageRaw);
	}

	public void write()
	{
		writer.write(this.symbols);
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Private functionality
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	private String[] getRawManifestData(String[] lines)
	{
		int manifestFlagIndex1 = getLineNumberByRegEx(lines, REGEX_MANIFEST_FLAG_1);
		if(manifestFlagIndex1 != -1) {
			int manifestFlagIndex2 = getLineNumberByRegExFromIndex(lines, REGEX_MANIFEST_FLAG_2, manifestFlagIndex1);
			int manifestEndIndex = getLineNumberByRegExFromIndex(lines, REGEX_MANIFEST_END, manifestFlagIndex2) + 1;

			return copyRangeFromArray(lines, manifestFlagIndex2, manifestEndIndex);
		}

		return null;
	}

	private String[] getRawStageData(String[] lines)
	{
		int stageFlagIndex = getLineNumberByRegEx(lines, REGEX_STAGE_FLAG) + 1;
		int stageEndIndex1 = getLineNumberByRegExFromIndex(lines, REGEX_END, stageFlagIndex);
		int stageEndIndex2 = getLineNumberByRegExFromIndex(lines, REGEX_BOUNDS, stageEndIndex1) + 1;

		return copyRangeFromArray(lines, stageFlagIndex, stageEndIndex2);
	}

	private String[] getRawSymbolData(String[] lines)
	{
		int stageFlagIndex = getLineNumberByRegEx(lines, REGEX_STAGE_FLAG);
		int stageEndIndex1 = getLineNumberByRegExFromIndex(lines, REGEX_END, stageFlagIndex);
		int stageEndIndex2 = getLineNumberByRegExFromIndex(lines, REGEX_BOUNDS, stageEndIndex1) + 1;
		String[] withoutStage = removeRangeFromArray(lines, stageFlagIndex, stageEndIndex2);

		int symbolFlagIndex = getLineNumberByRegEx(withoutStage, REGEX_SYMBOL_FLAG) + 1;
		int symbolEndIndex = getLineNumberByRegExFromIndex(withoutStage, REGEX_DOC_END_1, symbolFlagIndex);

		return copyRangeFromArray(withoutStage, symbolFlagIndex, symbolEndIndex);
	}

	/* -- TO BE REMOVED -- */

	private void logSymbol(Symbol symbol)
	{
		System.out.println("SYMBOL: ");
		System.out.println("id: " + symbol.getID());
		System.out.println("name: " + symbol.getName());
		System.out.println("children: " + symbol.getChildren().length);
		System.out.println("");
		logSymbolChildren(symbol.getChildren());
	}

	private void logSymbolChildren(Instance[] children)
	{
		Instance child;
		Shape childShape;
		Symbol childSymbol;
		for(int i = 0;i < children.length;i++) {
			child = children[i];
			if(child instanceof Symbol) {
				childSymbol = (Symbol) child;
				// logSymbol(childSymbol);

			} else if(child instanceof Shape) {
				childShape = (Shape) child;
				logShape(childShape);

			}
		}
	}

	private void logShape(Shape shape)
	{
		System.out.println("SHAPE: ");
		System.out.println("id: " + shape.getID());
		System.out.println("name: " + shape.getName());

		Instruction[] graphics = shape.getGraphics();
		if(graphics != null) {
			for(int i = 0;i < graphics.length;i++) {
				Instruction inst = graphics[i];
				System.out.println(inst.toString());
			}
		}
		System.out.println("");
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Getters & Setters
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//
}

// private static final long serialVersionUID = 4748880101699843087L;
//
// private static final String VELOCITY_PROPERTIES = "velocity.properties";
// private static final String VELOCITY_ENCODING = "utf-8";
//
// private StringWriter writer;
// private VelocityContext context;
// private VelocityEngine velocityEngine;

// private void initVelocity()
// {
// Velocity.init();
//
// context = new VelocityContext();
// context.put("name", "mofo!!");
// writer = new StringWriter();
//
// try {
// // Load the velocity properties from the class path
// Properties properties = new Properties();
// properties.load(getClass().getClassLoader().getResourceAsStream(VELOCITY_PROPERTIES));
//
// // Create and initialize the template engine
// velocityEngine = new VelocityEngine(properties);
// velocityEngine.mergeTemplate("templates/shape.vm", VELOCITY_ENCODING, context, writer);
//
// } catch (Exception e) {
// e.printStackTrace();
// }
// }
//
// public String getContent()
// {
// return writer.toString();
// }
