
package com.actionjava.mvn.plugins.assets.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import com.actionjava.mvn.plugins.assets.data.Instance;
import com.actionjava.mvn.plugins.assets.data.Symbol;
import com.actionjava.mvn.plugins.assets.vo.Bounds;

public class SymbolParser extends InstanceParser
{
	private static final String REGEX_SYMBOL_BEGIN = "^\\(lib\\.(.*)(\\s*)=(\\s*)function\\(\\)(\\s*)\\{(.*)$";
	private static final String REGEX_SYMBOL_END_1 = "this\\.addChild\\((.*);$";
	private static final String REGEX_SYMBOL_END_2 = "^\\}\\)\\.prototype(\\s*)=(\\s*)p(.*);$";
	private static final String REGEX_SYMBOL_END_3 = "^p\\.nominalBounds(.*);$";

	private static final String REGEX_INSTANCE_BEGIN = "this.([a-zA-Z0-9_]+)\\s=\\snew";
	private static final String REGEX_INSTANCE_END = "this.([a-zA-Z0-9_]+).setTransform";

	private static final String REGEX_SYMBOL_NAME = "^\\(lib\\.([a-zA-Z0-9_]+)\\s*=\\s*function\\(\\)\\s*\\{.*$";
	private static final String REGEX_SYMBOL_TYPE = "^\\}\\)\\.prototype\\s*=\\s*p\\s*=\\s*new\\s*cjs\\.(\\w+)\\(\\);$";
	// private static final String REGEX_SYMBOL_BOUNDS = "^p\\.nominalBounds(.*);$";
	private static final String REGEX_SYMBOL_BOUNDS = "^p\\.nominalBounds\\s*=\\s*new cjs\\.Rectangle\\((.*)\\);$";
	private static final String REGEX_BITMAP_BODY = "^\\s*this.initialize\\(img\\.(.*)\\);$";

	private String symbolPackage;
	private ManifestParser manifestParser;
	private Map<String, Symbol> symbols = new HashMap<String, Symbol>();

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Constructor & initialize
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public SymbolParser(ManifestParser manifestParser, String symbolPackage)
	{
		super();
		initialize(manifestParser, symbolPackage);
	}

	private void initialize(ManifestParser manifestParser, String symbolPackage)
	{
		this.manifestParser = manifestParser;
		this.symbolPackage = symbolPackage;
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Public functionality
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	public Symbol[] parse(String[] symbolsRaw)
	{
		int beginSymbolIndex = 0;
		int endSymbolIndex = 0;
		List<String[]> symbolsScraped = new ArrayList<String[]>();
		while (true) {
			beginSymbolIndex = getLineNumberByRegExFromIndex(symbolsRaw, REGEX_SYMBOL_BEGIN, endSymbolIndex);
			if(beginSymbolIndex != -1) {
				endSymbolIndex = getLineNumberByRegExFromIndex(symbolsRaw, REGEX_SYMBOL_END_3, beginSymbolIndex) + 1;
				if(endSymbolIndex != -1) {
					symbolsScraped.add(copyRangeFromArray(symbolsRaw, beginSymbolIndex, endSymbolIndex));
					continue;
				}
			}
			break;
		}

		createReferenceList(symbolsScraped);

		Iterator<Entry<String, Symbol>> it = symbols.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry<String, Symbol> pair = it.next();
			parseSymbol(pair.getValue());
		}

		return symbols.values().toArray(new Symbol[symbols.size()]);
	}

	@Override
	public Symbol getSymbolByName(String name)
	{
		return symbols.get(name);
	}

	//
	//
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	// - - Private functionality
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
	//
	//

	private void parseSymbol(Symbol s)
	{
		if(s.getType().equals(Symbol.SPRITE_TYPE)) {
			parseSymbolBody(s, s.getRawData());
		} else if(s.getType().equals(Symbol.BITMAP_TYPE)) {
			parseBitmapBody(s, s.getRawData());
		}
	}

	private void createReferenceList(List<String[]> symbolsRaw)
	{
		Symbol symbol;
		String[] symbolRaw;
		for(int i = 0;i < symbolsRaw.size();i++) {
			symbolRaw = symbolsRaw.get(i);
			symbol = createSymbolReference(symbolRaw);
			symbols.put(symbol.getName(), symbol);
		}
	}

	private Symbol createSymbolReference(String[] symbolRaw)
	{
		int symbolNameRaw = getLineNumberByRegEx(symbolRaw, REGEX_SYMBOL_NAME);
		int symbolTypeRaw = getLineNumberByRegEx(symbolRaw, REGEX_SYMBOL_TYPE);
		int symbolBoundsRaw = getLineNumberByRegEx(symbolRaw, REGEX_SYMBOL_BOUNDS);

		String symbolName = getSymbolName(symbolRaw[symbolNameRaw]);
		String symbolType = getSymbolType(symbolRaw[symbolTypeRaw]);
		Bounds symbolBounds = getSymbolBounds(symbolRaw[symbolBoundsRaw]);
		String[] symbolBody = getSymbolBody(symbolRaw);

		Symbol s = new Symbol(symbolName, symbolName, symbolPackage);
		s.setType(symbolType);
		s.setBounds(symbolBounds);
		s.setRawData(symbolBody);

		return s;
	}

	private String getSymbolName(String rawLine)
	{
		Pattern pattern = Pattern.compile(REGEX_SYMBOL_NAME);
		Matcher matcher = pattern.matcher(rawLine);

		if(matcher.find()) { return matcher.group(1); }

		return null;
	}

	private String getSymbolType(String rawLine)
	{
		Pattern pattern = Pattern.compile(REGEX_SYMBOL_TYPE);
		Matcher matcher = pattern.matcher(rawLine);

		if(matcher.find()) {
			String rawType = matcher.group(1);

			if(rawType.equals("Container")) {
				return Symbol.SPRITE_TYPE;
			} else if(rawType.equals("Bitmap")) {
				return Symbol.BITMAP_TYPE;
			} else if(rawType.equals("MovieClip")) { return Symbol.MC_TYPE; }
		}

		return Symbol.UNKNOWN_TYPE;
	}

	private Bounds getSymbolBounds(String rawLine)
	{
		Pattern pattern = Pattern.compile(REGEX_SYMBOL_BOUNDS);
		Matcher matcher = pattern.matcher(rawLine);

		if(matcher.find()) {
			String rawBounds = matcher.group(1);
			String[] splitRawBounds = Pattern.compile(",").split(rawBounds);
			if(splitRawBounds.length == 4) {
				Bounds b = new Bounds(Double.valueOf(splitRawBounds[0]), Double.valueOf(splitRawBounds[1]), Double.valueOf(splitRawBounds[2]), Double.valueOf(splitRawBounds[3]));
				return b;
			}
		}

		return null;
	}

	private String[] getSymbolBody(String[] rawLines)
	{
		List<String> newRawLines = new ArrayList<String>();
		for(int i = 0;i < rawLines.length;i++) {
			if((Pattern.matches("^(\\s*)this\\.(.*);$", rawLines[i])) && (!Pattern.matches("^(\\s*)this\\.addChild(.*);$", rawLines[i]))) {
				newRawLines.add(rawLines[i]);
			}
		}

		return newRawLines.toArray(new String[newRawLines.size()]);
	}

	// private Symbol[] parseSymbols(List<String[]> symbolsRaw)
	// {
	// List<Symbol> symbols = new ArrayList<Symbol>();
	// for(int i = 0;i < symbolsRaw.size();i++) {
	// symbols.add(parseSymbol(symbolsRaw.get(i)));
	// }
	//
	// return symbols.toArray(new Symbol[symbols.size()]);
	// }
	//
	// private Symbol parseSymbol(String[] symbolRaw)
	// {
	// List<String> symbolRawList = new LinkedList<String>(Arrays.asList(symbolRaw));
	//
	// String symbolNameRaw = symbolRawList.remove(0);
	// String symbolBoundsRaw = symbolRawList.remove(symbolRawList.size() - 1);
	// symbolRawList.remove(symbolRawList.size() - 1);
	// symbolRawList.remove(symbolRawList.size() - 1);
	//
	// Symbol symbol = createSymbol(symbolNameRaw);
	// parseSymbolBounds(symbol, symbolBoundsRaw);
	// parseSymbolBody(symbol, symbolRawList.toArray(new String[symbolRawList.size()]));
	//
	// return symbol;
	// }
	//
	// private Symbol createSymbol(String symbolNameRaw)
	// {
	// symbolNameRaw = symbolNameRaw.replaceAll("\\s+", "");
	// int start = 5;
	// int end = symbolNameRaw.indexOf("=");
	//
	// String name = symbolNameRaw.substring(start, end);
	// Symbol symbol = new Symbol(name, name, symbolPackage);
	//
	// return symbol;
	// }
	//
	// private void parseSymbolBounds(Symbol symbol, String symbolBoundsRaw)
	// {
	// symbolBoundsRaw = symbolBoundsRaw.replaceAll("\\s+", "");
	// int start = symbolBoundsRaw.indexOf("(") + 1;
	// int end = symbolBoundsRaw.lastIndexOf(")");
	//
	// String boundParamsRaw = symbolBoundsRaw.substring(start, end);
	// Parameter[] boundParams = parseParams(boundParamsRaw);
	// if(boundParams.length == 4) {
	// symbol.setBounds(boundParams[0].getNumberValue(), boundParams[1].getNumberValue(), boundParams[2].getNumberValue(), boundParams[3].getNumberValue());
	// } else {
	// symbol.setBounds(0, 0, 0, 0);
	// }
	// }
	//
	private void parseSymbolBody(Symbol symbol, String[] symbolBodyRaw)
	{
		int startLine = 0;
		int endLine = 0;
		boolean lookForBegin = true;

		Pattern pattern;
		Matcher matcher;

		String symbolBodyLineRaw;
		List<String> instanceListRaw = new ArrayList<String>();

		for(int i = 0;i < symbolBodyRaw.length;i++) {
			symbolBodyLineRaw = symbolBodyRaw[i];

			if(lookForBegin) {
				pattern = Pattern.compile(REGEX_INSTANCE_BEGIN);
				matcher = pattern.matcher(symbolBodyLineRaw);

				if(matcher.find()) {
					startLine = i;
					lookForBegin = false;

				}
			} else {
				pattern = Pattern.compile(REGEX_INSTANCE_END);
				matcher = pattern.matcher(symbolBodyLineRaw);

				if(matcher.find()) {
					endLine = i + 1;

					for(int j = startLine;j < endLine;j++) {
						String symbolBodyLine = symbolBodyRaw[j];
						instanceListRaw.add(symbolBodyLine);
					}

					startLine = 0;
					endLine = 0;
					lookForBegin = true;

					Instance inst = parseInstance(instanceListRaw.toArray(new String[instanceListRaw.size()]));
					instanceListRaw.clear();

					if(inst != null) {
						symbol.addChild(inst);
					}
				}
			}
		}
	}

	private void parseBitmapBody(Symbol symbol, String[] symbolBodyRaw)
	{
		for(int i = 0;i < symbolBodyRaw.length;i++) {
			Pattern p = Pattern.compile(REGEX_BITMAP_BODY);
			Matcher m = p.matcher(symbolBodyRaw[i]);
			if(m.find()) {
				String group1 = m.group(1);
				symbol.setBitmapUrl(manifestParser.getHrefByID(group1));
			}

		}
	}
}
