/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
 */

package com.ats.element.test;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;

import org.openqa.selenium.NoAlertPresentException;

import com.ats.element.DialogBox;
import com.ats.element.FoundElement;
import com.ats.element.SearchedElement;
import com.ats.executor.ActionStatus;
import com.ats.executor.ActionTestScript;
import com.ats.executor.channels.Channel;
import com.ats.generator.objects.MouseDirection;
import com.ats.generator.variables.CalculatedProperty;
import com.ats.generator.variables.CalculatedValue;
import com.ats.recorder.IVisualRecorder;

public class TestElementDialog extends TestElement {

	private static final String ID_CRITERIA = "id";
	private static final String TEXT_CRITERIA = "text";
	private static final String TYPE_CRITERIA = "type";
	private static final String INDEX_CRITERIA = "index";

	private static final String TITLE = "title";

	public static final String DIALOG = "dialog";
	public static final String ACCEPT = "accept";
	public static final String DISMISS = "dismiss";
	
	public static final String DIALOG_NOT_FOUND = "Dialog box not found";

	private DialogBox alert = null;
	private String alertAction = null;
	private String alertCriteria = null;

	public TestElementDialog() {}

	public TestElementDialog(ActionTestScript script, Channel channel) {
		super(script, channel);
	}

	public TestElementDialog(ActionTestScript script, Channel channel, int maxTry, SearchedElement searchElement, Predicate<Integer> predicate) {
		super(script, channel, maxTry, predicate);
		initSearch(searchElement.getCriterias());
	}

	public TestElementDialog(ActionTestScript script, Channel channel, int maxTry, List<CalculatedProperty> criterias) {
		super(script, channel, maxTry);
		initSearch(criterias);
	}

	public TestElementDialog(ActionTestScript script, FoundElement element, Channel currentChannel) {
		super(script, element, currentChannel);
	}

	public String getAlertAction() {
		return alertAction;
	}

	private void initSearch(List<CalculatedProperty> properties) {

		if(properties.size() > 0) {
			alertAction = properties.get(0).getValue().getDataListItem();
			alertCriteria = properties.get(0).getName();
		}

		try {

			alert = getChannel().switchToAlert();

			setFoundElements(new ArrayList<>(Arrays.asList(new FoundElement())));
			setCount(1);

		}catch(NoAlertPresentException ex) {
			getChannel().sleep(500);
			setCount(0);
		}
	}

	@Override
	public void updateScreen() {
		recorder.updateScreen(true);
	}

	@Override
	public FoundElement getFoundElement() {
		return null;
	}

	@Override
	public void over(ActionStatus status, MouseDirection position, boolean desktopDragDrop, int offsetX, int offsetY) {
	}

	@Override
	public String enterText(ActionStatus status, CalculatedValue text, ActionTestScript script, int waitChar) {
		sendText(script, status, text, waitChar);
		return text.getCalculated();
	}

	@Override
	public void clearText(ActionStatus status, MouseDirection md) {
	}

	@Override
	public String sendText(ActionTestScript script, ActionStatus status, CalculatedValue text, int waitChar) {
		final String enterText = text.getCalculated();
		getChannel().sleep(alert.getWaitBox());

		if (alertCriteria != null) {
			switch (alertCriteria) {
			case INDEX_CRITERIA:
				try {
					alert.sendKeys(enterText, Integer.parseInt(alertAction));
				} catch (NumberFormatException ignored) { }
				break;
			case ID_CRITERIA:
				alert.sendKeys(enterText, alertAction);
				break;
			case TEXT_CRITERIA:
			case TYPE_CRITERIA:
				alert.sendKeys(enterText);
				break;
			}
		} else {
			alert.sendKeys(enterText);
		}

		if (ACCEPT.equals(alertAction)) {
			alert.accept(status);
		} else if(DISMISS.equals(alertAction)) {
			alert.dismiss(status);
		}

		return enterText;
	}

	@Override
	public String getAttribute(ActionStatus status, String name) {

		if(alert == null) {
			return "";
		}

		getChannel().sleep(alert.getWaitBox());
		if(TITLE.equals(name.toLowerCase())) {
			return alert.getTitle();
		}
		return alert.getText();
	}

	@Override
	public void click(ActionStatus status, MouseDirection position) {
		getChannel().sleep(alert.getWaitBox());
		status.setPassed(true);

		if (alertCriteria != null) {
			switch (alertCriteria) {
				case INDEX_CRITERIA:
					try {
						alert.clickButtonAtIndex(Integer.parseInt(alertAction), status);
					} catch (NumberFormatException ignored) { }
					break;
				case ID_CRITERIA:
					alert.clickButtonId(alertAction, status);
					break;
				case TEXT_CRITERIA:
					alert.clickButtonText(alertAction, status);
					break;
				case TYPE_CRITERIA:
					if(DISMISS.equals(alertAction)) {
						alert.dismiss(status);
					} else if(ACCEPT.equals(alertAction)) {
						alert.accept(status);
					} else {
						alert.defaultButton(status);
					}
					break;
			}
		} else {
			alert.defaultButton(status);
		}

		getChannel().sleep(alert.getWaitBox());
		getChannel().switchToDefaultContent(true);
	}
		
	@Override
	public CalculatedProperty[] getHtmlAttributes() {
		return getAttributes(true);
	}

	@Override
	public CalculatedProperty[] getAttributes(boolean reload) {
		if(alert == null) {
			return new CalculatedProperty[0];
		}
		return alert.getAttributes();
	}

	@Override
	public void updateScreen(ActionStatus status, IVisualRecorder rec) {
		rec.updateScreen(status, getAlertAction());
	}

	@Override
	public String getNotFoundDescription() {
		return DIALOG_NOT_FOUND;
	}
	
	
}