/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
 */

package com.ats.script.actions;

import java.util.ArrayList;
import java.util.function.Predicate;

import com.ats.element.SearchedElement;
import com.ats.executor.ActionTestScript;
import com.ats.generator.variables.CalculatedValue;
import com.ats.generator.variables.Variable;
import com.ats.script.Script;
import com.ats.script.actions.condition.ExecuteOptions;
import com.ats.script.AtsScript;
import com.google.gson.JsonObject;

public class ActionScripting extends ActionReturnVariable {

	public static final String JAVASCRIPT_LABEL = "javascript";
	public static final String SCRIPT_LABEL = "scripting";
	
	public static final Predicate<String> PREDICATE = g -> SCRIPT_LABEL.equals(g) || JAVASCRIPT_LABEL.equals(g);

	private CalculatedValue jsCode;

	public ActionScripting() {}

	public ActionScripting(AtsScript script, ExecuteOptions options, int stopPolicy, String code, Variable variable, ArrayList<String> objectArray) {
		super(script, options, stopPolicy, objectArray, variable);
		setJsCode(new CalculatedValue(script, code));
	}

	public ActionScripting(Script script, ExecuteOptions options, int stopPolicy, int maxTry, int delay, SearchedElement element, CalculatedValue jsCode, Variable variable) {
		super(script, options, stopPolicy, maxTry, delay, element, variable);
		setJsCode(jsCode);
	}

	//---------------------------------------------------------------------------------------------------------------------------------
	// Code Generator
	//---------------------------------------------------------------------------------------------------------------------------------

	@Override
	public StringBuilder getJavaCode() {
		String variableCode = "null";
		if(getVariable() != null) {
			variableCode = getVariable().getName();
		}

		StringBuilder codeBuilder = super.getJavaCode();
		codeBuilder.append(", ").append(jsCode.getJavaCode()).append(", ").append(variableCode).append(")");
		return codeBuilder;
	}

	@Override
	public ArrayList<String> getKeywords() {
		ArrayList<String> keywords = super.getKeywords();
		keywords.add(jsCode.getKeywords());
		return keywords;
	}

	//---------------------------------------------------------------------------------------------------------------------------------
	//---------------------------------------------------------------------------------------------------------------------------------

	@Override
	public void terminateExecution(ActionTestScript ts) {

		super.terminateExecution(ts);

		if(variable != null) {
			if(status.isPassed()) {
				final Object result = getTestElement().executeScript(status, jsCode.getCalculated(), true);
				if(result != null) {
					updateVariableValue(result.toString());
				}
			}else {
				status.setMessage("");
				updateVariableValue("");
			}
		}else {
			getTestElement().executeScript(status, jsCode.getCalculated(), false);
		}

		status.endAction();
		ts.getRecorder().updateScreen(status, jsCode.getCalculated());
	}

	@Override
	public StringBuilder getActionLogs(String scriptName, int scriptLine, JsonObject data) {
		data.addProperty("code", jsCode.getCalculated());
		return super.getActionLogs(scriptName, scriptLine, data);
	}

	//--------------------------------------------------------
	// getters and setters for serialization
	//--------------------------------------------------------

	public CalculatedValue getJsCode() {
		return jsCode;
	}

	public void setJsCode(CalculatedValue jsCode) {
		this.jsCode = jsCode;
	}
}