package com.ats.tools.report;

import com.ats.recorder.TestError;
import com.ats.script.actions.ActionApi;
import com.ats.script.actions.ActionCallscript;
import com.ats.script.actions.ActionComment;
import com.ats.tools.ResourceContent;
import com.ats.tools.logger.levels.AtsLogger;
import com.ats.tools.report.actions.HtmlReportAction;
import com.ats.tools.report.general.HtmlReport;
import com.ats.tools.report.general.HtmlReportExecution;
import com.ats.tools.report.general.HtmlReportPlaylist;
import com.ats.tools.report.general.HtmlReportTestInfo;
import com.ats.tools.report.models.*;
import com.ats.tools.report.utils.AtsReportXmlGenerator;
import com.ats.tools.report.utils.FileUtils;
import com.ats.tools.report.utils.JasperSummaryUtils;
import com.ctc.wstx.api.WstxInputProperties;
import com.ctc.wstx.stax.WstxInputFactory;
import com.fasterxml.jackson.databind.ObjectMapper;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.stream.Collectors;

public class HtmlCampaignReportGenerator {
    public static final String ATS_TEST_REPORT_XML_NAME = "ats-report.xml";
    public static final String ATS_TEST_REPORT_HTML_DATA_FILE_NAME = "actions.xml";
    public static final String INCONSISTENCY_WARNING = "<div class='inconsistency-warning'><div class='warning-icon'></div><div>Inconsistency in execution results has been found! Please contact support.</div></div>";
    public static final String ATS_RESULTS_JSON_FILE_NAME = "ats-results.json";
    public static final String FAIL_STATUS = "FAIL";
    public static final String PASS_STATUS = "PASS";
    private final Path outputFolderPath;
    private final File jsonSuiteFilesFile;
    private final int devReportLevel;
    private final int validReportLevel;

    private final SuitesReportItem suiteItem;

    private final boolean isValidationReport;
    private static final String styles = ResourceContent.getHtmlReportCss();
    private static final String javascript = ResourceContent.getHtmlReportJavascript();
    private static final String actionTemplate = ResourceContent.getHtmlReportActionHtmlTemplate();
    private static final String mainTemplate = ResourceContent.getHtmlReportTemplate();
    private static final String playlistTemplate = ResourceContent.getHtmlReportPlaylistTemplate();
    private static final String reportSuiteTemplate = ResourceContent.getHtmlReportSuiteTemplate();
    private static final String mainExecutionTemplate = ResourceContent.getHtmlReportExecutionTemplate();
    private AtsReportXmlGenerator atsReportXmlGenerator;
    private Map<String, Integer> testSuiteActionsCount = new HashMap<>();
    private HtmlReportPlaylist htmlReportPlaylist;
    private ObjectMapper objectMapper = new ObjectMapper();
    private List<Summary> testSummariesForExecutions;

    public static class InputFactory extends WstxInputFactory {
        public InputFactory() {
            super();
            setProperty(WstxInputProperties.P_MAX_ATTRIBUTE_SIZE, Integer.MAX_VALUE);
        }
    }

    public HtmlCampaignReportGenerator(Path outputFolderPath, File jsonSuiteFilesFile, int devReportLevel, int validReport, SuitesReportItem suiteItem) {
        this.outputFolderPath = outputFolderPath;
        this.jsonSuiteFilesFile = jsonSuiteFilesFile;
        this.devReportLevel = devReportLevel;
        this.validReportLevel = validReport;
        this.isValidationReport = validReport == 1;
        this.suiteItem = suiteItem;
    }

    public void generateSuitsHtmlReports() throws IOException, XMLStreamException {

        long startTime = System.nanoTime();

        List<Script> scripts = FileUtils.getScripts(outputFolderPath.toString());
        HtmlReportProject project = objectMapper.readValue(Files.readString(jsonSuiteFilesFile.toPath()), HtmlReportProject.class);
        Results results = objectMapper.readValue(outputFolderPath.resolve(ATS_RESULTS_JSON_FILE_NAME).toFile(), Results.class);

        atsReportXmlGenerator = new AtsReportXmlGenerator(project, results, scripts, devReportLevel, outputFolderPath, validReportLevel, ATS_TEST_REPORT_XML_NAME);
        atsReportXmlGenerator.writeProjectInfo();
        testSummariesForExecutions = atsReportXmlGenerator.getTestSummariesForExecutionsResult();

        HtmlReportExecution htmlReportExecution = new HtmlReportExecution(mainExecutionTemplate, project, outputFolderPath.toString(), devReportLevel, isValidationReport, suiteItem);
        htmlReportExecution.processMainExecutionsFile();

        project.getSuites().stream().forEach(suite -> {
            SuiteInfo suiteInfo = results.getSuites().stream().sorted((o1, o2) -> Math.toIntExact(o2.getStarted() - o1.getStarted())).filter(s -> s.getName().equals(suite.getName())).findFirst().orElse(null);
            if (suiteInfo != null) {
                suite.setSuiteInfo(suiteInfo);
            }
        });

        project.getSuites().forEach(suite -> generateHtmlReports(suite, htmlReportExecution.getFileWriter(), htmlReportExecution.getValidationFileWriter(), project, scripts));

        htmlReportExecution.closeWriter();
        atsReportXmlGenerator.writeFileEnd();

        if (!suiteItem.isNoSuiteLaunch() && FileUtils.isFileSizeLessThan(Path.of(outputFolderPath + File.separator + ATS_TEST_REPORT_XML_NAME).toFile(), 500)) {
            JasperSummaryUtils.generateSummaryPdf(outputFolderPath, ATS_TEST_REPORT_XML_NAME);
        }


        AtsLogger.printLog(
                new StringBuilder("report generated in ").
                        append((System.nanoTime() - startTime) / 1000000000).
                        append(" second(s)")
        );
    }

    private void generateHtmlReports(Suite suite, OutputStream fileWriter, OutputStream validationFileWriter, HtmlReportProject project, List<Script> scripts) {
        List<Script> suiteScripts = scripts.stream().filter(script -> script.getSuite().equals(suite.getName())).collect(Collectors.toList());

        String scriptsStatus = scripts.stream()
                .filter(script -> script.getSuite().equals(suite.getName()))
                .anyMatch(script -> !script.isPassed()) ? FAIL_STATUS : PASS_STATUS;

        htmlReportPlaylist = new HtmlReportPlaylist(playlistTemplate, suite,
                fileWriter, validationFileWriter, project, suiteScripts,
                isValidationReport, suiteItem.isNoSuiteLaunch(), scriptsStatus);
        htmlReportPlaylist.processPlaylistData();

        atsReportXmlGenerator.writeSuiteData(suite, scripts);

        suite.getTests().forEach(test -> generateHtmlReport(suite.getName(), test));

        atsReportXmlGenerator.writeSuiteEnd();
        htmlReportPlaylist.processPlaylistFooter();
    }

    private void generateHtmlReport(String suiteName, String testName) {
        final Path targetPath = outputFolderPath.resolve(suiteName).resolve(testName + "_xml");

        if (!Files.exists(targetPath)) {//test has been filtered
            return;
        }

        Map<String, String> appIcons = new HashMap<>();

        boolean creatingObject = false;
        Action action = new Action();
        AppDataJson appDataJson = new AppDataJson();
        ActionElement actionElement = new ActionElement();
        TestInfo testInfo = new TestInfo();
        testInfo.setTestName(testName);
        try {
            System.getProperties().put("javax.xml.stream.XMLInputFactory", InputFactory.class.getName());

            Optional<Summary> first = testSummariesForExecutions.stream().filter(summary -> summary.getTestName().equals(testName)).findFirst();

            XMLInputFactory factory = XMLInputFactory.newInstance();
            InputStream inputStream = new FileInputStream(targetPath.resolve(XmlReport.REPORT_DATA_FILE).toFile());
            if (first.isPresent()) {
                atsReportXmlGenerator.writeScript(targetPath.resolve(ATS_TEST_REPORT_HTML_DATA_FILE_NAME).toFile(), devReportLevel);
            }
            XMLStreamReader reader = factory.createXMLStreamReader(inputStream);
            OutputStream outputStream = new FileOutputStream(targetPath.resolve(CampaignReportGenerator.ATS_TEST_REPORT_HTML).toFile());

            HtmlReport htmlReport = createReportHeader();
            HtmlReportAction htmlReportAction = new HtmlReportAction();
            boolean isTestInfo = false;

            while (reader.hasNext()) {
                int event = reader.next();
                if (event == XMLStreamConstants.START_DOCUMENT) {
                    continue;
                }

                if (event == XMLStreamConstants.START_ELEMENT) {
                    if (reader.getLocalName().equals("action")) {
                        isTestInfo = false;
                        action.setIndex(Integer.parseInt(reader.getAttributeValue(0)));
                        String type = reader.getAttributeValue(1).split("\\.")[reader.getAttributeValue(1).split("\\.").length - 1];
                        action.setType(type);
                        creatingObject = true;
                    } else if (!creatingObject && reader.getLocalName().equals("script")) {
                        creatingObject = true;
                        isTestInfo = true;
                    }
                    if (creatingObject) {
                        if (isTestInfo) {
                            switch (reader.getLocalName()) {
                                case "script": {
                                    testInfo.setExternalId(reader.getAttributeValue("", "externalId"));
                                    testInfo.setTestId(reader.getAttributeValue("", "testId"));
                                    break;
                                }
                                case "description": {
                                    testInfo.setDescription(reader.getElementText());
                                    break;
                                }
                                case "author": {
                                    testInfo.setAuthor(reader.getElementText());
                                    break;
                                }
                                case "prerequisite": {
                                    testInfo.setPrerequisite(reader.getElementText());
                                    break;
                                }
                                case "startedFormated": {
                                    testInfo.setStartedFormatted(reader.getElementText());
                                    break;
                                }
                                case "started": {
                                    testInfo.setStarted(reader.getElementText());
                                    break;
                                }
                                case "quality": {
                                    testInfo.setQuality(reader.getElementText());
                                    break;
                                }
                                case "group": {
                                    testInfo.getGroups().add(reader.getElementText());
                                    break;
                                }
                                case "project": {
                                    HtmlReportProject project = new HtmlReportProject();
                                    testInfo.setProject(project);
                                    break;
                                }
                                case "id": {
                                    testInfo.getProject().setProjectId(reader.getElementText());
                                    break;
                                }
                                case "name": {
                                    testInfo.getProject().setProjectName(reader.getElementText());
                                    break;
                                }
                                case "summary": {
                                    Summary summary = new Summary(reader);
                                    testInfo.setSummary(summary);
                                    break;
                                }
                                case "errors": {
                                    testInfo.getSummary().setTestErrors(new ArrayList<>());
                                    break;
                                }
                                case "error": {
                                    TestError error = new TestError();
                                    error.setLine(Integer.parseInt(reader.getAttributeValue("", "line")));
                                    error.setScript(reader.getAttributeValue("", "script"));
                                    error.setTestErrorStatus(TestError.TestErrorStatus.valueOf(reader.getAttributeValue("", "testErrorStatus")));
                                    error.setMessage(reader.getElementText());
                                    testInfo.getSummary().getTestErrors().add(error);
                                    break;
                                }
                                case "data": {
                                    testInfo.getSummary().setData(reader.getElementText());
                                    break;
                                }
                                case "actionsType": {
                                    testInfo.setActionTypeChart(reader.getElementText());
                                }
                            }
                        }
                        if (!isTestInfo) {
                            switch (reader.getLocalName()) {
                                case "line": {
                                    action.setLine(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                                case "script": {
                                    action.setScript(reader.getElementText());
                                    break;
                                }
                                case "timeLine": {
                                    action.setTimeLine(Long.parseLong(reader.getElementText()));
                                    break;
                                }
                                case "error": {
                                    action.setError(reader.getElementText());
                                    break;
                                }
                                case "stop": {
                                    action.setStop(Boolean.parseBoolean(reader.getElementText()));
                                    break;
                                }
                                case "duration": {
                                    action.setDuration(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                                case "passed": {
                                    action.setPassed(Boolean.parseBoolean(reader.getElementText()));
                                    break;
                                }
                                case "value": {
                                    action.setValue(reader.getElementText());
                                    break;
                                }
                                case "data": {
                                    if (isTestInfo) {
                                        testInfo.getSummary().setData(reader.getElementText());
                                    } else {
                                        action.setData(reader.getElementText());
                                    }
                                    break;
                                }
                                case "parameter": {

                                    if (ActionApi.class.getSimpleName().equals(action.getType()) || ActionCallscript.class.getSimpleName().equals(action.getType())) {
                                        switch (reader.getAttributeCount()) {
                                            case 2: {
                                                appDataJson.getData().put(reader.getAttributeValue(0), reader.getAttributeValue(1));
                                                break;
                                            }
                                            case 3: {
                                                appDataJson.getData().put(reader.getAttributeValue(0), reader.getAttributeValue(2));
                                                break;
                                            }
                                            default: {
                                                appDataJson.getData().put(reader.getAttributeValue(0), "Ø");

                                            }
                                        }
                                    } else {
                                        switch (reader.getAttributeValue(0)) {
                                            case "name": {
                                                appDataJson.setName(reader.getAttributeValue(1));
                                                action.setAppDataJson(appDataJson);
                                                break;
                                            }
                                            case "app": {
                                                appDataJson.setApp(reader.getAttributeValue(1));
                                                break;
                                            }
                                            case "appVersion": {
                                                appDataJson.setAppVersion(reader.getAttributeValue(1));
                                                break;
                                            }
                                            case "os": {
                                                appDataJson.setOs(reader.getAttributeValue(1));
                                                break;
                                            }
                                            case "appIcon": {
                                                appDataJson.setAppIcon(reader.getAttributeValue(1));
                                                appIcons.put(appDataJson.getApp(), appDataJson.getAppIcon());
                                                break;
                                            }
                                        }
                                    }
                                    break;
                                }
                                case "img": {
                                    if (reader.getAttributeCount() > 0) {
                                        Image image = new Image(reader);
                                        if (image.getType().equals("api")) {
                                            image.useApiIcon();
                                        }
                                        action.setImage(image);
                                    }
                                    break;
                                }
                                case "channel": {
                                    Channel channel = new Channel();
                                    channel.setName(reader.getAttributeValue(0));
                                    action.setChannel(channel);
                                    break;
                                }
                                case "bound": {
                                    Bound bound = new Bound();
                                    action.getChannel().setBound(bound);
                                    break;
                                }
                                case "x": {
                                    action.getChannel().getBound().setX(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                                case "y": {
                                    action.getChannel().getBound().setY(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                                case "width": {
                                    action.getChannel().getBound().setWidth(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                                case "height": {
                                    action.getChannel().getBound().setHeight(Integer.parseInt(reader.getElementText()));
                                    break;
                                }

                                case "element": {
                                    actionElement.setTag(reader.getAttributeValue(0));
                                    break;
                                }

                                case "criterias": {
                                    actionElement.setCriterias(reader.getElementText());
                                    break;
                                }

                                case "foundElements": {
                                    actionElement.setFoundElements(Integer.parseInt(reader.getElementText()));
                                    break;
                                }

                                case "searchDuration": {
                                    actionElement.setSearchDuration(Integer.parseInt(reader.getElementText()));
                                    break;
                                }
                            }
                        }
                    }
                }

                if (event == XMLStreamConstants.END_ELEMENT && reader.getLocalName().equals("script") && isTestInfo) {
                    htmlReport.setMainReport(new HtmlReportTestInfo(testInfo, htmlReport.getMainReport(), null, suiteName, suiteItem).getResult());
                    testSuiteActionsCount.put(testInfo.getTestName() + "|" + testInfo.getSummary().getSuiteName(), Integer.valueOf(testInfo.getSummary().getActions()));
                    outputStream.write(htmlReport.getMainReport().getBytes());
                    htmlReportPlaylist.processPlaylistTestInfo(new HtmlReportTestInfo(testInfo, reportSuiteTemplate, targetPath.resolve(CampaignReportGenerator.ATS_TEST_REPORT_HTML).toString(), suiteName, suiteItem).getResult());
                }

                if (event == XMLStreamConstants.END_ELEMENT && com.ats.script.actions.Action.DATA_JSON.equals(reader.getLocalName())) {
                    action.setAppDataJson(appDataJson);
                    appDataJson = new AppDataJson();
                }

                if (event == XMLStreamConstants.END_ELEMENT && reader.getLocalName().equals("element")) {
                    action.setActionElement(actionElement);
                    actionElement = new ActionElement();
                }

                if (event == XMLStreamConstants.END_ELEMENT && reader.getLocalName().equals("action")) {
                    String processedAction = htmlReportAction.processAction(action, appIcons, testInfo, devReportLevel);
                    outputStream.write(processedAction.getBytes());
                    htmlReportPlaylist.processPlaylistTestAction(processedAction, isValidationReport, (!ActionComment.class.getSimpleName().equals(action.getType()) && !ActionCallscript.class.getSimpleName().equals(action.getType())));
                    htmlReportAction.setActionTemplate(actionTemplate);
                    action = new Action();
                }
            }
            outputStream.write(htmlReport.getFooter().getBytes());
            htmlReportPlaylist.processTestCaseFooter();

            reader.close();
            inputStream.close();
            outputStream.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private HtmlReport createReportHeader() {
        return new HtmlReport(mainTemplate, styles, javascript);
    }
}