package com.adgem.android;

import android.content.Context;

import androidx.annotation.IntDef;
import androidx.annotation.MainThread;

import com.adgem.android.internal.RealGem;

import java.lang.annotation.Retention;

import static java.lang.annotation.RetentionPolicy.SOURCE;

/**
 * Main entry point for communications with AdGem SDK. Use {@link #get()} to acquire instance of this class.
 */
public abstract class AdGem {
    @Retention(SOURCE)
    @IntDef({STATE_ERROR, STATE_DISABLED, STATE_NEEDS_INITIALIZATION, STATE_INITIALIZING, STATE_NEEDS_CAMPAIGN_REFRESH,
            STATE_REFRESHING_CAMPAIGN, STATE_NEEDS_DOWNLOAD, STATE_DOWNLOADING, STATE_READY})
    public @interface State {
    }

    /**
     * Identifies that an error occurred. AdGem will attempt to recover from error automatically. Last known error is
     * available via {@link #getError()}.
     */
    public static final int STATE_ERROR = -2;

    /**
     * Identifies that the component owning this state was disabled by the SDK configuration or server.
     */
    public static final int STATE_DISABLED = -1;

    /**
     * Identifies that data is in unknown state and needs initialization.
     */
    public static final int STATE_NEEDS_INITIALIZATION = 0;

    /**
     * Identifies that data is in initializing.
     */
    public static final int STATE_INITIALIZING = 1;

    public static final int STATE_NEEDS_CAMPAIGN_REFRESH = 2;

    public static final int STATE_REFRESHING_CAMPAIGN = 3;

    public static final int STATE_NEEDS_DOWNLOAD = 4;

    public static final int STATE_DOWNLOADING = 5;

    public static final int STATE_READY = 6;

    /**
     * Obtains global instance of the AdGem framework.
     */
    public static AdGem get() {
        return RealGem.get();
    }

    /**
     * Identifies whether AdGem is ready to show an interstitial advertisement via {@link #showInterstitialAd(Context)}.
     *
     * @return <code>true</code> if AdGem is ready to show next ad, <code>false</code> otherwise.
     */
    public boolean isInterstitialAdReady() {
        return getInterstitialAdState() == STATE_READY;
    }

    @State
    public abstract int getInterstitialAdState();

    /**
     * Identifies whether AdGem is ready to show a rewarded advertisement via {@link #showRewardedAd(Context)}.
     *
     * @return <code>true</code> if AdGem is ready a rewarded ad, <code>false</code> otherwise.
     */
    public boolean isRewardedAdReady() {
        return getRewardedAdState() == STATE_READY;
    }

    @State
    public abstract int getRewardedAdState();

    /**
     * Identifies whether AdGem is ready to show offer wall via the {@link #showOfferWall(Context)}}.
     *
     * @return <code>true</code> if AdGem is ready to show offer wall, <code>false</code> otherwise.
     * @see #registerOfferWallCallback(OfferWallCallback)
     */
    public abstract boolean isOfferWallReady();

    /**
     * Gets detailed state of the offer wall
     *
     * @return offer wall state as defined in {@link State}
     * @see #registerOfferWallCallback(OfferWallCallback)
     */
    @State
    public abstract int getOfferWallState();

    /**
     * Registers callback to receive AdGem SDK events. Each registered callback will receive corresponding events.
     *
     * @param callback instance of callback interface to receive events from AdGem. All events will be delivered on a main
     *                 thread. Reference to this
     *                 callback will be kept internally until unregistered via {@link #unregisterCallback(AdGemCallback)}.
     * @see #registerOfferWallCallback(OfferWallCallback) to receive offer wall events.
     * @see #unregisterCallback(AdGemCallback)
     */
    @MainThread
    public abstract void registerCallback(AdGemCallback callback);

    /**
     * Unregisters AdGem events callback.
     *
     * @param callback instance of {@link AdGemCallback} previously registered via {@link #registerCallback(AdGemCallback)}.
     * @see #unregisterCallback(AdGemCallback)
     * @see #unregisterOfferWallCallback(OfferWallCallback)
     */
    @MainThread
    public abstract void unregisterCallback(AdGemCallback callback);

    /**
     * Registers callback to receive updates about the offer wall. Each registered callback will receive corresponding events.
     *
     * @param callback instance of callback to receive offer wall events from AdGem. All events will be delivered on a main
     *                 thread. Reference to this callback will be kept internally until unregistered via
     *                 {@link #unregisterOfferWallCallback(OfferWallCallback)}.
     * @see #registerCallback(AdGemCallback) to receive other AdGem events
     * @see #unregisterOfferWallCallback(OfferWallCallback)
     */
    @MainThread
    public abstract void registerOfferWallCallback(OfferWallCallback callback);

    /**
     * Unregisters AdGem events callback.
     *
     * @param callback instance of {@link OfferWallCallback} previously registered via {@link #registerOfferWallCallback(OfferWallCallback)}.
     */
    @MainThread
    public abstract void unregisterOfferWallCallback(OfferWallCallback callback);

    /**
     * Shows interstitial advertisement. AdGem will notify about completion via {@link AdGemCallback#onInterstitialAdClosed()}.
     * <p>
     * If interstitial ad is not in {@link #getRewardedAdState() ready} state, then nothing is shown.
     *
     * @throws IllegalStateException if AdGem was not configured to serve interstitial ads.
     */
    public abstract void showInterstitialAd(Context context);

    /**
     * Shows rewarded advertisement. AdGem will notify about completion via {@link AdGemCallback#onRewardedAdComplete()}.
     * <p>
     * If a rewarded ad is not in {@link #getRewardedAdState() ready} state, then nothing is shown.
     *
     * @throws IllegalStateException if AdGem was not configured to serve rewarded ads.
     */
    public abstract void showRewardedAd(Context context);

    /**
     * Shows offer wall. Offer wall status callbacks (including even when offer wall is closed) will be delivered via the {@link OfferWallCallback}.
     */
    public abstract void showOfferWall(Context context);

    /**
     * Returns last known error.
     *
     * @return last known error. Error state is notified via one of {@link AdGemCallback#onOfferWallStateChanged(int)},
     * {@link AdGemCallback#onInterstitialAdStateChanged(int)} or {@link AdGemCallback#onRewardedAdStateChanged(int)}.
     */
    public abstract Error getError();

    /**
     * Sets additional metadata specific to the user of this application.
     *
     * @param metaData extra information about the player.
     * @see PlayerMetadata.Builder for additional fields.
     */
    public abstract void setPlayerMetaData(PlayerMetadata metaData);
}
