package com.adgem.android;

import androidx.annotation.IntRange;
import androidx.annotation.NonNull;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

public final class PlayerMetadata {
    /**
     * Represents player's gender.
     */
    public enum Gender {
        MALE,
        FEMALE
    }

    public final Map<String, String> entries;

    private PlayerMetadata(Map<String, String> entries) {
        this.entries = Collections.unmodifiableMap(entries);
    }

    public static final class Builder {
        private static final int MAX_LENGTH = 65535;
        private final Map<String, String> mEntries = new HashMap<>();

        private Builder(String playerId) {
            this.id(playerId);
        }

        public static Builder createWithPlayerId(String playerId) {
            return new Builder(playerId);
        }

        /**
         * Sets player ID.
         *
         * @param playerId unique player ID.
         * @return instance of this builder.
         */
        @NonNull
        public Builder id(@NonNull String playerId) {
            //noinspection ConstantConditions
            if (playerId == null) {
                throw new IllegalArgumentException("Player ID can not be null");
            }
            if (playerId.length() > MAX_LENGTH) {
                throw new IllegalArgumentException("Player ID can not be more than " + MAX_LENGTH + " characters");
            }
            mEntries.put("player_id", playerId);

            // This extra `playerid` entry is required to properly start a new session since the
            // corresponding request still relies on legacy parameter name
            mEntries.put("playerid", playerId);

            return this;
        }

        /**
         * Sets player's age.
         *
         * @param age value between 0 and 99.
         * @return instance of this builder.
         */
        @NonNull
        public Builder age(@IntRange(from = 0, to = 99) int age) {
            if (age < 0 || age > 99) {
                throw new IllegalArgumentException("Age must be between 0 and 99");
            }
            mEntries.put("player_age", Integer.toString(age));
            return this;
        }


        /**
         * Sets player's gender.
         *
         * @param gender one of {@link Gender} values.
         * @return instance of this builder.
         */
        @NonNull
        public Builder gender(@NonNull Gender gender) {
            //noinspection ConstantConditions
            if (gender == null) {
                throw new IllegalArgumentException("Gender can not be null");
            }
            switch (gender) {
                case MALE:
                    mEntries.put("player_gender", "male");
                    break;
                case FEMALE:
                    mEntries.put("player_gender", "female");
                    break;
                default:
                    throw new IllegalStateException("Unknown gender");
            }
            return this;
        }

        /**
         * Sets player's level.
         *
         * @param level between 0 and 1000000000.
         * @return instance of this builder.
         */
        @NonNull
        public Builder level(@IntRange(from = 0, to = 1000000000) int level) {
            if (level < 0 || level > 1000000000) {
                throw new IllegalArgumentException("Level must be between 0 and 1,000,000,000");
            }
            mEntries.put("player_level", Integer.toString(level));
            return this;
        }

        /**
         * Sets player's placement/rank.
         *
         * @param rank between 0 and 1000000000.
         * @return instance of this builder.
         */
        @NonNull
        public Builder placement(long rank) {
            if (rank < 0 || rank > 1000000000) {
                throw new IllegalArgumentException("Rank must be between 0 and 1,000,000,000");
            }
            mEntries.put("placement", Long.toString(rank));
            return this;
        }

        /**
         * Sets value indicating whether this player is a payer.
         *
         * @param payer <code>true</code> if this player is a payer, <code>false</code> otherwise.
         * @return instance of this builder.
         */
        @NonNull
        public Builder isPayer(boolean payer) {
            mEntries.put("player_payer", Boolean.toString(payer));
            return this;
        }

        /**
         * @param value between 0 and 99999.9
         * @return instance of this builder.
         */
        @NonNull
        public Builder iapTotalUsd(float value) {
            if (value < 0 || value > 99999.99f) {
                throw new IllegalArgumentException("Value should be in a range between 0 and 99999.99");
            }
            mEntries.put("player_iap_total_usd", Float.toString(value));
            return this;
        }

        /**
         * Sets player creation date and time.
         *
         * @param timestamp creation time in YYYY-mm-dd HH:mm:ss format. Example: 2018-05-12 14:44:45
         * @return instance of this builder.
         */
        @NonNull
        public Builder createdAt(@NonNull String timestamp) {
            //noinspection ConstantConditions
            if (timestamp == null) {
                throw new IllegalArgumentException("Timestamp can not be null");
            }
            mEntries.put("player_created_at", timestamp);
            return this;
        }

        /**
         * Sets custom fields sent to AdGem on every request.
         *
         * @param fields first 5 elements only. Each field will be sent using c1..c5 key.
         * @return instance of this builder.
         *
         * @Deprecated use {@link Builder#customField1(String)}..{@link Builder#customField5(String)} instead.
         */
        public Builder customFields(@NonNull String... fields) {
            //noinspection ConstantConditions
            if (fields == null) {
                throw new IllegalArgumentException("Fields can not be null");
            }

            if (fields.length > 0) {
                for (int i = 1; i <= 5 && i <= fields.length; i++) {
                    String field = fields[i - 1];
                    if (field == null) {
                        throw new IllegalArgumentException("Field on position " + i + " can not be null");
                    }
                    if (field.length() > 99) {
                        throw new IllegalArgumentException("Field on position " + i + " can not be over 99 characters");
                    }
                    mEntries.put("c" + Integer.toString(i), fields[i - 1]);
                }
            }
            return this;
        }

        /**
         * Sets custom field sent to AdGem on every request.
         *
         * @param field custom parameter value as set by the publisher. Field will be sent using c1 key.
         * @return instance of this builder.
         */
        @NonNull
        public Builder customField1(@NonNull String field) {
            return customField(field, 1);
        }

        /**
         * Sets custom field sent to AdGem on every request.
         *
         * @param field custom parameter value as set by the publisher. Field will be sent using c2 key.
         * @return instance of this builder.
         */
        @NonNull
        public Builder customField2(@NonNull String field) {
            return customField(field, 2);
        }

        /**
         * Sets custom field sent to AdGem on every request.
         *
         * @param field custom parameter value as set by the publisher. Field will be sent using c3 key.
         * @return instance of this builder.
         */
        @NonNull
        public Builder customField3(@NonNull String field) {
            return customField(field, 3);
        }

        /**
         * Sets custom field sent to AdGem on every request.
         *
         * @param field custom parameter value as set by the publisher. Field will be sent using c4 key.
         * @return instance of this builder.
         */
        @NonNull
        public Builder customField4(@NonNull String field) {
            return customField(field, 4);
        }

        /**
         * Sets custom field sent to AdGem on every request.
         *
         * @param field custom parameter value as set by the publisher. Field will be sent using c5 key.
         * @return instance of this builder.
         */
        @NonNull
        public Builder customField5(@NonNull String field) {
            return customField(field, 5);
        }

        private Builder customField(String field, int i) {
            //noinspection ConstantConditions
            if (i < 1 || i > 5) {
                throw new IllegalArgumentException("Field index should be in a range between 1 and 5");
            }
            if (field == null) {
                throw new IllegalArgumentException("Field c" + i + " can not be null");
            }
            if (field.length() > 99) {
                throw new IllegalArgumentException("Field c" + i + " can not be over 99 characters");
            }
            mEntries.put("c" + Integer.toString(i), field);
            return this;
        }

        @NonNull
        public PlayerMetadata build() {
            return new PlayerMetadata(mEntries);
        }
    }
}
