package com.adlibsoftware.authorize;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.io.StringWriter;
import java.net.URL;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Base64;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ssl.HttpsURLConnection;


/**
 * Class used to authorize Adlib by getting a Token using Oauth 2.0
 * @author mmanley
 *
 */
public class TokenAuthorizer {

	private String token = null;	
	private LocalDateTime tokenAcquiredOn;
	
	private int tokenRefreshRateHours = 29;
	
	private static final Pattern PATTERN = Pattern.compile(".*\"access_token\"\\s*:\\s*\"([^\"]+)\".*");
	private static final String CLIENT_ID = "AdlibElevate";//clientId
	private static final String CLIENT_SECRET = "Adlib";//client secret
	private static final String AUTH = CLIENT_ID + ":" + CLIENT_SECRET;
	private static final String AUTHENTICATION = Base64.getEncoder().encodeToString(AUTH.getBytes());
	
	private String username;
	private String encryptedPassword;
	private String tokenUrl;
	
	public TokenAuthorizer() {
		
	}
	
	public TokenAuthorizer(String username, String encryptedPassword, URL tokenUrl) {
		setTokenProperties(username, encryptedPassword, tokenUrl);
	}
	
	/**
	 * Get stored token, or retrieve new one if first time/expired
	 * @return
	 * @throws Exception
	 */
	public synchronized String getToken() throws Exception {

		if (username == null || tokenUrl == null) {
			throw new Exception("setTokenProperties not called yet");
		}
		if (IsTokenExpired()) {
			getNewToken();
		}
		return token;
	}
	
	private void getNewToken() throws Exception {
	    String content = "grant_type=password&username=" + username + "&password=" + AES.decrypt(encryptedPassword);
	    BufferedReader reader = null;
	    HttpsURLConnection connection = null;
	    String tokenValue = null;
	    try {
	    	URL url = new URL(tokenUrl);
	        connection = (HttpsURLConnection) url.openConnection();
	        connection.setRequestMethod("POST");
	        connection.setDoOutput(true);
	        connection.setRequestProperty("Authorization", "Basic " + AUTHENTICATION);
	        connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
	        connection.setRequestProperty("Accept", "application/json");
	        PrintStream os = new PrintStream(connection.getOutputStream());
	        os.print(content);
	        os.close();
	        reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));
	        String line = null;
	        StringWriter out = new StringWriter(connection.getContentLength() > 0 ? connection.getContentLength() : 2048);
	        while ((line = reader.readLine()) != null) {
	            out.append(line);
	        }
	        String response = out.toString();
	        Matcher matcher = PATTERN.matcher(response);
	        if (matcher.matches() && matcher.groupCount() > 0) {
	            tokenValue = matcher.group(1);
	        }
	    } 
	    catch (Exception e) {
	    	token = null;
	    	throw e;
	    }
	    finally {
	        if (reader != null) {
	            try {
	                reader.close();
	            } catch (IOException e) {
	            }
	        }
	        if (connection != null) {
	        	connection.disconnect();
	        }	        
	    }
	    if (tokenValue != null) {
	    	tokenAcquiredOn = LocalDateTime.now();
	    	token = tokenValue;
	    } else {
	    	throw new Exception("Could not acquire token");
	    }
	}
	
	public void setTokenProperties(String username, String encryptedPassword, URL tokenUrl) {
		this.username = username;
		this.encryptedPassword = encryptedPassword;
		this.tokenUrl = tokenUrl.toString();
	}	
	
	private boolean IsTokenExpired() {
		if (token == null) {
			return true;
		}
		LocalDateTime now = LocalDateTime.now();
		
		Duration duration = Duration.between(now, tokenAcquiredOn);
		
		long durationInHours = Math.abs(duration.toHours());
		
		if (durationInHours >= tokenRefreshRateHours) {
			return true;
		}
		
		return false;
	}

	/**
	 * @return the tokenRefreshRateHours
	 */
	public int getTokenRefreshRateHours() {
		return tokenRefreshRateHours;
	}

	/**
	 * @param tokenRefreshRateHours the tokenRefreshRateHours to set
	 */
	public void setTokenRefreshRateHours(int tokenRefreshRateHours) {
		this.tokenRefreshRateHours = tokenRefreshRateHours;
	}
}
