package com.adlibsoftware.integration;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Comparator;
import java.util.TreeSet;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import com.adlibsoftware.client.ArrayOfJobFile;
import com.adlibsoftware.client.ArrayOfMetadataItem;
import com.adlibsoftware.client.ArrayOfOrchestrationFileStatusResponse;
import com.adlibsoftware.client.ArrayOflong;
import com.adlibsoftware.client.ContentType;
import com.adlibsoftware.client.JobFile;
import com.adlibsoftware.client.JobManagement;
import com.adlibsoftware.client.JobState;
import com.adlibsoftware.client.JobStatusResponse;
import com.adlibsoftware.client.MetadataItem;
import com.adlibsoftware.client.MetadataType;
import com.adlibsoftware.client.OrchestrationFileStatusResponse;
import com.adlibsoftware.client.Payload;
import com.adlibsoftware.exceptions.AdlibTimeoutException;

/**
 * Common class for static helper methods
 * @author mmanley
 *
 */
public class Common {
	

	private static Object locker = new Object();
	
	/**
	 * Rudimentary JobManagement connection test
	 * @param repositoryName
	 * @param jobManagementService
	 * @throws Exception
	 */
	public static void testConnection(String repositoryName, JobManagement jobManagementService) throws Exception {
		jobManagementService.getProcessedJobsForRepository(repositoryName);
	}
	
	/**
	 * Metadata helper method
	 * @param metadataList
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	public static Object getMetadataValueByName(ArrayOfMetadataItem metadataList, String name, Object defaultValue) {
		for (MetadataItem item : metadataList.getMetadataItem()) {
			if (item.getName().equalsIgnoreCase(name)) {
				if (item.getValue() instanceof org.w3c.dom.Element) {
					org.w3c.dom.Element itemNode = (org.w3c.dom.Element) item.getValue();
					return itemNode.getFirstChild().getNodeValue();
				}
				return item.getValue();
			}
		}
		return defaultValue;
	}
	
	/***
	 * This method will ensure all input files exist
	 * and will ensure metadata is properly set
	 * @param inputPayload the payload to validate
	 * @param skipFileCheck - Skip file exists check
	 * @throws IOException 
	 */
	public static void validatePayload(Payload inputPayload, boolean skipFileCheck) throws IOException {
		if (inputPayload.getFiles().getJobFile().size() == 1 
				&& inputPayload.getFiles().getJobFile().get(0).getMetadata().getMetadataItem().size() > 0) {
			
			inputPayload.getMetadata().getMetadataItem().addAll(inputPayload.getFiles().getJobFile().get(0).getMetadata().getMetadataItem());

			Collection<MetadataItem> uniqueMetadata = null;
			synchronized (locker) {
				// remove duplicates after merging
				uniqueMetadata = inputPayload.getMetadata().getMetadataItem()
		                .stream() // get stream for original list
		                .collect(Collectors.toCollection(//distinct elements stored into new SET
		                    () -> new TreeSet<>(Comparator.comparing(MetadataItem::getName)))
		                        );
			}		
			
			inputPayload.getMetadata().getMetadataItem().clear();
			inputPayload.getMetadata().getMetadataItem().addAll(uniqueMetadata);
			inputPayload.getFiles().getJobFile().get(0).getMetadata().getMetadataItem().clear();
			inputPayload.getFiles().getJobFile().get(0).getMetadata().getMetadataItem().addAll(uniqueMetadata);
		}
		
		if (skipFileCheck) {
			return;
		}
		
		for (JobFile jobFile : inputPayload.getFiles().getJobFile()) {				
			
			File localFile = new File(jobFile.getPath());
			
			if (!localFile.exists()) {
				throw new FileNotFoundException(String.format("Could not find local file %s", jobFile.getPath()));
			}
		}
		
	}
	
	/**
	 * Used by exception handling.  Cancels the job and waits for it to be cancelled.
	 * @param jobId Job ID found when getting Orchestration Status
	 * @param jobManagementService
	 * @param stateToComplete usually Cancelled or Failed
	 * @param waitForCancel when false, timeout and pollingInterval are ignored and does not wait for cancel and does not call complete
	 * @param timeout
	 * @param pollingInterval
	 * @return
	 */
	public static boolean cancelAndCompleteJob(long jobId, JobManagement jobManagementService, JobState stateToComplete, boolean waitForCancel, Duration timeout, Duration pollingInterval) {
		try {
			ArrayOflong jobIds = new ArrayOflong();
			jobIds.getLong().add(jobId);
			
			jobManagementService.cancelJob(jobId);
			
			if (!waitForCancel) {
				return true;
			}
			
			LocalDateTime startTime = LocalDateTime.now();
			JobStatusResponse response = jobManagementService.getJobsStatus(jobIds).getJobStatusResponse().get(0);

			while (response.getState() != JobState.CANCELLED && response.getState() != JobState.FAILED  && response.getState() != JobState.SUCCESSFUL) {
				Duration duration = Duration.between(LocalDateTime.now(), startTime);
				long durationInMilliseconds = Math.abs(duration.toMillis());
				if (durationInMilliseconds > timeout.toMillis()) {
					throw new AdlibTimeoutException("Cancel operation timed out.", jobId);
				}	
				Thread.sleep(pollingInterval.toMillis());
				response = jobManagementService.getJobsStatus(jobIds).getJobStatusResponse().get(0);
			}			
			try {
				jobManagementService.completeJob(jobId, stateToComplete);
			} catch (Exception e) {
				// do nothing
			}			
			return true;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}
	
	/**
	 * Waits for job to process or timeout to be reached.
	 * @param fileId
	 * @param jobManagementService
	 * @param timeout
	 * @param pollingInterval
	 * @return JobStatusResponse
	 * @throws TimeoutException, Exception
	 */
	public static OrchestrationFileStatusResponse waitForJobToProcess(long fileId, JobManagement jobManagementService, Duration timeout, Duration pollingInterval) throws AdlibTimeoutException, Exception {
		
		ArrayOflong fileIds = new ArrayOflong();
		fileIds.getLong().add(fileId);		
		
		LocalDateTime startTime = LocalDateTime.now();
		
		ArrayOfOrchestrationFileStatusResponse response = jobManagementService.getOrchestrationFileStatus(fileIds);
		while (response == null || response.getOrchestrationFileStatusResponse().size() == 0 || !isOrchestrationComplete(response.getOrchestrationFileStatusResponse().get(0))) {
			Duration duration = Duration.between(LocalDateTime.now(), startTime);			
			long durationInMilliseconds = Math.abs(duration.toMillis());
			if (durationInMilliseconds > timeout.toMillis()) {
				throw new AdlibTimeoutException(String.format("Operation timed out for File ID %s after %s seconds.", fileId, timeout.toMillis() / 1000.0)
						, fileId);
			}			
			Thread.sleep(pollingInterval.toMillis());
			response = jobManagementService.getOrchestrationFileStatus(fileIds);
		}		
		return response.getOrchestrationFileStatusResponse().get(0);		
	}
	
	/**
	 * Determines whether Orchestration is complete (whether success or fail)
	 * @param response OrchestrationFileStatusResponse
	 * @return
	 * @throws Exception
	 */
	public static boolean isOrchestrationComplete(OrchestrationFileStatusResponse response) throws Exception {
		if (response == null) {
			return false;
		}
		return response.isAlertActive() || (response.getCompletedTime() != null && response.getCompletedTime().isValid() && response.getCompletedTime().getYear() > 2000 &&
				response.getCurrentStage() == response.getTotalStage());
	}
	
	/**
	 * Makes a JobFile object based on File and ArrayOfMetadataItem
	 * @param file
	 * @param metadataList if null, will seed OriginalFileName metadata
	 * @return
	 */
	public static JobFile makeJobFile(File file, ArrayOfMetadataItem metadataList) {
		JobFile jobFile = new JobFile();
		ContentType contentType = new ContentType();
		contentType.setName("Unknown");
		jobFile.setContentType(contentType);
		jobFile.setPath(file.getAbsolutePath());
		if (metadataList == null) {
			// Set default file metadata
			ArrayOfMetadataItem fileMetadata = new ArrayOfMetadataItem();
			MetadataItem metadata = new MetadataItem();
			metadata.setName("OriginalFileName");
			metadata.setValue(file.getName());
			metadata.setType(MetadataType.STRING);
			fileMetadata.getMetadataItem().add(metadata);
			jobFile.setMetadata(fileMetadata);
		} else {
			jobFile.setMetadata(metadataList);
		}		
		return jobFile;
	}
	
	/**
	 * Makes a Payload object from list of payload metadata and list of files
	 * @param jobFiles
	 * @param payloadMetadata
	 * @param validate whether or not to validate payload files exist and fix metadata
	 * @param skipFileCheck Skip file exists check
	 * @return payload to submit with
	 * @throws IOException
	 */
	public static Payload makePayload(File[] jobFiles, ArrayOfMetadataItem payloadMetadata, boolean validate, boolean skipFileCheck) throws IOException {
		Payload inputPayload = new Payload();
		ArrayOfJobFile arrayOfJobFiles = Common.setInputFiles(jobFiles);
		
		inputPayload.setFiles(arrayOfJobFiles);
		inputPayload.setMetadata(payloadMetadata);
		
		if (validate) {
			Common.validatePayload(inputPayload, skipFileCheck);
		}
		
		return inputPayload;
	}
	
	/**
	 * Creates ArrayOfJobFile with default JobFile Metadata (OriginalFileName)
	 * @param files
	 * @return
	 */
	public static ArrayOfJobFile setInputFiles(File[] files) {
		ArrayOfJobFile inputFiles = new ArrayOfJobFile();
		
		setInputFiles(files, inputFiles, false);
		
		return inputFiles;
	}
	
	private static void setInputFiles(File[] files, ArrayOfJobFile inputFiles, boolean recurse) {
		
		for (File file : files) {
	        if (file.isDirectory() && recurse) {
	            setInputFiles(file.listFiles(), inputFiles, recurse);
	        } else {
	            JobFile jobFile = makeJobFile(file, null);
	            inputFiles.getJobFile().add(jobFile);
	        }
	    }
	}
	
	public static String getFriendlyError(Exception e) {
		
		if (e.getMessage().contains("Connection timed out")) {
			return "E105: Connection timed out (Adlib server is not reachable).";
		} else if (e.getMessage().contains("Server returned HTTP response code: 400 for URL")) {
			return "E100: Invalid username and/or password for Token Service.";
		} else if (e.getMessage().contains("unable to find valid certification path to requested target")) {
			return "E102: Adlib Server Certificate is not trusted.  If it is self-signed, you must add it to the keystore using keytool";
		} else if (e.getMessage().contains("Failed to access the WSDL at:")) {
			return "E103: Invalid Server Name for Job Management Service (or cannot access service).";
		} else if (e.getMessage().contains("unable to find valid certification path to requested target")) {
			return "E104: Adlib Server certificate is not trusted.  If it is self-signed, please add the .cer to JVM keystore using keytool";
		} else if (e.getMessage().contains("Repository") && e.getMessage().contains("does not exist")) {
			return "E200: Invalid Repository Name";
		} else if (e.getMessage().contains("does not have access permissions for repository")) {
			return "E201: User does not have sufficient rights for repository";
		} else if (e.getMessage().contains("A workflow must first be assigned to the repository")) {
			return "E202: A workflow must first be assigned to this repository in Adlib Vision Console.";
		} else {
			return e.getMessage();
		}
	}
}
