package com.adlibsoftware.integration;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeoutException;

import javax.xml.ws.Binding;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.soap.AddressingFeature;

import com.adlibsoftware.authorize.TokenAuthorizer;
import com.adlibsoftware.client.ArrayOfOrchestrationFileStatusResponse;
import com.adlibsoftware.client.ArrayOfSubmitFileInfo;
import com.adlibsoftware.client.ArrayOflong;
import com.adlibsoftware.client.GetJobFileOptions;
import com.adlibsoftware.client.GetJobFilePartResponse2;
import com.adlibsoftware.client.GetLibraryFilePartResponse2;
import com.adlibsoftware.client.HashAlgorithm;
import com.adlibsoftware.client.JobDetails;
import com.adlibsoftware.client.JobFile;
import com.adlibsoftware.client.JobManagement;
import com.adlibsoftware.client.JobManagementBeginGetLibraryFileExceptionDetailFaultFaultMessage;
import com.adlibsoftware.client.JobManagementGetLibraryFilePartExceptionDetailFaultFaultMessage;
import com.adlibsoftware.client.JobManagement_Service;
import com.adlibsoftware.client.JobState;
import com.adlibsoftware.client.ObjectFactory;
import com.adlibsoftware.client.OrchestrationFileStatusResponse;
import com.adlibsoftware.client.Payload;
import com.adlibsoftware.client.RenditionType;
import com.adlibsoftware.client.StartGetJobFileResponse;
import com.adlibsoftware.client.StartGetLibraryFileResponse;
import com.adlibsoftware.client.SubmitFileInfo;
import com.adlibsoftware.exceptions.AdlibException;
import com.adlibsoftware.exceptions.AdlibNotConfiguredException;
import com.adlibsoftware.exceptions.AdlibTimeoutException;
import com.adlibsoftware.handlers.HeaderHandler;

/**
 * This is a helper class for API integrations to Adlib Elevate Job Management Web Service
 * @author mmanley
 * @apiNote The methods in this class are thread-safe (stateless)
 * 
 */
public class JobManagementServiceClient {

	private Settings settings;
	private JobManagement jobManagement;
	private JobManagement_Service service;
	
	/**
	 * 
	 * @param settings The required settings
	 * @param autoInitialize If true, will initialize immediately
	 * @throws Exception 
	 */
	public JobManagementServiceClient(Settings settings, boolean autoInitialize) throws Exception {
		this.setSettings(settings);
		if (autoInitialize) {
			Initialize();
		}
	}
	
	/**
	 * Initializes (or re-initializes) Job Management Service and performs a connection test
	 * @throws Exception
	 */
	@SuppressWarnings("rawtypes")
	public void Initialize() throws Exception {
		
		service = new JobManagement_Service(getSettings().getJobManagementServiceWsdlUrl());	        
        
		jobManagement = service.getWS2007FederationHttpBindingJobManagement(new AddressingFeature(true, true));
		
		TokenAuthorizer tokenAuthorizer = new TokenAuthorizer(getSettings().getTokenServiceUsername(), getSettings().getTokenServiceEncryptedPassword(), getSettings().getTokenServiceUrl());
		tokenAuthorizer.setTokenRefreshRateHours(getSettings().getTokenRefreshRateHours());
		// setup the handler chain (to insert the token into SOAP header)
		HeaderHandler handler = new HeaderHandler(tokenAuthorizer);		
		Binding binding = ((BindingProvider)jobManagement).getBinding();
		List<Handler> handlerList = binding.getHandlerChain();
		handlerList.add(handler);
		binding.setHandlerChain(handlerList);
		
		// get first token
		tokenAuthorizer.getToken();
		// quick connection test for repository
		Common.testConnection(settings.getRepositoryName(), jobManagement);
	}
	
	/**
	 * Submits a job Payload to Adlib, and only returns after job is completed
	 * @param repositoryName
	 * @param inputPayload
	 * @param timeout
	 * @param pollingInterval
	 * @return ProcessedJobResponse
	 * @throws Exception
	 */
	public ProcessedJobResponse submitSynchronousJob(String repositoryName, Payload inputPayload, Duration timeout, Duration pollingInterval) throws Exception {
		
		if (timeout == null) {
			timeout = settings.getDefaultTimeout();
		}
		if (pollingInterval == null) {
			pollingInterval = settings.getDefaultPollingInterval();
		}
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = 0;
		try {
			
			ArrayOfSubmitFileInfo submitFileInfos = jobManagementService.submitFiles(settings.getRepositoryName(), settings.getLocationName(), inputPayload, true, false);
			
			if (submitFileInfos == null || submitFileInfos.getSubmitFileInfo().size() == 0) {
				throw new AdlibNotConfiguredException(String.format("Could not submit files (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
						, settings.getRepositoryName()
						, settings.getLocationName()));
			}
			
			fileId = submitFileInfos.getSubmitFileInfo().get(0).getFileId();
			
			jobManagementService.beginOrchestration(fileId, "");
			
			OrchestrationFileStatusResponse response = Common.
					waitForJobToProcess(fileId, jobManagementService, timeout, pollingInterval);
			
			JobDetails jobDetails = jobManagementService.getJobDetails(response.getJobId());
			
			return getProcessedJobResponse(fileId, jobDetails);
			
		} catch (Exception e) {
			if (fileId > 0) {
				throw new Exception(String.format("Error submitting to Adlib for fileID %s", fileId));
			}
			throw e;			
		}
	}
	
	/**
	 * Submits a job Payload to Adlib, and only returns after job is completed
	 * @param repositoryName
	 * @param inputPayload
	 * @return long FileId
	 * @throws Exception
	 */
	public long submitJob(String repositoryName, Payload inputPayload) throws Exception {
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = 0;
		try {
			
			ArrayOfSubmitFileInfo submitFileInfos = jobManagementService.submitFiles(settings.getRepositoryName(), settings.getLocationName(), inputPayload, true, false);
			
			if (submitFileInfos == null || submitFileInfos.getSubmitFileInfo().size() == 0) {
				throw new AdlibNotConfiguredException(String.format("Could not submit job (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
						, settings.getRepositoryName()
						, settings.getLocationName()));
			}
			
			fileId = submitFileInfos.getSubmitFileInfo().get(0).getFileId();
			
			jobManagementService.beginOrchestration(fileId, "");
			
			return fileId;
			
		} catch (Exception e) {
			if (fileId > 0) {
				throw new Exception(String.format("Error submitting to Adlib for fileID %s", fileId));
			}
			throw e;			
		}
	}
	
	public ProcessedJobResponse getProcessedJobResponse(OrchestrationFileStatusResponse orchestrationFileStatusResponse) throws Exception {		
		if (!Common.isOrchestrationComplete(orchestrationFileStatusResponse)) {
			return null;
		}
		
		JobDetails jobDetails = jobManagement.getJobDetails(orchestrationFileStatusResponse.getJobId());
		
		return getProcessedJobResponse(orchestrationFileStatusResponse.getFileId(), jobDetails);	
		
	}

	private ProcessedJobResponse getProcessedJobResponse(long fileId, JobDetails jobDetails) {
		ProcessedJobResponse processedJobResponse = new ProcessedJobResponse();
		processedJobResponse.setOutputPayload(jobDetails.getOutputPayload());
		processedJobResponse.setJobState(jobDetails.getState());
		processedJobResponse.setFileId(fileId);
		processedJobResponse.setSuccessful(jobDetails.getState() == JobState.SUCCESSFUL);	
		processedJobResponse.setLastJobId(jobDetails.getJobId());
		if (!processedJobResponse.isSuccessful()) {
			processedJobResponse.setExceptionInfo(jobDetails.getExceptionInfo());
		}
		return processedJobResponse;
	}
	
	/**
	 * Submits a job Payload to Adlib and streams files in Payload, and only returns after job is completed
	 * @param repositoryName
	 * @param inputPayload
	 * @param timeout
	 * @param pollingInterval
	 * @return ProcessedJobResponse
	 * @throws Exception
	 */
	public ProcessedJobResponse streamSynchronousJob(String repositoryName, Payload inputPayload, Duration timeout, Duration pollingInterval) throws Exception {
		
		if (timeout == null) {
			timeout = settings.getDefaultTimeout();
		}
		if (pollingInterval == null) {
			pollingInterval = settings.getDefaultPollingInterval();
		}
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = streamJob(repositoryName, inputPayload);		
				
		OrchestrationFileStatusResponse response = Common.
				waitForJobToProcess(fileId, jobManagementService, timeout, pollingInterval);
					
		JobDetails jobDetails = jobManagementService.getJobDetails(response.getJobId());
		
		return getProcessedJobResponse(fileId, jobDetails);		
		
	}
	
	/**
	 * Submits a job Payload to Adlib and streams files in Payload, and only returns after job is completed
	 * @param repositoryName
	 * @param inputPayload
	 * @param jobFiles - actual list of bytes of each input file (matching name list in payload)
	 * @param timeout
	 * @param pollingInterval
	 * @return ProcessedJobResponse
	 * @throws Exception
	 */
	public ProcessedJobResponse streamSynchronousJob(String repositoryName, 
			Payload inputPayload, 
			List<byte[]> jobFiles, 
			Duration timeout, 
			Duration pollingInterval) throws Exception {
		
		if (timeout == null) {
			timeout = settings.getDefaultTimeout();
		}
		if (pollingInterval == null) {
			pollingInterval = settings.getDefaultPollingInterval();
		}
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = streamJob(repositoryName, inputPayload, jobFiles);		
				
		OrchestrationFileStatusResponse response = Common.
				waitForJobToProcess(fileId, jobManagementService, timeout, pollingInterval);
					
		JobDetails jobDetails = jobManagementService.getJobDetails(response.getJobId());
		
		return getProcessedJobResponse(fileId, jobDetails);		
		
	}
	
	/**
	 * Submits the job via streaming and updates the Payload.  Does not wait for completion of job.
	 * @param repositoryName
	 * @param inputPayload
	 * @throws Exception
	 */
	public long streamJob(String repositoryName, Payload inputPayload) throws Exception {
		
		Common.validatePayload(inputPayload, true);
		
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = -1;
		try {
			SubmitFileInfo submitFileInfo = null;
			for (JobFile jobFile : inputPayload.getFiles().getJobFile()) {				
				
				File localFile = new File(jobFile.getPath());
				
				if (!localFile.exists()) {
					throw new FileNotFoundException(String.format("Could not find local file %s", jobFile.getPath()));
				}
				
				// set to just file name
				jobFile.setPath(localFile.getName());
				ArrayOfSubmitFileInfo arrayOfSubmitFileInfo = jobManagementService.beginSubmitFile(settings.getRepositoryName(), settings.getLocationName(), jobFile);
				
				if (arrayOfSubmitFileInfo == null || arrayOfSubmitFileInfo.getSubmitFileInfo().size() == 0) {
					throw new AdlibNotConfiguredException(String.format("Could not begin job (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
							, settings.getRepositoryName()
							, settings.getLocationName()));
				}
				
				submitFileInfo = arrayOfSubmitFileInfo.getSubmitFileInfo().get(0);
				fileId = submitFileInfo.getFileId();			
				String filePartContext = jobManagementService.beginPutLibraryFile(fileId, localFile.getName());
				
				FileInputStream inputFileStream = null;
				try {
					byte[] buffer = new byte[settings.getStreamingBufferSizeBytes()];
					inputFileStream = new FileInputStream(localFile);
					int bytesRead = inputFileStream.read(buffer);
					while(bytesRead != -1)
					{
						filePartContext = jobManagementService.putLibraryFilePart(filePartContext, buffer, bytesRead);
						// next read
						bytesRead = inputFileStream.read(buffer); 
					}
					
					byte[] hashValue = settings.getHashAlgorithm() == HashAlgorithm.NONE ? null : getFileHash(localFile);
					JobFile uploadedFile = jobManagementService.endPutLibraryFile(filePartContext, settings.getHashAlgorithm(), hashValue);
					uploadedFile.setMetadata(jobFile.getMetadata());
					
					uploadedFile = jobManagementService.endSubmitFile(fileId, uploadedFile);
					jobFile.setPath(uploadedFile.getPath());
				}
				finally {
					if (inputFileStream != null) { 
						inputFileStream.close();
					}
				}
			}					
			
			if (inputPayload.getFiles().getJobFile().size() > 1) {
				ArrayOfSubmitFileInfo submitFileInfos = jobManagementService.submitFiles(settings.getRepositoryName(), settings.getLocationName(), inputPayload, true, false);
				
				if (submitFileInfos == null || submitFileInfos.getSubmitFileInfo().size() == 0) {
					throw new AdlibNotConfiguredException(String.format("Could not begin job (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
							, settings.getRepositoryName()
							, settings.getLocationName()));
				}
				submitFileInfo = submitFileInfos.getSubmitFileInfo().get(0);
				fileId = submitFileInfo.getFileId();	
			}					
			
			jobManagementService.beginOrchestration(fileId, "");
			
			return fileId;
			
		} catch (Exception e) {
			if (fileId > 0) {
				throw new Exception(String.format("Error streaming to Adlib for fileID %s", fileId), e);
			}
			throw e;			
		}
	}
	
	
	/**
	 * Submits the job via streaming and updates the Payload.  Does not wait for completion of job.
	 * @param repositoryName
	 * @param inputPayload where JobFile objects are just the input file names
	 * @param jobFiles - actual list of bytes of each input file (matching list in payload)
	 * @throws Exception 
	 * @throws AdlibNotConfiguredException 
	 */
	public long streamJob(String repositoryName, Payload inputPayload, List<byte[]> jobFiles) 
			throws 
			Exception {		
		
		Common.validatePayload(inputPayload, true);
		
		JobManagement jobManagementService = this.getJobManagement();
		long fileId = -1;
		try {
			SubmitFileInfo submitFileInfo = null;
			int index = -1;
			for (JobFile jobFile : inputPayload.getFiles().getJobFile()) {				
				index++;
				ArrayOfSubmitFileInfo arrayOfSubmitFileInfo = jobManagementService.beginSubmitFile(settings.getRepositoryName(), settings.getLocationName(), jobFile);
				
				if (arrayOfSubmitFileInfo == null || arrayOfSubmitFileInfo.getSubmitFileInfo().size() == 0) {
					throw new AdlibNotConfiguredException(String.format("Could not begin job (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
							, settings.getRepositoryName()
							, settings.getLocationName()));
				}
				
				submitFileInfo = arrayOfSubmitFileInfo.getSubmitFileInfo().get(0);
				fileId = submitFileInfo.getFileId();			
				String filePartContext = jobManagementService.beginPutLibraryFile(fileId, jobFile.getPath());
				
				ByteArrayInputStream jobFileStream = null; 
				try {
					jobFileStream = new ByteArrayInputStream(jobFiles.get(index));
					byte[] buffer = new byte[settings.getStreamingBufferSizeBytes()];
					int bytesRead = jobFileStream.read(buffer);
					while(bytesRead != -1)
					{
						filePartContext = jobManagementService.putLibraryFilePart(filePartContext, buffer, bytesRead);
						// next read
						bytesRead = jobFileStream.read(buffer); 
					}
					
					byte[] hashValue = null;
					try {
						hashValue = settings.getHashAlgorithm() == HashAlgorithm.NONE ? null : getFileContentHash(jobFiles.get(index));
					} catch (NoSuchAlgorithmException e) {
						e.printStackTrace();
					}
					JobFile uploadedFile = jobManagementService.endPutLibraryFile(filePartContext, settings.getHashAlgorithm(), hashValue);
					uploadedFile.setMetadata(jobFile.getMetadata());
					
					uploadedFile = jobManagementService.endSubmitFile(fileId, uploadedFile);
					jobFile.setPath(uploadedFile.getPath());
				}
				finally {
					if (jobFileStream != null) { 
						jobFileStream.close();
					}
				}
			}					
			
			if (inputPayload.getFiles().getJobFile().size() > 1) {
				ArrayOfSubmitFileInfo submitFileInfos = jobManagementService.submitFiles(settings.getRepositoryName(), settings.getLocationName(), inputPayload, true, false);
				
				if (submitFileInfos == null || submitFileInfos.getSubmitFileInfo().size() == 0) {
					throw new AdlibNotConfiguredException(String.format("Could not submitFiles (double check that your Repository (%s) and Location (%s) are setup correctly in Adlib"
							, settings.getRepositoryName()
							, settings.getLocationName()));
				}
				submitFileInfo = submitFileInfos.getSubmitFileInfo().get(0);
				fileId = submitFileInfo.getFileId();	
			}					
			
			jobManagementService.beginOrchestration(fileId, "");
			
			return fileId;
			
		} catch (Exception e) {
			if (fileId > 0) {
				throw new AdlibException(String.format("Error streaming to Adlib for fileID %s", fileId), fileId, e);
			}
			throw e;			
		}
	}
	
	
	/**
	 * When there is a single output rendition, this will download from Adlib and return as byte array
	 * @param processedJob
	 * @param renditionType - should be PDF in most cases
	 * @param outputFileNameOverride
	 * @throws JobManagementBeginGetLibraryFileExceptionDetailFaultFaultMessage 
	 * @throws JobManagementGetLibraryFilePartExceptionDetailFaultFaultMessage 
	 * @throws IOException 
	 * @throws IllegalArgumentException
	 * @return byte[] - The resulting file as a byte array
	 */
	public byte[] downloadLibraryRendition(
			ProcessedJobResponse processedJob,
			RenditionType renditionType) 
					throws JobManagementBeginGetLibraryFileExceptionDetailFaultFaultMessage, 
					JobManagementGetLibraryFilePartExceptionDetailFaultFaultMessage, 
					IOException,
					IllegalArgumentException {
		JobManagement jobManagementService = this.getJobManagement();
		ObjectFactory factory = new ObjectFactory();
			
		Object libraryId = Common.getMetadataValueByName(processedJob.getOutputPayload().getMetadata(), "LibraryId", null);
		try {
			UUID.fromString(libraryId.toString()); 
		} catch (Exception e) {
			libraryId = null;
		}
		if (libraryId == null) {
			throw new IllegalArgumentException("Required output payload metadata not found: LibraryId");
		}		
		
		// set download options
		GetJobFileOptions options = new GetJobFileOptions();
		options.setFilePartSizeBytes(factory.createGetJobFileOptionsFilePartSizeBytes(settings.getStreamingBufferSizeBytes()));
		options.setHashAlgorithm(settings.getHashAlgorithm());
		// set BeginGetLibraryFile
		StartGetLibraryFileResponse startGetLibraryFileResponse = jobManagementService.beginGetLibraryFile(libraryId.toString(), processedJob.getFileId(), renditionType, options);
		long totalBytes = startGetLibraryFileResponse.getFileSize();
		long bytesRead = 0;
						
		String filePartContext = startGetLibraryFileResponse.getLibraryFilePartContext().getValue();
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		byte[] output = null;
		try {
			while (bytesRead < totalBytes) {
				// get next chunk
				GetLibraryFilePartResponse2 libraryFilePartResponse = jobManagementService.getLibraryFilePart(filePartContext);
				long filePartSize = libraryFilePartResponse.getFilePart().getValue().length;						
				
				outputStream.write(libraryFilePartResponse.getFilePart().getValue());
									
				bytesRead += filePartSize;						
				filePartContext = libraryFilePartResponse.getLibraryFilePartContext().getValue();
			}
			output = outputStream.toByteArray();
			
		} finally {
			outputStream.close();
		}	
		
		if (settings.getHashAlgorithm() != HashAlgorithm.NONE) {
			// compare both hashes
			boolean doHashesMatch = Arrays.equals(output, startGetLibraryFileResponse.getHashValue().getValue());
			if (!doHashesMatch) {
				throw new IOException(String.format("File content hashes do not match after download for Job %s", processedJob.getFileId()));
			}
		}		
		return output;
	}
	

	/**
	 * Downloads the library rendition for the processed job (after calling streamJob and waitForJobToProcess)
	 * @param processedJob
	 * @param downloadDirectory base directory to download to.  If file names are not unique, indexed sub-folder will be created
	 * @param renditionType normally this is PDF except in custom cases
	 * @param fileNamingMode the naming convention to use when downloading files.  Anything other than NONE requires that "OriginalFileName" JobFile metadata exists.
	 * @param outputFileNameOverride only set this to a value if you do not want to use OriginalFileName as base
	 * @return list of downloaded files
	 * @apiNote If JobFile has metadata called OriginalFileName, this will be used to set the output file name (appending output extension)
	 * @throws Exception
	 */
	public List<File> downloadLibraryRendition(
			ProcessedJobResponse processedJob, 
			File downloadDirectory, 
			RenditionType renditionType, 
			DownloadFileNamingMode fileNamingMode, 
			String outputFileNameOverride) throws Exception {
		
		List<File> downloadedFiles = new ArrayList<File>();
		JobManagement jobManagementService = this.getJobManagement();
		ObjectFactory factory = new ObjectFactory();
			
		Object libraryId = Common.getMetadataValueByName(processedJob.getOutputPayload().getMetadata(), "LibraryId", null);
		try {
			UUID.fromString(libraryId.toString()); 
		} catch (Exception e) {
			libraryId = null;
		}
		if (libraryId == null) {
			throw new Exception("Required output payload metadata not found: LibraryId");
		}		
			
		Object originalFileName = Common.getMetadataValueByName(processedJob.getOutputPayload().getMetadata(), "OriginalFileName", null);
		String outputFileName = null;
		String workFileName = null;
		if (fileNamingMode == DownloadFileNamingMode.OVERRIDE && outputFileNameOverride != null) {
			outputFileName = outputFileNameOverride;
		}
		if (originalFileName == null && fileNamingMode != DownloadFileNamingMode.OVERRIDE && processedJob.getOutputPayload().getFiles().getJobFile().size() > 0) {
			originalFileName = Common.getMetadataValueByName(processedJob.getOutputPayload().getFiles().getJobFile().get(0).getMetadata(), "OriginalFileName", null);
			if (originalFileName == null) {
				File workFile = new File(processedJob.getOutputPayload().getFiles().getJobFile().get(0).getPath());
				originalFileName = workFile.getName();
				workFileName = workFile.getName();
				int i = originalFileName.toString().lastIndexOf('.');
				if (i > 0) {
					originalFileName = originalFileName.toString().substring(0, i);
				}
			}
		}
		if (originalFileName == null && outputFileName == null) {
			throw new Exception("Cannot continue without either using OrginalFileName metadata, having an output payload file, or having a valid override");
		}
		
		if (fileNamingMode != DownloadFileNamingMode.NONE) {
			// Get original file name that was stored from input metadata
			
			if (originalFileName != null) {
				String outputExtension = "";
				if (workFileName != null) {
					int i = workFileName.lastIndexOf('.');
					if (i > 0) {
						outputExtension = "." + workFileName.substring(i+1);
					}
				}
				if (outputExtension.length() == 0) {
					switch (renditionType) {
					case TEXT:
						outputExtension = ".txt";
						break;
					case THUMBNAIL:
						outputExtension = ".jpg";
						break;
					default:
						outputExtension = ".pdf";
						break;
					}
				}
				
				String originalFileNoExtension = originalFileName.toString();
				int i = originalFileName.toString().lastIndexOf('.');
				if (i > 0) {
					originalFileNoExtension = originalFileName.toString().substring(0, i);
				}
				switch (fileNamingMode) {
				case APPEND_OUTPUT_EXTENSION:
					outputFileName = originalFileName.toString() + outputExtension;
					break;
				case REPLACE_EXTENSION:
					outputFileName = originalFileNoExtension + outputExtension;
					break;
				default:					
					break;
				}
			}
		}	
		if (outputFileName == null) {
			throw new Exception("No output file name could be detected/defined");
		}
		
		File localFile = new File(downloadDirectory, outputFileName);
		if (localFile.exists()) {
			localFile.delete();
		}
		
		// set download options
		GetJobFileOptions options = new GetJobFileOptions();
		options.setFilePartSizeBytes(factory.createGetJobFileOptionsFilePartSizeBytes(settings.getStreamingBufferSizeBytes()));
		options.setHashAlgorithm(settings.getHashAlgorithm());
		// set BeginGetLibraryFile
		StartGetLibraryFileResponse startGetLibraryFileResponse = jobManagementService.beginGetLibraryFile(libraryId.toString(), processedJob.getFileId(), renditionType, options);
		long totalBytes = startGetLibraryFileResponse.getFileSize();
		FileOutputStream outputFileStream = new FileOutputStream(localFile);
		long bytesRead = 0;
		try {					
			String filePartContext = startGetLibraryFileResponse.getLibraryFilePartContext().getValue();
			while (bytesRead < totalBytes) {
				// get next chunk
				GetLibraryFilePartResponse2 libraryFilePartResponse = jobManagementService.getLibraryFilePart(filePartContext);
				long filePartSize = libraryFilePartResponse.getFilePart().getValue().length;						
				outputFileStream.write(libraryFilePartResponse.getFilePart().getValue());						
				bytesRead += filePartSize;						
				filePartContext = libraryFilePartResponse.getLibraryFilePartContext().getValue();
			}
		}
		finally {
			if (outputFileStream != null) { 
				outputFileStream.close();
			}
		}			
		if (settings.getHashAlgorithm() != HashAlgorithm.NONE) {
			byte[] localFileHash = getFileHash(localFile);
			// compare both hashes
			boolean doHashesMatch = Arrays.equals(localFileHash, startGetLibraryFileResponse.getHashValue().getValue());
			if (!doHashesMatch) {
				throw new Exception(String.format("File hashes do not match after download (%s) for Job %s", outputFileName, processedJob.getFileId()));
			}
		}
		
		downloadedFiles.add(localFile);
		
		return downloadedFiles;
	}
	
	
	/**
	 * Downloads the files for the processed job that are contained in the OutputPayload (after calling streamJob and waitForJobToProcess)
	 * @param processedJob
	 * @param downloadDirectory base directory to download to.  If file names are not unique, indexed sub-folder will be created
	 * @param fileNamingMode the naming convention to use when downloading files.  Anything other than NONE requires that "OriginalFileName" JobFile metadata exists.
	 * @return files downloaded
	 * @apiNote If JobFile has metadata called OriginalFileName, this will be used to set the output file name (appending output extension)
	 * @throws Exception
	 */
	public List<File> downloadOutputPayloadFiles(ProcessedJobResponse processedJob, File downloadDirectory, DownloadFileNamingMode fileNamingMode) throws Exception {
		
		List<File> downloadedFiles = new ArrayList<File>();
		JobManagement jobManagementService = this.getJobManagement();
		ObjectFactory factory = new ObjectFactory();
		
		Integer fileIndex = 0;
		for (JobFile jobFile : processedJob.getOutputPayload().getFiles().getJobFile()) {
			
			File workFile = new File(jobFile.getPath());
			
			String outputFileName = workFile.getName();
			if (fileNamingMode != DownloadFileNamingMode.NONE) {
				// Get original file name that was stored from input metadata
				Object originalFileName = Common.getMetadataValueByName(jobFile.getMetadata(), "OriginalFileName", null);
				if (originalFileName != null) {
					String outputExtension = "";
					int i = workFile.getName().lastIndexOf('.');
					if (i > 0) {
						outputExtension = "." + workFile.getName().substring(i+1);
					}
					String originalFileNoExtension = "";
					i = originalFileName.toString().lastIndexOf('.');
					if (i > 0) {
						originalFileNoExtension = originalFileName.toString().substring(0, i);
					}
					switch (fileNamingMode) {
					case APPEND_OUTPUT_EXTENSION:
						outputFileName = originalFileName.toString() + outputExtension;
						break;
					case REPLACE_EXTENSION:
						outputFileName = originalFileNoExtension + outputExtension;
						break;
					case NONE:
						default:
							// do nothing
						break;
					}
				}
			}				
			
			File localFile = new File(downloadDirectory, outputFileName);
			if (localFile.exists()) {
				localFile = new File(new File(downloadDirectory, fileIndex.toString()), outputFileName);
				localFile.mkdir();
			}
			// set download options
			GetJobFileOptions options = new GetJobFileOptions();
			options.setFilePartSizeBytes(factory.createGetJobFileOptionsFilePartSizeBytes(settings.getStreamingBufferSizeBytes()));
			options.setHashAlgorithm(settings.getHashAlgorithm());
			// set beginJobFileProcess
			StartGetJobFileResponse startGetJobFileResponse = jobManagementService.beginGetJobFile(processedJob.getLastJobId(), jobFile.getPath(), options);
			long totalBytes = startGetJobFileResponse.getFileSize();
			FileOutputStream outputFileStream = new FileOutputStream(localFile);
			long bytesRead = 0;
			try {					
				String filePartContext = startGetJobFileResponse.getFilePartContext().getValue();
				while (bytesRead < totalBytes) {
					// get next chunk
					GetJobFilePartResponse2 jobFilePartResponse = jobManagementService.getJobFilePart(filePartContext);
					long filePartSize = jobFilePartResponse.getFilePart().getValue().length;						
					outputFileStream.write(jobFilePartResponse.getFilePart().getValue());						
					bytesRead += filePartSize;						
					filePartContext = jobFilePartResponse.getFilePartContext().getValue();
				}
			}
			finally {
				if (outputFileStream != null) { 
					outputFileStream.close();
				}
			}			
			if (settings.getHashAlgorithm() != HashAlgorithm.NONE) {
				byte[] localFileHash = getFileHash(localFile);
				// compare both hashes
				boolean doHashesMatch = Arrays.equals(localFileHash, startGetJobFileResponse.getHashValue().getValue());
				if (!doHashesMatch) {
					throw new Exception(String.format("File hashes do not match after download (%s) for Job %s", workFile.getName(), processedJob.getLastJobId()));
				}
			}				
			downloadedFiles.add(localFile);
			fileIndex++;
		}	
		return downloadedFiles;
	}
	

	/**
	 * @param localFile
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 */
	private byte[] getFileHash(File localFile) throws IOException, NoSuchAlgorithmException {
		byte[] fileContent = Files.readAllBytes(localFile.toPath());
		return getFileContentHash(fileContent);
	}	
	
	/**
	 * @param fileContent
	 * @return
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 */
	private byte[] getFileContentHash(byte[] fileContent) throws IOException, NoSuchAlgorithmException {
		MessageDigest digest = MessageDigest.getInstance(settings.getHashAlgorithm().toString().replace("_", "-"));
		byte[] hashValue = digest.digest(fileContent);
		return hashValue;
	}	
	
	
	/**
	 * Will wait for job to be processed
	 * @param fileId
	 * @param timeout
	 * @param pollingInterval
	 * @return ProcessedJobResponse
	 * @throws TimeoutException, Exception
	 */
	public ProcessedJobResponse waitForJobToProcess(long fileId, Duration timeout, Duration pollingInterval) throws TimeoutException, Exception {
		if (timeout == null) {
			timeout = settings.getDefaultTimeout();
		}
		if (pollingInterval == null) {
			pollingInterval = settings.getDefaultPollingInterval();
		}
		
		OrchestrationFileStatusResponse orchestrationFileStatusResponse = Common.waitForJobToProcess(fileId, this.getJobManagement(), timeout, pollingInterval);
		
		JobDetails jobDetails = this.getJobManagement().getJobDetails(orchestrationFileStatusResponse.getJobId());
		ProcessedJobResponse processedJobResponse = getProcessedJobResponse(orchestrationFileStatusResponse.getFileId(), jobDetails);
		
		return processedJobResponse;		
	}
	
	/**
	 * Get orchestration status (from FileId)
	 * @param fileId
	 * @return
	 * @throws Exception
	 */
	public OrchestrationFileStatusResponse getOrchestrationStatus(long fileId) throws Exception {
		
		ArrayOflong fileIds = new ArrayOflong();
		fileIds.getLong().add(fileId);		
		
		ArrayOfOrchestrationFileStatusResponse response = jobManagement.getOrchestrationFileStatus(fileIds);
		
		if (response == null || response.getOrchestrationFileStatusResponse().size() == 0) {
			return null;
		}
		
		return response.getOrchestrationFileStatusResponse().get(0);
	}	
	
	/**
	 * Get list of orchestration status (from FileIds)
	 * @param fileIds
	 * @return
	 * @throws Exception
	 */
	public List<OrchestrationFileStatusResponse> getOrchestrationsStatus(List<Long> fileIds) throws Exception {
		
		ArrayOflong fileIdsList = new ArrayOflong();
		fileIdsList.getLong().addAll(fileIds);
		
		ArrayOfOrchestrationFileStatusResponse response = jobManagement.getOrchestrationFileStatus(fileIdsList);
		
		if (response == null || response.getOrchestrationFileStatusResponse().size() == 0) {
			return null;
		}
		
		return response.getOrchestrationFileStatusResponse();
	}
	
	
	/**
	 * Will wait for all jobs to be processed
	 * @param fileIds
	 * @param timeout
	 * @param pollingInterval
	 * @return
	 * @throws Exception
	 */
	public List<ProcessedJobResponse> waitForJobsToProcess(ArrayOflong fileIds, Duration timeout, Duration pollingInterval) throws AdlibTimeoutException, Exception {
		
		List<ProcessedJobResponse> jobResponses = new ArrayList<ProcessedJobResponse>();
		if (timeout == null) {
			timeout = settings.getDefaultTimeout();
		}
		if (pollingInterval == null) {
			pollingInterval = settings.getDefaultPollingInterval();
		}
		JobManagement jobManagementService = this.getJobManagement();
		LocalDateTime startTime = LocalDateTime.now();
		
		ArrayOflong remainingJobIds = new ArrayOflong();
		remainingJobIds.getLong().addAll(fileIds.getLong());
		
		while (jobResponses.size() != fileIds.getLong().size()) {
			Duration duration = Duration.between(LocalDateTime.now(), startTime);			
			long durationInMilliseconds = Math.abs(duration.toMillis());
			if (durationInMilliseconds > timeout.toMillis()) {
				throw new AdlibTimeoutException(
						String.format("Operation timed out (still %s incomplete) after %s seconds.", remainingJobIds.getLong().size(), timeout.toMillis() / 1000.0),
						0);
			}	
				
			ArrayOfOrchestrationFileStatusResponse orchestrationFileStatusResponses = jobManagementService.getOrchestrationFileStatus(remainingJobIds);

			if (orchestrationFileStatusResponses != null) {
				for (OrchestrationFileStatusResponse orchestrationFileStatusResponse : orchestrationFileStatusResponses.getOrchestrationFileStatusResponse()) {
					if (Common.isOrchestrationComplete(orchestrationFileStatusResponse)) {
						JobDetails jobDetails = jobManagementService.getJobDetails(orchestrationFileStatusResponse.getJobId());
						ProcessedJobResponse processedJobResponse = getProcessedJobResponse(orchestrationFileStatusResponse.getFileId(), jobDetails);
						jobResponses.add(processedJobResponse);
						// remove from remaining list
						remainingJobIds.getLong().removeIf(j -> (j == orchestrationFileStatusResponse.getFileId()));
					}
				}
			}
			
			if (jobResponses.size() != fileIds.getLong().size()) {
				Thread.sleep(pollingInterval.toMillis());
			}			
		}		
		return jobResponses;
	}
	
	/**
	 * @return the jobManagement
	 */
	public JobManagement getJobManagement() {
		return jobManagement;
	}

	/**
	 * @return the settings
	 */
	public Settings getSettings() {
		return settings;
	}

	/**
	 * @param settings the settings to set
	 */
	public void setSettings(Settings settings) {
		this.settings = settings;
	}

	
}
