package com.adlibsoftware.integration;

import java.net.URL;
import java.time.Duration;

import com.adlibsoftware.authorize.AES;
import com.adlibsoftware.client.HashAlgorithm;

/**
 * Settings class used by JobManagementServiceClient
 * @author mmanley
 * @see com.adlibsoftware.integration.JobManagementServiceClient
 *
 */
public class Settings {

	private URL jobManagementServiceWsdlUrl;
	
	private URL tokenServiceUrl;
	
	private String tokenServiceUsername;
	
	private String tokenServiceEncryptedPassword;
	
	private String repositoryName;
	
	private String locationName;
	
	private Duration defaultTimeout;
	
	private Duration defaultPollingInterval;
	
	private int streamingBufferSizeBytes;
	
	private int tokenRefreshRateHours;
	
	private boolean downloadJobPayloadOutputFiles;
	
	private boolean isStreaming;
	
	private boolean isSynchronous;
	
	private HashAlgorithm hashAlgorithm;
	
	public Settings() {
		setDefaultTimeout(Duration.ofMinutes(5));
		setDefaultPollingInterval(Duration.ofMillis(500));
		setStreamingBufferSizeBytes(1024 * 1024);
		setTokenRefreshRateHours(24);
		setHashAlgorithm(HashAlgorithm.NONE);
		setLocationName("MonitoredLocation");
	}

	/**
	 * @return the jobManagementServiceUrl
	 */
	public URL getJobManagementServiceWsdlUrl() {
		return jobManagementServiceWsdlUrl;
	}

	/**
	 * @param jobManagementServiceUrl the jobManagementServiceUrl to set, example https://serverName.domainName.com:55583/Adlib/Services/JobManagement.svc?wsd
	 */
	public void setJobManagementServiceWsdlUrl(URL jobManagementServiceUrl) {
		this.jobManagementServiceWsdlUrl = jobManagementServiceUrl;
	}

	/**
	 * @return the tokenServiceUrl
	 */
	public URL getTokenServiceUrl() {
		return tokenServiceUrl;
	}

	/**
	 * 
	 * @param tokenServiceUrl the tokenServiceUrl to set, example https://serverName.domainName.com:8088/connect/token
	 * @apiNote 
	 */
	public void setTokenServiceUrl(URL tokenServiceUrl) {
		this.tokenServiceUrl = tokenServiceUrl;
	}

	/**
	 * @return the tokenServiceUsername
	 */
	public String getTokenServiceUsername() {
		return tokenServiceUsername;
	}

	/**
	 * @param tokenServiceUsername the tokenServiceUsername to set
	 */
	public void setTokenServiceUsername(String tokenServiceUsername) {
		this.tokenServiceUsername = tokenServiceUsername;
	}

	/**
	 * @return the tokenServiceEncryptedPassword
	 */
	public String getTokenServiceEncryptedPassword() {
		return tokenServiceEncryptedPassword;
	}

	/**
	 * @param tokenServiceEncryptedPassword the tokenServiceEncryptedPassword to set
	 */
	public void setTokenServiceEncryptedPassword(String tokenServiceEncryptedPassword) {
		this.tokenServiceEncryptedPassword = tokenServiceEncryptedPassword;
	}

	/**
	 * @return the repositoryName
	 */
	public String getRepositoryName() {
		return repositoryName;
	}

	/**
	 * @param locationName the locationName to set
	 * @apiNote This must exist and be configured in Adlib Console
	 */
	public void setLocationName(String locationName) {
		this.locationName = locationName;
	}
	
	/**
	 * @return the locationName
	 */
	public String getLocationName() {
		return locationName;
	}

	/**
	 * @param repositoryName the repositoryName to set
	 * @apiNote This must exist and be configured in Adlib Vision Console
	 */
	public void setRepositoryName(String repositoryName) {
		this.repositoryName = repositoryName;
	}

	/**
	 * @return the defaultTimeout
	 */
	public Duration getDefaultTimeout() {
		return defaultTimeout;
	}

	/**
	 * @param defaultTimeout the defaultTimeout to set
	 * @apiNote This will be used when waiting for job or job batches to complete
	 */
	public void setDefaultTimeout(Duration defaultTimeout) {
		this.defaultTimeout = defaultTimeout;
	}

	/**
	 * @return the defaultPollingInterval
	 */
	public Duration getDefaultPollingInterval() {
		return defaultPollingInterval;
	}

	/**
	 * @param defaultPollingInterval the defaultPollingInterval to set
	 * @apiNote Determines the polling interval between calls to Adlib for Job Status.  Default is 500 milliseconds
	 */
	public void setDefaultPollingInterval(Duration defaultPollingInterval) {
		this.defaultPollingInterval = defaultPollingInterval;
	}

	/**
	 * @return the streamingBufferSizeBytes
	 */
	public int getStreamingBufferSizeBytes() {
		return streamingBufferSizeBytes;
	}

	/**
	 * @param streamingBufferSizeBytes the streamingBufferSizeBytes to set
	 * @apiNote When streaming files to Adlib, this is the chunk size.  Defaults to 1 MB
	 */
	public void setStreamingBufferSizeBytes(int streamingBufferSizeBytes) {
		this.streamingBufferSizeBytes = streamingBufferSizeBytes;
	}

	/**
	 * @return the tokenRefreshRateHours
	 */
	public int getTokenRefreshRateHours() {
		return tokenRefreshRateHours;
	}

	/**
	 * @param tokenRefreshRateHours the tokenRefreshRateHours to set.
	 * @apiNote This should match or be less than the defined Adlib server token lifetime.  Default in Adlib is 29 hours, and this default is 24 hours.
	 */
	public void setTokenRefreshRateHours(int tokenRefreshRateHours) {
		this.tokenRefreshRateHours = tokenRefreshRateHours;
	}

	/**
	 * @return the hashAlgorithm
	 */
	public HashAlgorithm getHashAlgorithm() {
		return hashAlgorithm;
	}

	/**
	 * @param hashAlgorithm the hashAlgorithm to set when streaming to verify uploaded/downloaded files
	 * @apiNote Turning this to something other than NONE may affect performance (due to the nature of file hashing).
	 */
	public void setHashAlgorithm(HashAlgorithm hashAlgorithm) {
		this.hashAlgorithm = hashAlgorithm;
	}

	/**
	 * @param secretKey the secretKey to set
	 */
	public void setSecretKey(String secretKey) {
		AES.setKey(secretKey);
	}

	/**
	 * @return the downloadJobPayloadOutputFiles
	 */
	public boolean isDownloadJobPayloadOutputFiles() {
		return downloadJobPayloadOutputFiles;
	}

	/**
	 * @param downloadJobPayloadOutputFiles the downloadJobPayloadOutputFiles to set
	 */
	public void setDownloadJobPayloadOutputFiles(boolean downloadJobPayloadOutputFiles) {
		this.downloadJobPayloadOutputFiles = downloadJobPayloadOutputFiles;
	}

	/**
	 * @return the isStreaming
	 */
	public boolean isStreaming() {
		return isStreaming;
	}

	/**
	 * @param isStreaming the isStreaming to set
	 */
	public void setStreaming(boolean isStreaming) {
		this.isStreaming = isStreaming;
	}

	/**
	 * @return the isSynchronous
	 */
	public boolean isSynchronous() {
		return isSynchronous;
	}

	/**
	 * @param isSynchronous the isSynchronous to set
	 */
	public void setSynchronous(boolean isSynchronous) {
		this.isSynchronous = isSynchronous;
	}
	
	
}
