/*
 * Copyright (C) 2015 jsonwebtoken.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package acscommons.io.jsonwebtoken.impl.crypto;

import acscommons.io.jsonwebtoken.SignatureAlgorithm;
import acscommons.io.jsonwebtoken.lang.Assert;
import acscommons.io.jsonwebtoken.security.SignatureException;

import java.math.BigInteger;
import java.security.Key;
import java.security.PublicKey;
import java.security.Signature;
import java.security.interfaces.ECKey;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;

public class EllipticCurveSignatureValidator extends EllipticCurveProvider implements SignatureValidator {

    private static final String EC_PUBLIC_KEY_REQD_MSG =
            "Elliptic Curve signature validation requires an ECPublicKey instance.";

    private static final String DER_ENCODING_SYS_PROPERTY_NAME =
            "acscommons.io.jsonwebtoken.impl.crypto.EllipticCurveSignatureValidator.derEncodingSupported";

    public EllipticCurveSignatureValidator(SignatureAlgorithm alg, Key key) {
        super(alg, key);
        Assert.isTrue(key instanceof ECPublicKey, EC_PUBLIC_KEY_REQD_MSG);
    }

    @Override
    public boolean isValid(byte[] data, byte[] concatSignature) {
        Signature sig = createSignatureInstance();
        PublicKey publicKey = (PublicKey) key;
        try {
            // mandated per https://datatracker.ietf.org/doc/html/rfc7518#section-3.4 :
            int requiredConcatByteLength = getSignatureByteArrayLength(alg);

            byte[] derSignature;

            if (requiredConcatByteLength != concatSignature.length) {

                /*
                 * If the expected size is not valid for JOSE, fall back to ASN.1 DER signature IFF the application
                 * is configured to do so.  This fallback is for backwards compatibility ONLY (to support tokens
                 * generated by early versions of jjwt) and backwards compatibility will be removed in a future
                 * version of this library.  This fallback is only enabled if the system property is set to 'true' due to
                 * the risk of CVE-2022-21449 attacks on early JVM versions 15, 17 and 18.
                 */
                // TODO: remove for 1.0 (DER-encoding support is not in the JWT RFCs)
                if (concatSignature[0] == 0x30 && "true".equalsIgnoreCase(System.getProperty(DER_ENCODING_SYS_PROPERTY_NAME))) {
                    derSignature = concatSignature;
                } else {
                    String msg = "Provided signature is " + byteSizeString(concatSignature.length) + " but " +
                            alg.name() + " signatures must be exactly " + byteSizeString(requiredConcatByteLength) + " per " +
                            "[RFC 7518, Section 3.4 (validation)](https://datatracker.ietf.org/doc/html/rfc7518#section-3.4).";
                    throw new SignatureException(msg);
                }
            } else {

                //guard for JVM security bug CVE-2022-21449:
                ECKey ecKey = (ECKey) publicKey; // we can cast here because of the assertions made in the constructor
                BigInteger order = ecKey.getParams().getOrder();
                BigInteger r = new BigInteger(1, Arrays.copyOfRange(concatSignature, 0, this.fieldByteLength));
                BigInteger s = new BigInteger(1, Arrays.copyOfRange(concatSignature, this.fieldByteLength, concatSignature.length));
                if (r.signum() < 1 || s.signum() < 1 || r.compareTo(order) >= 0 || s.compareTo(order) >= 0) {
                    return false;
                }

                // Convert from concat to DER encoding since
                // 1) SHAXXXWithECDSAInP1363Format algorithms are only available on >= JDK 9 and
                // 2) the SignatureAlgorithm enum JCA alg names are all SHAXXXwithECDSA (which expects DER formatting)
                derSignature = transcodeConcatToDER(concatSignature);
            }
            return doVerify(sig, publicKey, data, derSignature);
        } catch (Exception e) {
            String msg = "Unable to verify Elliptic Curve signature using configured ECPublicKey. " + e.getMessage();
            throw new SignatureException(msg, e);
        }
    }

    protected boolean doVerify(Signature sig, PublicKey publicKey, byte[] data, byte[] signature)
            throws java.security.InvalidKeyException, java.security.SignatureException {
        sig.initVerify(publicKey);
        sig.update(data);
        return sig.verify(signature);
    }

}
