/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Sets;
import java.util.Calendar;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.api.jmx.IndexStatsMBean;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.commit.AnnotatingConflictHandler;
import org.apache.jackrabbit.oak.plugins.commit.ConflictHook;
import org.apache.jackrabbit.oak.plugins.commit.ConflictValidatorProvider;
import org.apache.jackrabbit.oak.plugins.index.IndexEditorProvider;
import org.apache.jackrabbit.oak.plugins.index.IndexUpdate;
import org.apache.jackrabbit.oak.plugins.index.IndexUpdateCallback;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.plugins.memory.PropertyStates;
import org.apache.jackrabbit.oak.spi.commit.CommitHook;
import org.apache.jackrabbit.oak.spi.commit.CommitInfo;
import org.apache.jackrabbit.oak.spi.commit.CompositeHook;
import org.apache.jackrabbit.oak.spi.commit.EditorDiff;
import org.apache.jackrabbit.oak.spi.commit.EditorHook;
import org.apache.jackrabbit.oak.spi.commit.VisibleEditor;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStateDiff;
import org.apache.jackrabbit.oak.spi.state.NodeStore;
import org.apache.jackrabbit.util.ISO8601;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AsyncIndexUpdate
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(AsyncIndexUpdate.class);
    static final String ASYNC = ":async";
    private static final long DEFAULT_LIFETIME = TimeUnit.DAYS.toMillis(1000L);
    private static final CommitFailedException CONCURRENT_UPDATE = new CommitFailedException("Async", 1, "Concurrent update detected");
    private static final long ASYNC_TIMEOUT;
    private final String name;
    private final NodeStore store;
    private final IndexEditorProvider provider;
    private final String lastIndexedTo;
    private final long lifetime = DEFAULT_LIFETIME;
    private boolean failing = false;
    private final AsyncIndexStats indexStats = new AsyncIndexStats();
    private final boolean switchOnSync;
    private final Set<String> reindexedDefinitions = new HashSet<String>();
    private final IndexUpdate.MissingIndexProviderStrategy missingStrategy = new DefaultMissingIndexProviderStrategy();

    public AsyncIndexUpdate(@Nonnull String name, @Nonnull NodeStore store, @Nonnull IndexEditorProvider provider, boolean switchOnSync) {
        this.name = (String)Preconditions.checkNotNull((Object)name);
        this.lastIndexedTo = name + "-LastIndexedTo";
        this.store = (NodeStore)Preconditions.checkNotNull((Object)store);
        this.provider = (IndexEditorProvider)Preconditions.checkNotNull((Object)provider);
        this.switchOnSync = switchOnSync;
    }

    public AsyncIndexUpdate(@Nonnull String name, @Nonnull NodeStore store, @Nonnull IndexEditorProvider provider) {
        this(name, store, provider, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized void run() {
        NodeState before;
        if (this.indexStats.isPaused()) {
            return;
        }
        log.debug("Running background index task {}", (Object)this.name);
        NodeState root = this.store.getRoot();
        NodeState async = root.getChildNode(ASYNC);
        long leaseEndTime = async.getLong(this.name + "-lease");
        long currentTime = System.currentTimeMillis();
        if (leaseEndTime > currentTime) {
            log.debug("Another copy of the {} index update is already running; skipping this update. Time left for lease to expire {}s", (Object)this.name, (Object)((leaseEndTime - currentTime) / 1000L));
            return;
        }
        String beforeCheckpoint = async.getString(this.name);
        if (beforeCheckpoint != null) {
            NodeState state = this.store.retrieve(beforeCheckpoint);
            if (state == null) {
                log.warn("Failed to retrieve previously indexed checkpoint {}; re-running the initial {} index update", (Object)beforeCheckpoint, (Object)this.name);
                beforeCheckpoint = null;
                before = EmptyNodeState.MISSING_NODE;
            } else {
                if (AsyncIndexUpdate.noVisibleChanges(state, root)) {
                    log.debug("No changes since last checkpoint; skipping the {} index update", (Object)this.name);
                    return;
                }
                before = state;
            }
        } else {
            log.info("Initial {} index update", (Object)this.name);
            before = EmptyNodeState.MISSING_NODE;
        }
        String afterTime = AsyncIndexUpdate.now();
        String afterCheckpoint = this.store.checkpoint(this.lifetime);
        NodeState after = this.store.retrieve(afterCheckpoint);
        if (after == null) {
            log.debug("Unable to retrieve newly created checkpoint {}, skipping the {} index update", (Object)afterCheckpoint, (Object)this.name);
            return;
        }
        String checkpointToRelease = afterCheckpoint;
        try {
            this.updateIndex(before, beforeCheckpoint, after, afterCheckpoint, afterTime);
            if (this.failing) {
                log.info("Index update {} no longer fails", (Object)this.name);
                this.failing = false;
            }
            checkpointToRelease = beforeCheckpoint;
            this.indexStats.setReferenceCheckpoint(afterCheckpoint);
            this.indexStats.setProcessedCheckpoint("");
            this.indexStats.releaseTempCheckpoint(afterCheckpoint);
        }
        catch (CommitFailedException e) {
            if (e == CONCURRENT_UPDATE) {
                log.debug("Concurrent update detected in the {} index update", (Object)this.name);
            } else if (this.failing) {
                log.debug("The {} index update is still failing", (Object)this.name, (Object)e);
            } else {
                log.warn("The {} index update failed", (Object)this.name, (Object)e);
                this.failing = true;
            }
        }
        finally {
            if (checkpointToRelease != null && !this.store.release(checkpointToRelease)) {
                log.debug("Unable to reelase checkpoint {}", (Object)checkpointToRelease);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateIndex(NodeState before, String beforeCheckpoint, NodeState after, String afterCheckpoint, String afterTime) throws CommitFailedException {
        Stopwatch watch = Stopwatch.createStarted();
        AsyncIndexUpdate.preAsyncRunStatsStats(this.indexStats);
        AsyncUpdateCallback callback = new AsyncUpdateCallback(beforeCheckpoint, afterCheckpoint);
        try {
            NodeBuilder builder = this.store.getRoot().builder();
            IndexUpdate indexUpdate = new IndexUpdate(this.provider, this.name, after, builder, callback).withMissingProviderStrategy(this.missingStrategy);
            CommitFailedException exception = EditorDiff.process(VisibleEditor.wrap(indexUpdate), before, after);
            if (exception != null) {
                throw exception;
            }
            builder.child(ASYNC).setProperty(this.name, afterCheckpoint);
            builder.child(ASYNC).setProperty(PropertyStates.createProperty(this.lastIndexedTo, (Object)afterTime, Type.DATE));
            if (callback.isDirty() || before == EmptyNodeState.MISSING_NODE) {
                if (this.switchOnSync) {
                    this.reindexedDefinitions.addAll(indexUpdate.getReindexedDefinitions());
                } else {
                    AsyncIndexUpdate.postAsyncRunStatsStatus(this.indexStats);
                }
            } else {
                if (this.switchOnSync) {
                    log.debug("No changes detected after diff; will try to switch to synchronous updates on {}", this.reindexedDefinitions);
                    for (String path : this.reindexedDefinitions) {
                        NodeBuilder c = builder;
                        for (String p : PathUtils.elements(path)) {
                            c = c.getChildNode(p);
                        }
                        if (!c.exists() || c.getBoolean("reindex")) continue;
                        c.removeProperty("async");
                    }
                    this.reindexedDefinitions.clear();
                }
                AsyncIndexUpdate.postAsyncRunStatsStatus(this.indexStats);
            }
            this.mergeWithConcurrencyCheck(builder, beforeCheckpoint, callback.lease);
            if (indexUpdate.isReindexingPerformed()) {
                log.info("Reindexing completed for indexes: {} in {}", indexUpdate.getReindexStats(), (Object)watch);
            }
        }
        finally {
            callback.close();
        }
    }

    private void mergeWithConcurrencyCheck(NodeBuilder builder, final String checkpoint, final long lease) throws CommitFailedException {
        CommitHook concurrentUpdateCheck = new CommitHook(){

            @Override
            @Nonnull
            public NodeState processCommit(NodeState before, NodeState after, CommitInfo info) throws CommitFailedException {
                NodeState async = before.getChildNode(AsyncIndexUpdate.ASYNC);
                if (checkpoint == null || Objects.equal((Object)checkpoint, (Object)async.getString(AsyncIndexUpdate.this.name)) && lease == async.getLong(AsyncIndexUpdate.this.name + "-lease")) {
                    return after;
                }
                throw CONCURRENT_UPDATE;
            }
        };
        CompositeHook hooks = new CompositeHook(new ConflictHook(new AnnotatingConflictHandler()), new EditorHook(new ConflictValidatorProvider()), concurrentUpdateCheck);
        this.store.merge(builder, hooks, CommitInfo.EMPTY);
    }

    private static void preAsyncRunStatsStats(AsyncIndexStats stats) {
        stats.start(AsyncIndexUpdate.now());
    }

    private static void postAsyncRunStatsStatus(AsyncIndexStats stats) {
        stats.done(AsyncIndexUpdate.now());
    }

    private static String now() {
        return ISO8601.format(Calendar.getInstance());
    }

    public AsyncIndexStats getIndexStats() {
        return this.indexStats;
    }

    public boolean isFinished() {
        return this.indexStats.getStatus() == "done";
    }

    private static boolean noVisibleChanges(NodeState before, NodeState after) {
        return after.compareAgainstBaseState(before, new NodeStateDiff(){

            @Override
            public boolean propertyAdded(PropertyState after) {
                return AsyncIndexUpdate.isHidden(after.getName());
            }

            @Override
            public boolean propertyChanged(PropertyState before, PropertyState after) {
                return AsyncIndexUpdate.isHidden(after.getName());
            }

            @Override
            public boolean propertyDeleted(PropertyState before) {
                return AsyncIndexUpdate.isHidden(before.getName());
            }

            @Override
            public boolean childNodeAdded(String name, NodeState after) {
                return AsyncIndexUpdate.isHidden(name);
            }

            @Override
            public boolean childNodeChanged(String name, NodeState before, NodeState after) {
                return AsyncIndexUpdate.isHidden(name) || after.compareAgainstBaseState(before, this);
            }

            @Override
            public boolean childNodeDeleted(String name, NodeState before) {
                return AsyncIndexUpdate.isHidden(name);
            }
        });
    }

    private static boolean isHidden(String name) {
        return name.charAt(0) == ':';
    }

    public boolean isFailing() {
        return this.failing;
    }

    static {
        int value = 15;
        try {
            value = Integer.parseInt(System.getProperty("oak.async.lease.timeout", "15"));
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        ASYNC_TIMEOUT = TimeUnit.MINUTES.toMillis(value);
    }

    static class DefaultMissingIndexProviderStrategy
    extends IndexUpdate.MissingIndexProviderStrategy {
        private final Set<String> ignore = Sets.newHashSet((Object[])new String[]{"disabled"});

        DefaultMissingIndexProviderStrategy() {
        }

        @Override
        public void onMissingIndex(String type, NodeBuilder definition) throws CommitFailedException {
            if (this.ignore.contains(type)) {
                return;
            }
            throw new CommitFailedException("Async", 2, "Missing index provider detected");
        }
    }

    final class AsyncIndexStats
    implements IndexStatsMBean {
        private String start = "";
        private String done = "";
        private String status = "init";
        private String referenceCp = "";
        private String processedCp = "";
        private Set<String> tempCps = new HashSet<String>();
        private volatile boolean isPaused;
        private volatile long updates;

        AsyncIndexStats() {
        }

        public void start(String now) {
            this.status = "running";
            this.start = now;
            this.done = "";
        }

        public void done(String now) {
            this.status = "done";
            this.start = "";
            this.done = now;
        }

        @Override
        public String getStart() {
            return this.start;
        }

        @Override
        public String getDone() {
            return this.done;
        }

        @Override
        public String getStatus() {
            return this.status;
        }

        @Override
        public String getLastIndexedTime() {
            PropertyState ps = AsyncIndexUpdate.this.store.getRoot().getChildNode(AsyncIndexUpdate.ASYNC).getProperty(AsyncIndexUpdate.this.lastIndexedTo);
            return ps != null ? ps.getValue(Type.STRING) : null;
        }

        @Override
        public void pause() {
            log.debug("Pausing the async indexer");
            this.isPaused = true;
        }

        @Override
        public void resume() {
            log.debug("Resuming the async indexer");
            this.isPaused = false;
        }

        @Override
        public boolean isPaused() {
            return this.isPaused;
        }

        void setUpdates(long updates) {
            this.updates = updates;
        }

        @Override
        public long getUpdates() {
            return this.updates;
        }

        void setReferenceCheckpoint(String checkpoint) {
            this.referenceCp = checkpoint;
        }

        @Override
        public String getReferenceCheckpoint() {
            return this.referenceCp;
        }

        void setProcessedCheckpoint(String checkpoint) {
            this.processedCp = checkpoint;
        }

        @Override
        public String getProcessedCheckpoint() {
            return this.processedCp;
        }

        void setTempCheckpoints(Set<String> tempCheckpoints) {
            this.tempCps = tempCheckpoints;
        }

        void releaseTempCheckpoint(String tempCheckpoint) {
            this.tempCps.remove(tempCheckpoint);
        }

        @Override
        public String getTemporaryCheckpoints() {
            return this.tempCps.toString();
        }

        public String toString() {
            return "AsyncIndexStats [start=" + this.start + ", done=" + this.done + ", status=" + this.status + ", paused=" + this.isPaused + ", updates=" + this.updates + ", referenceCheckpoint=" + this.referenceCp + ", processedCheckpoint=" + this.processedCp + " ,tempCheckpoints=" + this.tempCps + " ]";
        }
    }

    private class AsyncUpdateCallback
    implements IndexUpdateCallback {
        private final String checkpoint;
        private long lease;
        private long updates = 0L;
        private final String leaseName;
        private final String tempCpName;

        public AsyncUpdateCallback(String checkpoint, String afterCheckpoint) throws CommitFailedException {
            long now = System.currentTimeMillis();
            this.checkpoint = checkpoint;
            this.lease = now + 2L * ASYNC_TIMEOUT;
            this.leaseName = AsyncIndexUpdate.this.name + "-lease";
            this.tempCpName = AsyncIndexUpdate.this.name + "-temp";
            NodeState root = AsyncIndexUpdate.this.store.getRoot();
            long beforeLease = root.getChildNode(AsyncIndexUpdate.ASYNC).getLong(this.leaseName);
            if (beforeLease > now) {
                throw CONCURRENT_UPDATE;
            }
            NodeBuilder builder = root.builder();
            NodeBuilder async = builder.child(AsyncIndexUpdate.ASYNC);
            async.setProperty(this.leaseName, this.lease);
            this.updateTempCheckpoints(async, checkpoint, afterCheckpoint);
            AsyncIndexUpdate.this.mergeWithConcurrencyCheck(builder, checkpoint, beforeLease);
            AsyncIndexUpdate.this.indexStats.setUpdates(this.updates);
        }

        private void updateTempCheckpoints(NodeBuilder async, String checkpoint, String afterCheckpoint) {
            AsyncIndexUpdate.this.indexStats.setReferenceCheckpoint(checkpoint);
            AsyncIndexUpdate.this.indexStats.setProcessedCheckpoint(afterCheckpoint);
            HashSet temps = Sets.newHashSet();
            for (String cp : this.getStrings(async, this.tempCpName)) {
                if (cp.equals(checkpoint)) {
                    temps.add(cp);
                    continue;
                }
                boolean released = AsyncIndexUpdate.this.store.release(cp);
                log.debug("Releasing temporary checkpoint {}: {}", (Object)cp, (Object)released);
                if (released) continue;
                temps.add(cp);
            }
            temps.add(afterCheckpoint);
            async.setProperty(this.tempCpName, temps, Type.STRINGS);
            AsyncIndexUpdate.this.indexStats.setTempCheckpoints(temps);
        }

        private Iterable<String> getStrings(NodeBuilder b, String p) {
            PropertyState ps = b.getProperty(p);
            if (ps != null) {
                return ps.getValue(Type.STRINGS);
            }
            return Sets.newHashSet();
        }

        boolean isDirty() {
            return this.updates > 0L;
        }

        void close() throws CommitFailedException {
            NodeBuilder builder = AsyncIndexUpdate.this.store.getRoot().builder();
            NodeBuilder async = builder.child(AsyncIndexUpdate.ASYNC);
            async.removeProperty(this.leaseName);
            AsyncIndexUpdate.this.mergeWithConcurrencyCheck(builder, async.getString(AsyncIndexUpdate.this.name), this.lease);
        }

        @Override
        public void indexUpdate() throws CommitFailedException {
            ++this.updates;
            if (this.updates % 100L == 0L) {
                AsyncIndexUpdate.this.indexStats.setUpdates(this.updates);
                long now = System.currentTimeMillis();
                if (now + ASYNC_TIMEOUT > this.lease) {
                    long newLease = now + 2L * ASYNC_TIMEOUT;
                    NodeBuilder builder = AsyncIndexUpdate.this.store.getRoot().builder();
                    builder.child(AsyncIndexUpdate.ASYNC).setProperty(this.leaseName, newLease);
                    AsyncIndexUpdate.this.mergeWithConcurrencyCheck(builder, this.checkpoint, this.lease);
                    this.lease = newLease;
                }
            }
        }
    }
}

