/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexDefinition;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.NoLockFactory;

class OakDirectory
extends Directory {
    static final String PROP_BLOB_SIZE = "blobSize";
    protected final NodeBuilder directoryBuilder;
    private final IndexDefinition definition;
    private LockFactory lockFactory = NoLockFactory.getNoLockFactory();
    static final int DEFAULT_BLOB_SIZE = 32768;

    public OakDirectory(NodeBuilder directoryBuilder, IndexDefinition definition) {
        this.directoryBuilder = directoryBuilder;
        this.definition = definition;
    }

    public String[] listAll() throws IOException {
        return (String[])Iterables.toArray(this.directoryBuilder.getChildNodeNames(), String.class);
    }

    public boolean fileExists(String name) throws IOException {
        return this.directoryBuilder.hasChildNode(name);
    }

    public void deleteFile(String name) throws IOException {
        this.directoryBuilder.getChildNode(name).remove();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long fileLength(String name) throws IOException {
        NodeBuilder file = this.directoryBuilder.getChildNode(name);
        OakIndexInput input = new OakIndexInput(name, file);
        try {
            long l = input.length();
            return l;
        }
        finally {
            input.close();
        }
    }

    public IndexOutput createOutput(String name, IOContext context) throws IOException {
        NodeBuilder file;
        if (!this.directoryBuilder.hasChildNode(name)) {
            file = this.directoryBuilder.child(name);
            file.setProperty(PROP_BLOB_SIZE, this.definition.getBlobSize());
        } else {
            file = this.directoryBuilder.child(name);
        }
        return new OakIndexOutput(name, file);
    }

    public IndexInput openInput(String name, IOContext context) throws IOException {
        NodeBuilder file = this.directoryBuilder.getChildNode(name);
        if (file.exists()) {
            return new OakIndexInput(name, file);
        }
        throw new FileNotFoundException(name);
    }

    public Lock makeLock(String name) {
        return this.lockFactory.makeLock(name);
    }

    public void clearLock(String name) throws IOException {
        this.lockFactory.clearLock(name);
    }

    public void sync(Collection<String> names) throws IOException {
    }

    public void close() throws IOException {
    }

    public void setLockFactory(LockFactory lockFactory) throws IOException {
        this.lockFactory = lockFactory;
    }

    public LockFactory getLockFactory() {
        return this.lockFactory;
    }

    private final class OakIndexOutput
    extends IndexOutput {
        private final OakIndexFile file;

        public OakIndexOutput(String name, NodeBuilder file) throws IOException {
            this.file = new OakIndexFile(name, file);
        }

        public long length() {
            return this.file.length;
        }

        public long getFilePointer() {
            return this.file.position;
        }

        public void seek(long pos) throws IOException {
            this.file.seek(pos);
        }

        public void writeBytes(byte[] b, int offset, int length) throws IOException {
            this.file.writeBytes(b, offset, length);
        }

        public void writeByte(byte b) throws IOException {
            this.writeBytes(new byte[]{b}, 0, 1);
        }

        public void flush() throws IOException {
            this.file.flush();
        }

        public void close() throws IOException {
            this.flush();
        }
    }

    private static class OakIndexInput
    extends IndexInput {
        private final OakIndexFile file;

        public OakIndexInput(String name, NodeBuilder file) {
            super(name);
            this.file = new OakIndexFile(name, file);
        }

        private OakIndexInput(OakIndexInput that) {
            super(that.toString());
            this.file = new OakIndexFile(that.file);
        }

        public OakIndexInput clone() {
            return new OakIndexInput(this);
        }

        public void readBytes(byte[] b, int o, int n) throws IOException {
            this.file.readBytes(b, o, n);
        }

        public byte readByte() throws IOException {
            byte[] b = new byte[1];
            this.readBytes(b, 0, 1);
            return b[0];
        }

        public void seek(long pos) throws IOException {
            this.file.seek(pos);
        }

        public long length() {
            return this.file.length;
        }

        public long getFilePointer() {
            return this.file.position;
        }

        public void close() {
        }
    }

    private static class OakIndexFile {
        private final String name;
        private final NodeBuilder file;
        private final int blobSize;
        private long position = 0L;
        private long length;
        private final List<Blob> data;
        private boolean dataModified = false;
        private int index = -1;
        private final byte[] blob;
        private boolean blobModified = false;

        public OakIndexFile(String name, NodeBuilder file) {
            this.name = name;
            this.file = file;
            this.blobSize = OakIndexFile.determineBlobSize(file);
            this.blob = new byte[this.blobSize];
            PropertyState property = file.getProperty("jcr:data");
            this.data = property != null && property.getType() == Type.BINARIES ? Lists.newArrayList(property.getValue(Type.BINARIES)) : Lists.newArrayList();
            this.length = (long)this.data.size() * (long)this.blobSize;
            if (!this.data.isEmpty()) {
                Blob last = this.data.get(this.data.size() - 1);
                this.length -= (long)this.blobSize - last.length();
            }
        }

        private OakIndexFile(OakIndexFile that) {
            this.name = that.name;
            this.file = that.file;
            this.blobSize = that.blobSize;
            this.blob = new byte[this.blobSize];
            this.position = that.position;
            this.length = that.length;
            this.data = Lists.newArrayList(that.data);
            this.dataModified = that.dataModified;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void loadBlob(int i) throws IOException {
            Preconditions.checkElementIndex((int)i, (int)this.data.size());
            if (this.index != i) {
                this.flushBlob();
                Preconditions.checkState((!this.blobModified ? 1 : 0) != 0);
                int n = (int)Math.min((long)this.blobSize, this.length - (long)(i * this.blobSize));
                InputStream stream = this.data.get(i).getNewStream();
                try {
                    ByteStreams.readFully((InputStream)stream, (byte[])this.blob, (int)0, (int)n);
                }
                finally {
                    stream.close();
                }
                this.index = i;
            }
        }

        private void flushBlob() throws IOException {
            if (this.blobModified) {
                int n = (int)Math.min((long)this.blobSize, this.length - (long)(this.index * this.blobSize));
                Blob b = this.file.createBlob(new ByteArrayInputStream(this.blob, 0, n));
                if (this.index < this.data.size()) {
                    this.data.set(this.index, b);
                } else {
                    Preconditions.checkState((this.index == this.data.size() ? 1 : 0) != 0);
                    this.data.add(b);
                }
                this.dataModified = true;
                this.blobModified = false;
            }
        }

        public void seek(long pos) throws IOException {
            if (pos < 0L || pos > this.length) {
                throw new IOException("Invalid seek request");
            }
            this.position = pos;
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            Preconditions.checkPositionIndexes((int)offset, (int)(offset + len), (int)((byte[])Preconditions.checkNotNull((Object)b)).length);
            if (len < 0 || this.position + (long)len > this.length) {
                String msg = String.format("Invalid byte range request [%s] : position : %d, length : %d, len : %d", this.name, this.position, this.length, len);
                throw new IOException(msg);
            }
            int i = (int)(this.position / (long)this.blobSize);
            int o = (int)(this.position % (long)this.blobSize);
            while (len > 0) {
                this.loadBlob(i);
                int l = Math.min(len, this.blobSize - o);
                System.arraycopy(this.blob, o, b, offset, l);
                offset += l;
                len -= l;
                this.position += (long)l;
                ++i;
                o = 0;
            }
        }

        public void writeBytes(byte[] b, int offset, int len) throws IOException {
            int i = (int)(this.position / (long)this.blobSize);
            int o = (int)(this.position % (long)this.blobSize);
            while (len > 0) {
                int l = Math.min(len, this.blobSize - o);
                if (this.index != i) {
                    if (o > 0 || l < this.blobSize && this.position + (long)l < this.length) {
                        this.loadBlob(i);
                    } else {
                        this.flushBlob();
                        this.index = i;
                    }
                }
                System.arraycopy(b, offset, this.blob, o, l);
                this.blobModified = true;
                offset += l;
                len -= l;
                this.position += (long)l;
                this.length = Math.max(this.length, this.position);
                ++i;
                o = 0;
            }
        }

        private static int determineBlobSize(NodeBuilder file) {
            if (file.hasProperty(OakDirectory.PROP_BLOB_SIZE)) {
                return Ints.checkedCast((long)file.getProperty(OakDirectory.PROP_BLOB_SIZE).getValue(Type.LONG));
            }
            return 32768;
        }

        public void flush() throws IOException {
            this.flushBlob();
            if (this.dataModified) {
                this.file.setProperty("jcr:lastModified", System.currentTimeMillis());
                this.file.setProperty("jcr:data", this.data, Type.BINARIES);
                this.dataModified = false;
            }
        }

        public String toString() {
            return this.name;
        }
    }
}

