/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment.file;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.BlobStoreBlob;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.plugins.segment.CompactionMap;
import org.apache.jackrabbit.oak.plugins.segment.Compactor;
import org.apache.jackrabbit.oak.plugins.segment.RecordId;
import org.apache.jackrabbit.oak.plugins.segment.Segment;
import org.apache.jackrabbit.oak.plugins.segment.SegmentId;
import org.apache.jackrabbit.oak.plugins.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.plugins.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.plugins.segment.SegmentStore;
import org.apache.jackrabbit.oak.plugins.segment.SegmentTracker;
import org.apache.jackrabbit.oak.plugins.segment.SegmentVersion;
import org.apache.jackrabbit.oak.plugins.segment.SegmentWriter;
import org.apache.jackrabbit.oak.plugins.segment.compaction.CompactionStrategy;
import org.apache.jackrabbit.oak.plugins.segment.file.BackgroundThread;
import org.apache.jackrabbit.oak.plugins.segment.file.CompactionGainEstimate;
import org.apache.jackrabbit.oak.plugins.segment.file.TarReader;
import org.apache.jackrabbit.oak.plugins.segment.file.TarWriter;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.gc.GCMonitor;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
implements SegmentStore {
    private static final Logger log = LoggerFactory.getLogger(FileStore.class);
    private static final int MB = 0x100000;
    private static final Pattern FILE_NAME_PATTERN = Pattern.compile("(data|bulk)((0|[1-9][0-9]*)[0-9]{4})([a-z])?.tar");
    private static final String FILE_NAME_FORMAT = "data%05d%s.tar";
    private static final String JOURNAL_FILE_NAME = "journal.log";
    private static final String LOCK_FILE_NAME = "repo.lock";
    static final boolean MEMORY_MAPPING_DEFAULT = "64".equals(System.getProperty("sun.arch.data.model", "32"));
    private final SegmentTracker tracker;
    private final File directory;
    private final BlobStore blobStore;
    private final int maxFileSize;
    private final boolean memoryMapping;
    private volatile List<TarReader> readers;
    private int writeNumber;
    private File writeFile;
    private TarWriter writer;
    private final RandomAccessFile journalFile;
    private final RandomAccessFile lockFile;
    private final FileLock lock;
    private final AtomicReference<RecordId> head;
    private final AtomicReference<RecordId> persistedHead;
    private final BackgroundThread flushThread;
    private final BackgroundThread compactionThread;
    private CompactionStrategy compactionStrategy = CompactionStrategy.NO_COMPACTION;
    private final AtomicBoolean cleanupNeeded = new AtomicBoolean(false);
    private final LinkedList<File> toBeRemoved = Lists.newLinkedList();
    private final SegmentVersion version = SegmentVersion.V_11;
    private final GCMonitor gcMonitor;

    @Nonnull
    public static Builder newFileStore(@Nonnull File directory) {
        return new Builder((File)Preconditions.checkNotNull((Object)directory));
    }

    @Deprecated
    public FileStore(BlobStore blobStore, File directory, int maxFileSizeMB, boolean memoryMapping) throws IOException {
        this(blobStore, directory, EmptyNodeState.EMPTY_NODE, maxFileSizeMB, 0, memoryMapping, GCMonitor.EMPTY);
    }

    @Deprecated
    public FileStore(File directory, int maxFileSizeMB, boolean memoryMapping) throws IOException {
        this(null, directory, maxFileSizeMB, memoryMapping);
    }

    @Deprecated
    public FileStore(File directory, int maxFileSizeMB) throws IOException {
        this(null, directory, maxFileSizeMB, MEMORY_MAPPING_DEFAULT);
    }

    @Deprecated
    public FileStore(File directory, int maxFileSizeMB, int cacheSizeMB, boolean memoryMapping) throws IOException {
        this(null, directory, EmptyNodeState.EMPTY_NODE, maxFileSizeMB, cacheSizeMB, memoryMapping, GCMonitor.EMPTY);
    }

    @Deprecated
    FileStore(File directory, NodeState initial, int maxFileSize) throws IOException {
        this(null, directory, initial, maxFileSize, -1, MEMORY_MAPPING_DEFAULT, GCMonitor.EMPTY);
    }

    @Deprecated
    public FileStore(BlobStore blobStore, File directory, NodeState initial, int maxFileSizeMB, int cacheSizeMB, boolean memoryMapping) throws IOException {
        this(blobStore, directory, initial, maxFileSizeMB, cacheSizeMB, memoryMapping, GCMonitor.EMPTY);
    }

    private FileStore(BlobStore blobStore, final File directory, NodeState initial, int maxFileSizeMB, int cacheSizeMB, boolean memoryMapping, GCMonitor gcMonitor) throws IOException {
        ((File)Preconditions.checkNotNull((Object)directory)).mkdirs();
        this.tracker = cacheSizeMB < 0 ? new SegmentTracker(this, 0, this.getVersion()) : (cacheSizeMB > 0 ? new SegmentTracker(this, cacheSizeMB, this.getVersion()) : new SegmentTracker(this, this.getVersion()));
        this.blobStore = blobStore;
        this.directory = directory;
        this.maxFileSize = maxFileSizeMB * 0x100000;
        this.memoryMapping = memoryMapping;
        this.gcMonitor = gcMonitor;
        this.journalFile = new RandomAccessFile(new File(directory, JOURNAL_FILE_NAME), "rw");
        this.lockFile = new RandomAccessFile(new File(directory, LOCK_FILE_NAME), "rw");
        Map<Integer, Map<Character, File>> map = FileStore.collectFiles(directory);
        this.readers = Lists.newArrayListWithCapacity((int)map.size());
        Object[] indices = map.keySet().toArray(new Integer[map.size()]);
        Arrays.sort(indices);
        for (int i = indices.length - 1; i >= 0; --i) {
            this.readers.add(TarReader.open(map.get(indices[i]), memoryMapping));
        }
        this.writeNumber = indices.length > 0 ? (Integer)indices[indices.length - 1] + 1 : 0;
        this.writeFile = new File(directory, String.format(FILE_NAME_FORMAT, this.writeNumber, "a"));
        this.writer = new TarWriter(this.writeFile);
        LinkedList heads = Lists.newLinkedList();
        String line = this.journalFile.readLine();
        while (line != null) {
            int space = line.indexOf(32);
            if (space != -1) {
                heads.add(line.substring(0, space));
            }
            line = this.journalFile.readLine();
        }
        RecordId id = null;
        while (id == null && !heads.isEmpty()) {
            RecordId last = RecordId.fromString(this.tracker, (String)heads.removeLast());
            SegmentId segmentId = last.getSegmentId();
            if (this.containsSegment(segmentId.getMostSignificantBits(), segmentId.getLeastSignificantBits())) {
                id = last;
                continue;
            }
            log.warn("Unable to access revision {}, rewinding...", (Object)last);
        }
        this.journalFile.seek(this.journalFile.length());
        this.lock = this.lockFile.getChannel().lock();
        if (id != null) {
            this.head = new AtomicReference<Object>(id);
            this.persistedHead = new AtomicReference<RecordId>(id);
        } else {
            NodeBuilder builder = EmptyNodeState.EMPTY_NODE.builder();
            builder.setChildNode("root", initial);
            this.head = new AtomicReference<RecordId>(this.tracker.getWriter().writeNode(builder.getNodeState()).getRecordId());
            this.persistedHead = new AtomicReference<Object>(null);
        }
        this.flushThread = new BackgroundThread("TarMK flush thread [" + directory + "]", 5000L, new Runnable(){

            @Override
            public void run() {
                try {
                    FileStore.this.flush();
                }
                catch (IOException e) {
                    log.warn("Failed to flush the TarMK at" + directory, (Throwable)e);
                }
            }
        });
        this.compactionThread = new BackgroundThread("TarMK compaction thread [" + directory + "]", -1L, new Runnable(){

            @Override
            public void run() {
                FileStore.this.maybeCompact(true);
            }
        });
        log.info("TarMK opened: {} (mmap={})", (Object)directory, (Object)memoryMapping);
    }

    public boolean maybeCompact(boolean cleanup) {
        long needed;
        log.info("TarMK compaction started");
        Runtime runtime = Runtime.getRuntime();
        long avail = runtime.totalMemory() - runtime.freeMemory();
        long delta = 0L;
        if (this.compactionStrategy.getCompactionMap() != null) {
            delta = this.compactionStrategy.getCompactionMap().getLastMergeWeight();
        }
        if ((needed = delta * (long)this.compactionStrategy.getMemoryThreshold()) >= avail) {
            this.gcMonitor.skipped("Not enough available memory {}, needed {}, last merge delta {}, so skipping compaction for now", IOUtils.humanReadableByteCount(avail), IOUtils.humanReadableByteCount(needed), IOUtils.humanReadableByteCount(delta));
            if (cleanup) {
                this.cleanupNeeded.set(true);
            }
            return false;
        }
        Stopwatch watch = Stopwatch.createStarted();
        this.compactionStrategy.setCompactionStart(System.currentTimeMillis());
        boolean compacted = false;
        CompactionGainEstimate estimate = this.estimateCompactionGain();
        long gain = estimate.estimateCompactionGain();
        if (gain >= 10L) {
            this.gcMonitor.info("Estimated compaction in {}, gain is {}% ({}/{}) or ({}/{}), so running compaction", watch, gain, estimate.getReachableSize(), estimate.getTotalSize(), IOUtils.humanReadableByteCount(estimate.getReachableSize()), IOUtils.humanReadableByteCount(estimate.getTotalSize()));
            if (!this.compactionStrategy.isPaused()) {
                this.compact();
                compacted = true;
            } else {
                this.gcMonitor.skipped("TarMK compaction paused", new Object[0]);
            }
        } else {
            this.gcMonitor.skipped("Estimated compaction in {}, gain is {}% ({}/{}) or ({}/{}), so skipping compaction for now", watch, gain, estimate.getReachableSize(), estimate.getTotalSize(), IOUtils.humanReadableByteCount(estimate.getReachableSize()), IOUtils.humanReadableByteCount(estimate.getTotalSize()));
        }
        if (cleanup) {
            this.cleanupNeeded.set(true);
        }
        return compacted;
    }

    static Map<Integer, Map<Character, File>> collectFiles(File directory) throws IOException {
        Map files;
        HashMap dataFiles = Maps.newHashMap();
        HashMap bulkFiles = Maps.newHashMap();
        for (File file : directory.listFiles()) {
            Matcher matcher = FILE_NAME_PATTERN.matcher(file.getName());
            if (!matcher.matches()) continue;
            Object index = Integer.parseInt(matcher.group(2));
            if ("data".equals(matcher.group(1))) {
                files = (Map)dataFiles.get(index);
                if (files == null) {
                    files = Maps.newHashMap();
                    dataFiles.put(index, files);
                }
                Character generation = Character.valueOf('a');
                if (matcher.group(4) != null) {
                    generation = Character.valueOf(matcher.group(4).charAt(0));
                }
                Preconditions.checkState((files.put(generation, file) == null ? 1 : 0) != 0);
                continue;
            }
            Preconditions.checkState((bulkFiles.put(index, file) == null ? 1 : 0) != 0);
        }
        if (!bulkFiles.isEmpty()) {
            Integer newIndex;
            int position;
            Object[] indices;
            log.info("Upgrading TarMK file names in {}", (Object)directory);
            if (!dataFiles.isEmpty()) {
                indices = dataFiles.keySet().toArray(new Integer[dataFiles.size()]);
                Arrays.sort(indices);
                position = Math.max((Integer)indices[indices.length - 1] + 1, bulkFiles.size());
                for (Object index : indices) {
                    files = (Map)dataFiles.remove(index);
                    newIndex = position++;
                    for (Character generation : Sets.newHashSet(files.keySet())) {
                        File file = (File)files.get(generation);
                        File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, generation));
                        log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                        file.renameTo(newFile);
                        files.put(generation, newFile);
                    }
                    dataFiles.put(newIndex, files);
                }
            }
            indices = bulkFiles.keySet().toArray(new Integer[bulkFiles.size()]);
            Arrays.sort(indices);
            position = 0;
            for (Object index : indices) {
                File file = (File)bulkFiles.remove(index);
                newIndex = position++;
                File newFile = new File(directory, String.format(FILE_NAME_FORMAT, newIndex, "a"));
                log.info("Renaming {} to {}", (Object)file, (Object)newFile);
                file.renameTo(newFile);
                dataFiles.put(newIndex, Collections.singletonMap(Character.valueOf('a'), newFile));
            }
        }
        return dataFiles;
    }

    public synchronized long size() throws IOException {
        long size = this.writeFile.length();
        for (TarReader reader : this.readers) {
            size += reader.size();
        }
        return size;
    }

    private synchronized int count() {
        int count = this.writer.count();
        for (TarReader reader : this.readers) {
            count += reader.count();
        }
        return count;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    CompactionGainEstimate estimateCompactionGain() {
        CompactionGainEstimate estimate = new CompactionGainEstimate(this.getHead(), this.count());
        FileStore fileStore = this;
        synchronized (fileStore) {
            for (TarReader reader : this.readers) {
                reader.accept(estimate);
            }
        }
        return estimate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush() throws IOException {
        AtomicReference<RecordId> atomicReference = this.persistedHead;
        synchronized (atomicReference) {
            FileStore fileStore;
            RecordId before = this.persistedHead.get();
            RecordId after = this.head.get();
            boolean cleanup = this.cleanupNeeded.getAndSet(false);
            if (cleanup || !after.equals(before)) {
                this.tracker.getWriter().flush();
                this.writer.flush();
                fileStore = this;
                synchronized (fileStore) {
                    log.debug("TarMK journal update {} -> {}", (Object)before, (Object)after);
                    this.journalFile.writeBytes(after.toString10() + " root\n");
                    this.journalFile.getChannel().force(false);
                    this.persistedHead.set(after);
                    if (cleanup) {
                        this.cleanup();
                    }
                }
            }
            fileStore = this;
            synchronized (fileStore) {
                Iterator iterator = this.toBeRemoved.iterator();
                while (iterator.hasNext()) {
                    File file = (File)iterator.next();
                    log.debug("TarMK GC: Attempting to remove old file {}", (Object)file);
                    if (file.exists() && !file.delete()) continue;
                    log.debug("TarMK GC: Removed old file {}", (Object)file);
                    iterator.remove();
                }
            }
        }
    }

    public synchronized void cleanup() throws IOException {
        Stopwatch watch = Stopwatch.createStarted();
        long initialSize = this.size();
        this.gcMonitor.info("TarMK revision cleanup started. Current repository size {}", IOUtils.humanReadableByteCount(initialSize));
        System.gc();
        HashSet ids = Sets.newHashSet();
        for (SegmentId id : this.tracker.getReferencedSegmentIds()) {
            ids.add(new UUID(id.getMostSignificantBits(), id.getLeastSignificantBits()));
        }
        this.writer.cleanup(ids);
        CompactionMap cm = this.tracker.getCompactionMap();
        ArrayList list = Lists.newArrayListWithCapacity((int)this.readers.size());
        for (TarReader reader : this.readers) {
            TarReader cleaned = reader.cleanup(ids, cm);
            if (cleaned == reader) {
                list.add(reader);
                continue;
            }
            if (cleaned != null) {
                list.add(cleaned);
            }
            File file = reader.close();
            this.gcMonitor.info("TarMK revision cleanup reclaiming {}", file.getName());
            this.toBeRemoved.addLast(file);
        }
        this.readers = list;
        long finalSize = this.size();
        this.gcMonitor.cleaned(initialSize - finalSize, finalSize);
        this.gcMonitor.info("TarMK revision cleanup completed in {}. Post cleanup size is {} and space reclaimed {}", watch, IOUtils.humanReadableByteCount(finalSize), IOUtils.humanReadableByteCount(initialSize - finalSize));
    }

    public void compact() {
        Preconditions.checkArgument((!this.compactionStrategy.equals(CompactionStrategy.NO_COMPACTION) ? 1 : 0) != 0, (Object)"You must set a compactionStrategy before calling compact");
        this.gcMonitor.info("TarMK compaction running, strategy={}", this.compactionStrategy);
        long start = System.currentTimeMillis();
        SegmentWriter writer = new SegmentWriter(this, this.tracker, this.getVersion());
        Compactor compactor = new Compactor(writer, this.compactionStrategy.cloneBinaries());
        SegmentNodeState before = this.getHead();
        long existing = before.getChildNode("checkpoints").getChildNodeCount(Long.MAX_VALUE);
        if (existing > 1L) {
            this.gcMonitor.warn("TarMK compaction found {} checkpoints, you might need to run checkpoint cleanup", existing);
        }
        SegmentNodeState after = compactor.compact(EmptyNodeState.EMPTY_NODE, before);
        SetHead setHead = new SetHead(before, after, compactor);
        try {
            while (!this.compactionStrategy.compacted((Callable)setHead)) {
                SegmentNodeState head = this.getHead();
                after = compactor.compact(after, head);
                setHead = new SetHead(head, after, compactor);
            }
            this.gcMonitor.info("TarMK compaction completed in {}ms", System.currentTimeMillis() - start);
        }
        catch (Exception e) {
            this.gcMonitor.error("Error while running TarMK compaction", e);
        }
    }

    public synchronized Iterable<SegmentId> getSegmentIds() {
        ArrayList ids = Lists.newArrayList();
        for (UUID uuid : this.writer.getUUIDs()) {
            ids.add(this.tracker.getSegmentId(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits()));
        }
        for (TarReader reader : this.readers) {
            for (UUID uuid : reader.getUUIDs()) {
                ids.add(this.tracker.getSegmentId(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits()));
            }
        }
        return ids;
    }

    @Override
    public SegmentTracker getTracker() {
        return this.tracker;
    }

    @Override
    public SegmentNodeState getHead() {
        return new SegmentNodeState(this.head.get());
    }

    @Override
    public boolean setHead(SegmentNodeState base, SegmentNodeState head) {
        RecordId id = this.head.get();
        return id.equals(base.getRecordId()) && this.head.compareAndSet(id, head.getRecordId());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.compactionThread.close();
        this.flushThread.close();
        FileStore fileStore = this;
        synchronized (fileStore) {
            try {
                this.flush();
                this.writer.close();
                this.tracker.getWriter().dropCache();
                List<TarReader> list = this.readers;
                this.readers = Lists.newArrayList();
                for (TarReader reader : list) {
                    reader.close();
                }
                this.lock.release();
                this.lockFile.close();
                this.journalFile.close();
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to close the TarMK at " + this.directory, e);
            }
        }
        System.gc();
        log.info("TarMK closed: {}", (Object)this.directory);
    }

    @Override
    public boolean containsSegment(SegmentId id) {
        if (id.getTracker() == this.tracker) {
            return true;
        }
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        return this.containsSegment(msb, lsb);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean containsSegment(long msb, long lsb) {
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        Iterator<TarReader> iterator = this;
        synchronized (iterator) {
            if (this.writer.containsEntry(msb, lsb)) {
                return true;
            }
        }
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Segment readSegment(SegmentId id) {
        ByteBuffer buffer;
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        for (TarReader reader : this.readers) {
            try {
                buffer = reader.readEntry(msb, lsb);
                if (buffer == null) continue;
                return new Segment(this.tracker, id, buffer);
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + reader, (Throwable)e);
            }
        }
        Iterator<TarReader> iterator = this;
        synchronized (iterator) {
            try {
                ByteBuffer buffer2 = this.writer.readEntry(msb, lsb);
                if (buffer2 != null) {
                    return new Segment(this.tracker, id, buffer2);
                }
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + this.writer, (Throwable)e);
            }
        }
        for (TarReader reader : this.readers) {
            try {
                buffer = reader.readEntry(msb, lsb);
                if (buffer == null) continue;
                return new Segment(this.tracker, id, buffer);
            }
            catch (IOException e) {
                log.warn("Failed to read from tar file " + reader, (Throwable)e);
            }
        }
        throw new SegmentNotFoundException(id);
    }

    @Override
    public synchronized void writeSegment(SegmentId id, byte[] data, int offset, int length) {
        try {
            long size = this.writer.writeEntry(id.getMostSignificantBits(), id.getLeastSignificantBits(), data, offset, length);
            if (size >= (long)this.maxFileSize) {
                this.writer.close();
                ArrayList list = Lists.newArrayListWithCapacity((int)(1 + this.readers.size()));
                list.add(TarReader.open(this.writeFile, this.memoryMapping));
                list.addAll(this.readers);
                this.readers = list;
                ++this.writeNumber;
                this.writeFile = new File(this.directory, String.format(FILE_NAME_FORMAT, this.writeNumber, "a"));
                this.writer = new TarWriter(this.writeFile);
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Blob readBlob(String blobId) {
        if (this.blobStore != null) {
            return new BlobStoreBlob(this.blobStore, blobId);
        }
        throw new IllegalStateException("Attempt to read external blob with blobId [" + blobId + "] " + "without specifying BlobStore");
    }

    @Override
    public BlobStore getBlobStore() {
        return this.blobStore;
    }

    @Override
    public void gc() {
        if (this.compactionStrategy == CompactionStrategy.NO_COMPACTION) {
            log.warn("Call to gc while compaction strategy set to {}. ", (Object)CompactionStrategy.NO_COMPACTION);
        }
        this.compactionThread.trigger();
    }

    public FileStore setCompactionStrategy(CompactionStrategy strategy) {
        this.compactionStrategy = strategy;
        return this;
    }

    public SegmentVersion getVersion() {
        return this.version;
    }

    private static class LoggingGCMonitor
    implements GCMonitor {
        public GCMonitor delegatee = GCMonitor.EMPTY;

        private LoggingGCMonitor() {
        }

        @Override
        public void info(String message, Object ... arguments) {
            log.info(message, arguments);
            this.delegatee.info(message, arguments);
        }

        @Override
        public void warn(String message, Object ... arguments) {
            log.warn(message, arguments);
            this.delegatee.warn(message, arguments);
        }

        @Override
        public void error(String message, Exception exception) {
            this.delegatee.error(message, exception);
        }

        @Override
        public void skipped(String reason, Object ... arguments) {
            log.info(reason, arguments);
            this.delegatee.skipped(reason, arguments);
        }

        @Override
        public void compacted() {
            this.delegatee.compacted();
        }

        @Override
        public void cleaned(long reclaimedSize, long currentSize) {
            this.delegatee.cleaned(reclaimedSize, currentSize);
        }
    }

    private class SetHead
    implements Callable<Boolean> {
        private final SegmentNodeState before;
        private final SegmentNodeState after;
        private final Compactor compactor;

        public SetHead(SegmentNodeState before, SegmentNodeState after, Compactor compactor) {
            this.before = before;
            this.after = after;
            this.compactor = compactor;
        }

        @Override
        public Boolean call() throws Exception {
            if (FileStore.this.setHead(this.before, this.after)) {
                CompactionMap cm = this.compactor.getCompactionMap();
                FileStore.this.tracker.setCompactionMap(cm);
                FileStore.this.compactionStrategy.setCompactionMap(cm);
                FileStore.this.tracker.getWriter().dropCache();
                FileStore.this.tracker.getWriter().flush();
                FileStore.this.gcMonitor.compacted();
                FileStore.this.tracker.clearSegmentIdTables(FileStore.this.compactionStrategy);
                return true;
            }
            return false;
        }
    }

    public static class Builder {
        private final File directory;
        private BlobStore blobStore;
        private NodeState root = EmptyNodeState.EMPTY_NODE;
        private int maxFileSize = 256;
        private int cacheSize;
        private boolean memoryMapping;
        private final LoggingGCMonitor gcMonitor = new LoggingGCMonitor();

        private Builder(File directory) {
            this.directory = directory;
        }

        @Nonnull
        public Builder withBlobStore(@Nonnull BlobStore blobStore) {
            this.blobStore = (BlobStore)Preconditions.checkNotNull((Object)blobStore);
            return this;
        }

        @Nonnull
        public Builder withRoot(@Nonnull NodeState root) {
            this.root = (NodeState)Preconditions.checkNotNull((Object)root);
            return this;
        }

        @Nonnull
        public Builder withMaxFileSize(int maxFileSize) {
            this.maxFileSize = maxFileSize;
            return this;
        }

        @Nonnull
        public Builder withCacheSize(int cacheSize) {
            this.cacheSize = cacheSize;
            return this;
        }

        @Nonnull
        public Builder withNoCache() {
            this.cacheSize = -1;
            return this;
        }

        @Nonnull
        public Builder withMemoryMapping(boolean memoryMapping) {
            this.memoryMapping = memoryMapping;
            return this;
        }

        @Nonnull
        public Builder withGCMonitor(@Nonnull GCMonitor gcMonitor) {
            this.gcMonitor.delegatee = (GCMonitor)Preconditions.checkNotNull((Object)gcMonitor);
            return this;
        }

        @Nonnull
        public FileStore create() throws IOException {
            return new FileStore(this.blobStore, this.directory, this.root, this.maxFileSize, this.cacheSize, this.memoryMapping, this.gcMonitor);
        }
    }
}

