/*******************************************************************************
 *
 *    ADOBE CONFIDENTIAL
 *     ___________________
 *
 *     Copyright 2021 Adobe
 *     All Rights Reserved.
 *
 *     NOTICE: All information contained herein is, and remains
 *     the property of Adobe and its suppliers, if any. The intellectual
 *     and technical concepts contained herein are proprietary to Adobe
 *     and its suppliers and are protected by all applicable intellectual
 *     property laws, including trade secret and copyright laws.
 *     Dissemination of this information or reproduction of this material
 *     is strictly forbidden unless prior written permission is obtained
 *     from Adobe.
 *
 ******************************************************************************/
package com.adobe.cq.cif.common.associatedcontent;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ProviderType;

import com.adobe.cq.dam.cfm.ContentFragment;
import com.day.cq.dam.api.Asset;
import com.day.cq.wcm.api.Page;

/**
 * This services provides access to content associated with products an categories. It uses queries, optimised for the product's
 * out-of-the-box indexes to retrieve them.
 */
@ProviderType
public interface AssociatedContentService {

    /**
     * Defines the parameters used to query {@link Asset}s (except {@link ContentFragment}s) as associated content.
     */
    @ProviderType
    final class AssetParams extends BaseParams<AssetParams> {
        private AssetParams() {}

        /**
         * Creates an {@code AssetParams} instance with a product or category identifier.
         *
         * @param identifier product identifier or category identifier
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifier
         */
        public static AssetParams of(String identifier) {
            return BaseParams.of(AssetParams::new, identifier);
        }

        /**
         * Creates an {@code AssetParams} instance with a collection of product identifiers or category identifiers.
         *
         * @param identifiers the product identifiers or category identifiers
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifiers collection
         */
        public static AssetParams of(Collection<String> identifiers) {
            return BaseParams.of(AssetParams::new, identifiers);
        }
    }

    /**
     * Defines the parameters used to query experience fragments as associated content.
     */
    @ProviderType
    final class XfParams extends BaseParams<XfParams> {
        private String location;

        private XfParams() {}

        /**
         * Returns the experience fragment location.
         *
         * @return content experience location
         */
        public String location() {
            return location;
        }

        /**
         * Sets the experience fragment location.
         *
         * @param location experience fragment location
         *
         * @return the parameters instance
         */
        public XfParams location(String location) {
            this.location = location;
            return this;
        }

        /**
         * Creates an {@code XfParams} instance with a product or category identifier.
         *
         * @param identifier product identifier or category identifier
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifier
         */
        public static XfParams of(String identifier) {
            return BaseParams.of(XfParams::new, identifier);
        }

        /**
         * Creates an {@code XfParams} instance with a collection of product identifiers or category identifiers.
         *
         * @param identifiers the product identifiers or category identifiers
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifiers collection
         */
        public static XfParams of(Collection<String> identifiers) {
            return BaseParams.of(XfParams::new, identifiers);
        }
    }

    /**
     * Defines the parameters used to query {@link ContentFragment}s as associated content.
     */
    @ProviderType
    final class CfParams extends BaseParams<CfParams> {
        private String model;
        private String property;

        private CfParams() {}

        /**
         * Returns the content fragment model path.
         *
         * @return content fragment model path
         */
        public String model() {
            return model;
        }

        /**
         * Sets the content fragment model path.
         *
         * @param model content fragment model path
         *
         * @return the parameters instance
         */
        public CfParams model(String model) {
            this.model = model;
            return this;
        }

        /**
         * Returns the content fragment model property to match the identifier.
         *
         * @return content fragment model property
         */
        public String property() {
            return property;
        }

        /**
         * Sets the content fragment model property to match the identifier.
         *
         * @param property content fragment model property
         *
         * @return the parameters instance
         */
        public CfParams property(String property) {
            this.property = property;
            return this;
        }

        /**
         * Creates a {@code CfParams} instance with a product or category identifier.
         *
         * @param identifier product identifier or category identifier
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifier
         */
        public static CfParams of(String identifier) {
            return BaseParams.of(CfParams::new, identifier);
        }

        /**
         * Creates a {@code CfParams} instance with a collection of product identifiers or category identifiers.
         *
         * @param identifiers the product identifiers or category identifiers
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifiers collection
         */
        public static CfParams of(Collection<String> identifiers) {
            return BaseParams.of(CfParams::new, identifiers);
        }
    }

    /**
     * Defines the parameters used to query {@link Page}s as associated content.
     */
    @ProviderType
    final class PageParams extends BaseParams<PageParams> {
        private PageParams() {}

        /**
         * Creates a {@code PageParams} instance with a product or category identifier.
         *
         * @param identifier product identifier or category identifier
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifier
         */
        public static PageParams of(String identifier) {
            return BaseParams.of(PageParams::new, identifier);
        }

        /**
         * Creates a {@code PageParams} instance with a collection of product identifiers or category identifiers.
         *
         * @param identifiers the product identifiers or category identifiers
         *
         * @return the new instance
         *
         * @throws IllegalArgumentException for null or empty identifiers collection
         */
        public static PageParams of(Collection<String> identifiers) {
            return BaseParams.of(PageParams::new, identifiers);
        }
    }

    /**
     * Returns an {@link AssociatedContentQuery} to query for all {@link Asset}s (excl. {@link ContentFragment}s),
     * that are associated to the product with the given identifier.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<Asset> listProductAssets(ResourceResolver rr, AssetParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all Experience Fragments, that are associated to the
     * product with the given identifier.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<Page> listProductExperienceFragments(ResourceResolver rr, XfParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all Experience Fragments, that are associated to the
     * category with the given identifier.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<Page> listCategoryExperienceFragments(ResourceResolver rr, XfParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all {@link ContentFragment}s, that are associated to the
     * product with the given identifier.
     * The both {@code CfParams.model} and {@code CfParams.property} should be provided to have effect,
     * they are ignored otherwise.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<ContentFragment> listProductContentFragments(ResourceResolver rr, CfParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all {@link ContentFragment}s, that are associated to the
     * category with the given identifier.
     * The both {@code CfParams.model} and {@code CfParams.property} should be provided to have effect,
     * they are ignored otherwise.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<ContentFragment> listCategoryContentFragments(ResourceResolver rr, CfParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all {@link Page}s, that are associated to the product
     * with the given identifier.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<Page> listProductContentPages(ResourceResolver rr, PageParams params);

    /**
     * Returns an {@link AssociatedContentQuery} to query for all {@link Page}s, that are associated to the category
     * with the given identifier.
     *
     * @param rr the {@link ResourceResolver} to use for querying
     * @param params search parameters
     *
     * @return a prepared {@link AssociatedContentQuery}
     */
    AssociatedContentQuery<Page> listCategoryContentPages(ResourceResolver rr, PageParams params);
}

@ProviderType
abstract class BaseParams<P extends BaseParams<P>> {
    Collection<String> identifiers;
    String path;

    BaseParams() {}

    /**
     * Returns the product or category identifiers.
     *
     * @return the product or category identifiers
     */
    public Collection<String> identifiers() {
        return identifiers;
    }

    /**
     * Returns the search path.
     *
     * @return search path
     */
    public String path() {
        return path;
    }

    /**
     * Sets the search path for associated content queries.
     *
     * @param path associated content search path
     *
     * @return this params instance
     */
    @SuppressWarnings("unchecked")
    public P path(String path) {
        this.path = path;
        return (P) this;
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "{" +
            "identifiers=" + identifiers +
            ", path='" + path + '\'' +
            Arrays.stream(getClass().getDeclaredFields()).filter(f -> !f.isSynthetic()).map(f -> {
                try {
                    f.setAccessible(true);
                    return f.getName() + "=" +
                        (String.class.equals(f.getType()) ? "'" + f.get(this) + "'" : f.get(this));
                } catch (Exception x) {
                    return f.getName() + "=?";
                }
            }).reduce("", (s1, s2) -> s1 + ", " + s2) +
            '}';
    }

    /**
     * Static factory for creating instances of concrete {@code BaseParams} implementations from a product or
     * category identifier.
     *
     * @param function constructor of concrete {@code BaseParams} subclass
     * @param identifier product identifier or category identifier
     *
     * @return the parameters instance
     *
     * @throws IllegalArgumentException for null or empty identifier
     */
    static <T extends BaseParams<T>> T of(Supplier<T> function, String identifier) {
        if (StringUtils.isBlank(identifier)) {
            throw new IllegalArgumentException("Missing identifier");
        }

        T params = function.get();
        params.identifiers = Collections.singletonList(identifier);

        return params;
    }

    /**
     * Static factory for creating instances of concrete {@code BaseParams} implementations from product or
     * category identifiers.
     *
     * @param function constructor of concrete {@code BaseParams} subclass
     * @param identifiers product identifiers or category identifiers
     *
     * @return the parameters instance
     *
     * @throws IllegalArgumentException for null or empty identifiers collection
     */
    static <T extends BaseParams<T>> T of(Supplier<T> function, Collection<String> identifiers) {
        if (identifiers == null) {
            throw new IllegalArgumentException("Missing identifiers");
        }

        identifiers = identifiers.stream().filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        if (identifiers.isEmpty()) {
            throw new IllegalArgumentException("Missing identifiers");
        }

        T params = function.get();
        params.identifiers = Collections.unmodifiableCollection(identifiers);

        return params;
    }
}
