/*
*
*	File: Font.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font;

import java.io.Serializable;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontDescription;

/**
 * The basic interface for all public font interaction.
 * 
 * <p>
 * This class provides the basic functionality common to all Fonts.
 * 
 * <h4>Metrics</h4>
 * 
 * <p>
 * Metrics are expressed in a metric space, which is related
 * to the em space by the matrix (1/ux 0, 0, 1/uy), where 
 * ux is the units per em in the X direction and uy is the units per em
 * in the Y direction. Those two quantities can be retrieved by the
 * {@link #getUnitsPerEmX()} and {@link #getUnitsPerEmY()} methods.
 * The values ux and uy may or may not be related to any data in the font; 
 * they are only guaranteed to be compatible with the values returned
 * by methods that retrieve metrics. In other words, if x is an
 * horizontal metric returned by some method, only the value x / ux is
 * meaningful, and the specific values x and ux may change from one execution
 * to the next (ux is constant for the life of a Font object).
 * 
 * <p>
 * Very often (but not always), the font metrics stored in the font are
 * expressed in the metric space (or conversely, the metric space is selected
 * to be the space in which the raw font data is expressed). Returning
 * metrics in the metric space, instead of in the em space, allows the 
 * conversion to the em space to be performed less often (e.g. only after
 * metrics such as the advance width have been cumulated). Another benefit
 * is that the conversion to em space is often followed by a conversion to
 * some kind of user space (e.g. scaling by the point size, or expressing
 * the result in some other units than points); those further conversions
 * can be combined with the metric to em space conversion, resulting in less
 * computations.
 * 
 * <h4>Glyphs<h4>
 * 
 * <p>A font contains a number of glyphs, which are identified by their 
 * gids. Gids are by construction in the range [0, numGlyphs-1[. The glyph 
 * with gid 0 has a special meaning: it is the .notdef glyph, which is used, 
 * e.g. when a font contains no glyph for a given character. 
 * 
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>
 * Implementations of this interface are synchronized as needed, so that users
 * can use a Font in multiple threads without synchronization on their side.
 * 
 * <h4>Handling of symbolic fonts</h4>
 * 
 * <p>
 * A <i>symbolic </i> font is an OpenType font (without or without OpenType
 * tables) which does not have Unicode cmap, but has a Microsoft/Symbol cmap.
 * The fonts Wingdings, Webdings, and Symbol that are provided by Microsoft are
 * symbolic fonts. The symbol cmaps, typically maps the code points 0xF0 <i>yz
 * </i>
 * 
 * <p>
 * AFE behaves as if those fonts had a Unicode cmap. If 0xF0 <i>yz </i> is
 * mapped in the symbolic cmap, then both U+F0 <i>yz </i> and U+00 <i>yz </i>
 * are mapped in the Unicode cmap. The purpose of this double mapping is to
 * approximate the behavior of GDI and applications such as Word.
 * 
 * <p>
 * The first set of mappings, from U+F0 <i>yz </i>, are rather innocuous, as
 * those code points are Private Use Area, and have very little semantic. The
 * second set of mappings, from U+00 <i>yz </i>, are very problematic, because
 * the the semantic of the source characters is typically not applicable. This
 * is particulary true for the glyphs mapped from U+0000 .. U+001F and U+0080 ..
 * U+009F (because those characters have the semantic of control character) and
 * the glyph mapped from U+00AD SOFT HYPHEN (which is a Cf character that is not
 * normally rendered). Other strange behavior can occur in bidi processing and
 * line breaking processing. To help clients avoid those problems, 
 * the {@link #isSymbolic} method returns true for symbolic fonts.
 *  
 */
public interface Font extends Serializable
{
    /** Return the units per em in the X direction. 
     * 
     * <p>Unless otherwise specified, horizontal metrics returned by AFE are
     * expressed in a metric space, and must be divided by the units per em 
     * in the X direction to get em values.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     */
    double getUnitsPerEmX () 
    throws UnsupportedFontException, InvalidFontException, FontLoadingException;

    /** Return the units per em in the Y direction. 
     * 
     * <p>Unless otherwise specified, vertical metrics returned by AFE are
     * expressed in a metric space, and must be divided by the units per em 
     * in the Y direction to get em values.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     */
    double getUnitsPerEmY () 
    throws UnsupportedFontException, InvalidFontException, FontLoadingException;

    /** Return the line metrics for this font, as expressed by the 
     * font designer. 
     * 
     * <p>The metrics are expressed in the metric space of the font.
     * 
     * <p>Some font formats do not support the notion of line metrics,
     * and in those cases, this method returns null.
     * 
     * <p>See also the {@link #getCoolTypeLineMetrics()} method.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     */
    LineMetrics getLineMetrics ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /** Emulates the CoolType API CTFontDict:GetHorizontalMetrics.
     * 
     * <p>The metrics are expressed in the metric space of the font.
     * 
     * <p>See also the {@link #getLineMetrics()} method.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    LineMetrics getCoolTypeLineMetrics ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /** Emulates the CoolType API CCTFontInstance::GetBBox.
     * 
     * In particular, it calculates all bboxes instead of relying on metrics in the font.
     * <p>The metrics are expressed in the metric space of the font.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException */
    Rect getCoolTypeGlyphBBox(int glyphID)
    	throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /** Emulates the CoolType API CTFontDict:GetUnderlineInfo.
     * 
     * <p>The metrics are expressed in the metric space of the font.
     * 
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    UnderlineMetrics getCoolTypeUnderlineMetrics ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /**
     * Emulates the CoolType API CCTFontDict::GetWritingScript.
     * @return The "primary script" associated with this font.
     * @throws FontLoadingException
     * @throws InvalidFontException
     * @throws UnsupportedFontException
     */
    public CoolTypeScript getCoolTypeScript() throws FontLoadingException, InvalidFontException, UnsupportedFontException;
    
    /** Emulates the CoolType API CTFontDict:GetIdeoEmBox.
     * 
     * <p>The metrics are expressed in the metric space of the font.
     *
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    Rect getCoolTypeIdeoEmBox ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /**
     * Emulates the CoolType API CTFontDict::hasPropRoman.
     * @return true if the font contains proportional roman glyphs.
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    boolean hasCoolTypeProportionalRoman()
    	throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    /** Emulates the CoolType API CTFontDict:GetICFBox.
     * 
     * <p>The metrics are expressed in the metric space of the font.
     *
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    Rect getCoolTypeIcfBox ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    

    /** Tell whether the font is symbolic.
     * @throws UnsupportedFontException
     * @throws InvalidFontException
     * @throws FontLoadingException
     */
    boolean isSymbolic ()
        throws UnsupportedFontException, InvalidFontException, FontLoadingException;
    
    //------------------------------------------------------- pdf generation ---
    
    boolean canEmbedForPrintAndPreview() 
    	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
    
    boolean canEmbedForEditting()
    	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
    
    /** Create a subset for this font. 
     */
    Subset createSubset() 
    		throws InvalidFontException, UnsupportedFontException, FontLoadingException;

    /**
     * Fetch data needed to construct a DefineFont2 or 3 tag in SWF.
     */
    SWFFontDescription getSWFFontDescription ()
    	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
    
    /**
     * Fetch data needed to construct a DefineFont4 tag in SWF. Returns null if
     * a font does not support DefineFont4.
     */
    SWFFont4Description getSWFFont4Description()
    	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
    
    //-------------------------------------------------------------------------
    /** Get the PDFFontDescription for this font. 
     */
    PDFFontDescription getPDFFontDescription ()
        throws InvalidFontException, UnsupportedFontException, FontLoadingException;       

    /** Get the XDCFontDescription for this font. 
     */
    XDCFontDescription getXDCFontDescription ()
        throws InvalidFontException, UnsupportedFontException, FontLoadingException;       

    /** Get the PostscriptFontDescriptions for this font. 
     */
    PostscriptFontDescription[] getPostscriptFontDescription ()
        throws InvalidFontException, UnsupportedFontException, FontLoadingException;       

    /** Get the CSS20FontDescriptions for this font. 
     */
    CSS20FontDescription[] getCSS20FontDescription ()
        throws InvalidFontException, UnsupportedFontException, FontLoadingException;
    
    /** Get the preferred CSS20FontDescriptions for this font. 
     * Returns null if there is no CSS20 description for this font.
     */
    CSS20FontDescription getPreferredCSS20FontDescription()
    	throws InvalidFontException, UnsupportedFontException, FontLoadingException;

	/**
	 * Get all of the FXG descriptions for the font.
	 * @return an array of all the FXG descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	FXGFontDescription[] getFXGFontDescription() 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
	
	/**
	 * Get all of the FXG descriptions for this font on the given platform.
	 * @param platform the platform to get descriptions for
	 * @return an array of all of the specified FXG descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	FXGFontDescription[] getFXGFontDescription(Platform platform)
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
	
	/**
	 * Get all of the FXG descriptions for this font on the given platform
	 * and for the given locale.
	 * @param platform the platform to get descriptions for
	 * @param locale the locale to get descriptions for
	 * @return an array of all of the specified FXG descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	FXGFontDescription[] getFXGFontDescription(Platform platform, ULocale locale)
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
	
	/**
	 * Get all of the Platform descriptions for the font.
	 * @return an array of all the Platform descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	PlatformFontDescription[] getPlatformFontDescription() 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
	
	/**
	 * Get all of the Platform descriptions for this font on the given platform.
	 * @param platform the platform to get descriptions for
	 * @return an array of all of the specified Platform descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	PlatformFontDescription[] getPlatformFontDescription(Platform platform)
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
	
	/**
	 * Get all of the Platform descriptions for this font on the given platform
	 * and for the given locale.
	 * @param platform the platform to get descriptions for
	 * @param locale the locale to get descriptions for
	 * @return an array of all of the specified Platform descriptions for the font 
	 * or <code>null</code> if there are none
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 * @throws FontLoadingException
	 */
	PlatformFontDescription[] getPlatformFontDescription(Platform platform, ULocale locale)
	throws InvalidFontException, UnsupportedFontException, FontLoadingException;
}
