/* ****************************************************************************
 *
 *	File: COSObjectRef.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Represents a COS object reference as defined in section 3.2.9 of the PDF
 * Reference Manual version 1.4.
 */
class CosObjectRef extends CosObject {
	/**
	 * @see com.adobe.internal.pdftoolkit.core.cos.CosObject#CosObject(CosDocument, CosObjectInfo)
	 */
	CosObjectRef(CosDocument doc, CosObjectInfo info)
	{
		super(doc, null);
		setInfo(info);
	}

	/**
	 * 
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_ObjectRef;
	}

	/**
	 * @throws IOException 
	 */
	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
	throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
			CosObjectInfo info = getInfo();
			int objNum = info.getObjNum();
			outStream.write(StringOps.toByteArray(Integer.toString(objNum)));
			outStream.write(' ');
			int objGen = info.getObjGen();
			if (objGen != 0 && getDocument().willBeCompressed(objNum))
				objGen = 0;
			outStream.write(StringOps.toByteArray(Integer.toString(objGen)));
			outStream.write(StringOps.toByteArray(" R"));
	}
	
	/**
	 * We return the value of the object being referenced
	 * @return Object value of CosObject being referenced.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	@Override
	public Object getValue()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		
		CosObject obj;
		try {
			obj = getDocument().resolveReference(this);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
		return obj.getValue();
	}
	
	/**
	 * Return the type of the object being referenced
	 * @return Object value of CosObject being referenced.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public int getValueType()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{	
		CosObject obj;
		try {
			obj = getDocument().resolveReference(this);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
		return obj.getType();
	}
	
	@Override
	public boolean equals(CosObject value){
		return (value == this);
	}
}
