/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.foundation.model.responsivegrid;

import com.adobe.cq.export.json.ComponentExporter;
import com.day.cq.wcm.api.NameConstants;
import com.day.cq.wcm.api.components.Component;
import com.day.cq.wcm.api.designer.ComponentStyle;
import com.day.cq.wcm.commons.WCMUtils;
import com.day.cq.wcm.foundation.model.responsivegrid.export.ResponsiveColumnExporter;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.AbstractResource;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceMetadata;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.models.factory.ModelFactory;
import org.osgi.annotation.versioning.ProviderType;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.day.cq.wcm.foundation.model.responsivegrid.ResponsiveConstants.BREAKPOINT_VARIANT_NAME_DEFAULT;
import static com.day.cq.wcm.foundation.model.responsivegrid.ResponsiveConstants.DEFAULT_COLUMNS;
import static com.day.cq.wcm.foundation.model.responsivegrid.ResponsiveConstants.DEFAULT_COLUMN_CSS_PREFIX;
import static com.day.cq.wcm.foundation.model.responsivegrid.ResponsiveConstants.OFFSET_CSS_VARIANT_NAME;
import static com.day.cq.wcm.foundation.model.responsivegrid.ResponsiveGridUtils.createClassname;

/**
 * Represents a column of a responsive grid.
 * A responsive grid's column is basically a virtual wrapper of a component that is part of the grid.
 */
@ProviderType
@JsonSerialize(as = ResponsiveColumnExporter.class)
public class ResponsiveColumn extends AbstractResource implements ResponsiveColumnExporter {

    private final Resource wrappedResource;
    private final String classNames;
    private final Map<String, Breakpoint> breakpoints;
    private final String columnCssClassPrefix;
    private ComponentExporter componentModel;
    private SlingHttpServletRequest slingRequest;
    private ModelFactory modelFactory;

    public ResponsiveColumn(Resource resource,
                             Map<String, Breakpoint> gridBreakpoints,
                             String classNamePrefix) {
        columnCssClassPrefix = classNamePrefix + DEFAULT_COLUMN_CSS_PREFIX;
        this.wrappedResource = resource;
        Map<String, Breakpoint> generatedBreakpoints = getBreakpoints(this.wrappedResource, gridBreakpoints);
        Map<String, Breakpoint> missingBreakpoints = ResponsiveGridUtils.getMissingBreakpoints(generatedBreakpoints, gridBreakpoints);
        this.breakpoints = combineBreakpointMaps(generatedBreakpoints, missingBreakpoints);
        this.classNames = ResponsiveGridUtils.createClassNames(
                columnCssClassPrefix + " " + getComponentClasses(resource),
                this.breakpoints,
                this::generateBreakpointCssClasses);
    }

    public ResponsiveColumn(Resource resource,
                            Map<String, Breakpoint> gridBreakpoints,
                            String classNamePrefix,
                            SlingHttpServletRequest slingRequest,
                            ModelFactory modelFactory) {
        this(resource, gridBreakpoints, classNamePrefix);
        this.slingRequest = slingRequest;
        this.modelFactory = modelFactory;
    }

    /**
     * @param resource             The current resource
     * @param referenceBreakpoints The reference breakpoints (breakpoints of the grid parent)
     * @return Returns a map of processed breakpoints. If necessary, the breakpoints of the current resource are adjusted with the ones of the given reference (often the grid parent).
     */
    Map<String, Breakpoint> getBreakpoints(Resource resource, Map<String, Breakpoint> referenceBreakpoints) {
        if (resource == null) {
            return Collections.emptyMap();
        }

        Resource responsiveCfg = resource.getChild(NameConstants.NN_RESPONSIVE_CONFIG);
        Map<String, Breakpoint> breakpointMap = new HashMap<>();

        if (responsiveCfg != null) {
            // Add the respective class names for each of the breakpoints
            for (Iterator<Resource> resCfgIt = responsiveCfg.listChildren(); resCfgIt.hasNext(); ) {
                Resource resCfg = resCfgIt.next();
                String breakpointName = resCfg.getName();
                ValueMap cfg = resCfg.getValueMap();
                int width = cfg.get("width", DEFAULT_COLUMNS);
                int offset = cfg.get("offset", 0);
                Breakpoint.ResponsiveBehavior behavior =
                        Breakpoint.ResponsiveBehavior.valueOf(
                                cfg.get("behavior",
                                        Breakpoint.ResponsiveBehavior.none.toString()));

                // Limit the width and offset to the grid width
                Breakpoint gridBreakpoint =
                        referenceBreakpoints != null ? referenceBreakpoints.get(breakpointName) : null;

                if (gridBreakpoint != null && width + offset > gridBreakpoint.getWidth()) {
                    if (width > gridBreakpoint.getWidth()) {
                        // The width is bigger than the parent
                        // We take the full size
                        width = gridBreakpoint.getWidth();
                        offset = 0;
                    } else if (offset < gridBreakpoint.getWidth()) {
                        // There is room for the offset and a width
                        width = gridBreakpoint.getWidth() - offset;
                    } else {
                        // There is no room for an offset
                        offset = 0;
                    }
                }

                breakpointMap.put(breakpointName, new Breakpoint(breakpointName, width, offset, behavior));
            }
        }

        return breakpointMap;
    }

    Map<String, Breakpoint> combineBreakpointMaps(Map<String, Breakpoint> columnBreakpoints, Map<String, Breakpoint> missingBreakpoints) {
        Map<String, Breakpoint> breakpoints = new HashMap<>();
        breakpoints.put(BREAKPOINT_VARIANT_NAME_DEFAULT, new Breakpoint(BREAKPOINT_VARIANT_NAME_DEFAULT, DEFAULT_COLUMNS, 0, null));
        breakpoints.putAll(missingBreakpoints);
        breakpoints.putAll(columnBreakpoints);
        return breakpoints;
    }


    String getComponentClasses(Resource resource) {
        String cssClass = "";
        ValueMap componentProperties;
        Component component = WCMUtils.getComponent(resource);
        if (component == null) {
            return cssClass;
        }

        componentProperties = component.getProperties();
        if (componentProperties != null) {
            cssClass = componentProperties.get(ComponentStyle.PN_CSS_CLASS, "");
        }
        return cssClass;
    }

    List<String> generateBreakpointCssClasses(Breakpoint breakpoint) {
        String breakpointName = breakpoint.getName();
        int offset = breakpoint.getOffset();
        int width = breakpoint.getWidth();
        Breakpoint.ResponsiveBehavior behavior = breakpoint.getResponsiveBehavior();
        List<String> breakpointClasses = new ArrayList<>();
        if (width > 0) {
            String widthBreakpointClassname = createClassname(Arrays.asList(columnCssClassPrefix, breakpointName, Integer.toString(width)));
            breakpointClasses.add(widthBreakpointClassname);
        }

        if(offset > -1) {
            String offsetBreakpointClassname = createClassname(Arrays.asList(columnCssClassPrefix, OFFSET_CSS_VARIANT_NAME, breakpointName, Integer.toString(offset)));
            breakpointClasses.add(offsetBreakpointClassname);
        }

        if (behavior != null) {
            String behaviorBreakpointClassname = createClassname(Arrays.asList(columnCssClassPrefix, breakpointName, behavior.toString()));
            breakpointClasses.add(behaviorBreakpointClassname);
        }
        return breakpointClasses;
    }

    @Override
    public Map<String, Breakpoint> getBreakpoints() {
        return this.breakpoints;
    }

    @Override
    public String getColumnClassNames() {
        return this.classNames;
    }

    @Nonnull
    @Override
    public ComponentExporter getExportedComponent() {
        if (componentModel == null) {
            componentModel = modelFactory.getModelFromWrappedRequest(slingRequest, wrappedResource, ComponentExporter.class);
        }
        return componentModel;
    }

    @Nonnull
    @Override
    public String getExportedType() {
        return wrappedResource.getResourceType();
    }

    /**
     * @return The wrapped {@link Resource} associated with the column.
     */
    @Nonnull
    public Resource getResource() {
        return wrappedResource;
    }

    @Nonnull
    public String getPath() {
        return wrappedResource.getPath();
    }

    /**
     * @deprecated Use {@link #getColumnClassNames()}
     */
    @Deprecated
    public String getCssClass() {
        return this.classNames;
    }

    /**
     * @deprecated Use {@link #getBreakpoints()}
     */
    @Deprecated
    public Set<String> getMissingBreakpointNames() {
        throw new UnsupportedOperationException();
    }

    /**
     * Returns the number of columns for the given breakpoint name
     *
     * @param breakpointName The name of the breakpoint
     * @return The number of columns for given breakpoint name.
     * @deprecated
     */
    @Deprecated
    public Integer getColumnCount(String breakpointName) {
        return breakpoints.get(breakpointName).getWidth();
    }

    /**
     * @deprecated
     */
    @Nonnull
    @Deprecated
    public String getResourceType() {
        return wrappedResource.getResourceType();
    }

    /**
     * @deprecated
     */
    @Deprecated
    public String getResourceSuperType() {
        return wrappedResource.getResourceSuperType();
    }

    /**
     * @deprecated
     */
    @Nonnull
    @Deprecated
    public ResourceMetadata getResourceMetadata() {
        return wrappedResource.getResourceMetadata();
    }

    /**
     * @deprecated
     */
    @Nonnull
    @Deprecated
    public ResourceResolver getResourceResolver() {
        return wrappedResource.getResourceResolver();
    }

    /**
     * @deprecated
     */
    @Deprecated
    public ValueMap getProperties() {
        return wrappedResource.adaptTo(ValueMap.class);
    }


}
