/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.msm.commons;

import java.util.Iterator;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.apache.commons.collections.FunctorException;
import org.apache.commons.collections.IteratorUtils;
import org.apache.commons.collections.Predicate;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.wcm.api.WCMException;
import com.day.cq.wcm.msm.api.LiveRelationship;

/**
 * A {@code BaseAction} providing basic item filtering functionality.
 * There are many cases when some special properties, nodes or node types should be skipped during the rollout process.
 * One simple example would be the jcr:uuid property that would cause problems if it is duplicated outside the version creating process.
 * <br>
 * This extension of the {@code BaseAction} uses pattern matching to skip execution on certain nodes, node types or properties.
 * The matching is performed by {@link com.day.cq.wcm.msm.commons.ItemFilterImpl ItemFilterImpl}  instances.<br>
 * To be able to have a different behavior for the Page's properties on its content-Node and on Page's Components two
 * Matcher are present.
 * <br>
 * By default, if no filters are set, the {@code RolloutManager} excludes will be used.
 *
 * @see ItemFilterImpl
 */
public abstract class FilteredAction extends BaseAction {

    private final ItemFilterImpl componentItemFilter;
    private final ItemFilterImpl pagePropertyFilter;

    /**
     * @param configuration       Pass properties containing your Instances configuration.
     * @param pageItemFilter      Filter set-up for Page's Content-Node
     * @param componentItemFilter Filter set-up for Page's Content-Node's children
     * @param factory             Factory that created this Instance
     */
    protected FilteredAction(ValueMap configuration,
                             ItemFilterImpl pageItemFilter,
                             ItemFilterImpl componentItemFilter,
                             BaseActionFactory<? extends FilteredAction> factory) {

        super(configuration, factory);
        this.pagePropertyFilter = pageItemFilter;
        this.componentItemFilter = componentItemFilter;
    }

    //------------------------------------------------# BaseAction #----------------------------------------------------

    /**
     * Extends the {@link com.day.cq.wcm.msm.commons.BaseAction BaseAction's} implementation.<br>
     * To the LiveAction's implementation's precondition, a FilteredAction's precondition is to match the ItemFilterImpl.
     * The ItemFilterImpl is applied and delegates to the LiveAction implementation for its precondition in the abstract
     * {@link #doHandle(org.apache.sling.api.resource.Resource, org.apache.sling.api.resource.Resource, com.day.cq.wcm.msm.api.LiveRelationship, boolean) doHandle - method}
     *
     * @param source       The Resource to roll-out.
     * @param target       The Resource to receive modification.
     * @param relation     LiveRelationship between the two given Resources
     * @param resetRollout True if rollout is run in reset mode
     * @return true if the ItemFilter allows to act on the Node and the LiveAction's precondition is met.
     * @throws RepositoryException Throws RepositoryException
     * @throws WCMException Throws WCMException
     * @see BaseAction#handles(org.apache.sling.api.resource.Resource, org.apache.sling.api.resource.Resource, com.day.cq.wcm.msm.api.LiveRelationship, boolean)
     */
    @Override
    protected boolean handles(Resource source, Resource target, LiveRelationship relation, boolean resetRollout)
            throws RepositoryException, WCMException {

        return filter(source, target) && doHandle(source, target, relation, resetRollout);
    }

    /**
     * Check if the precondition of this LiveAction implementation is met.
     *
     * @param source       Source of the LiveRelationship can be <code>null</code> in case the Source is not existing
     * @param target       Target of the LiveRelationship can be <code>null</code> in case the Target is not existing
     * @param relation     The LiveRelationship
     * @param resetRollout true in case the roll-out was called with the reset option.
     * @return true if the LiveAction's precondition is met else false
     * @throws RepositoryException Throws RepositoryException
     * @throws WCMException Throws WCMException
     */
    protected abstract boolean doHandle(Resource source, Resource target, LiveRelationship relation, boolean resetRollout)
            throws RepositoryException, WCMException;

    //------------------------------------------------# FilteredAction # 	------------------------------------------------

    /**
     * Gets the applicable {@code ItemFilterImpl} filter for the{@code Node}.
     *
     * @param node target {@code Node} of a filter operation
     * @return a {@code ItemFilterImpl} the pageFilter in case the Node is the Page's Content else the Component's Filter
     * @throws RepositoryException Throws RepositoryException
     */
    protected ItemFilterImpl getFilter(Node node) throws RepositoryException {
        if(isPage(node)) {
            return pagePropertyFilter;
        } else {
            return componentItemFilter;
        }
    }

    /**
     * Convenience to get an Iterator only containing Properties that match the related
     * {@link com.day.cq.wcm.msm.commons.ItemFilterImpl ItemFilterImpl}
     *
     * @param node to access the Properties from
     * @return Iterator containing the filtered Properties
     * @throws RepositoryException Throws RepositoryException
     */
    @SuppressWarnings("unchecked")
    protected Iterator<javax.jcr.Property> getFilteredProperties(Node node) throws RepositoryException {
        return IteratorUtils.filteredIterator(node.getProperties(), new ItemFilterPredicate(getFilter(node)));
    }

    /**
     * Convenience to get an Iterator only containing child-Nodes that match the related
     * {@link com.day.cq.wcm.msm.commons.ItemFilterImpl ItemFilterImpl}
     *
     * @param node to access the Nodes from
     * @return Iterator containing the filtered Nodes
     * @throws RepositoryException Throws RepositoryException
     */
    @SuppressWarnings("unchecked")
    protected Iterator<Node> getFilteredNodes(Node node) throws RepositoryException {
        return IteratorUtils.filteredIterator(node.getNodes(), new ItemFilterPredicate((getFilter(node))));
    }

    private boolean filter(Resource source, Resource target) throws RepositoryException {
        Node sourceNode = source!=null ? source.adaptTo(Node.class) : null;
        Node targetNode = target!=null ? target.adaptTo(Node.class) : null;
        Node checkNode = sourceNode!=null ? sourceNode : targetNode;
        return checkNode==null || !getFilter(checkNode).excludes(checkNode);
    }

    /**
     * -----------------------------------------------# Predicate #----------------------------------------------------
     * Evaluate based on an ItemFilterImpl
     */
    protected final static class ItemFilterPredicate implements Predicate {

        private final ItemFilterImpl filter;

        protected ItemFilterPredicate(ItemFilterImpl filter) {
            this.filter = filter;
        }

        public boolean evaluate(Object object) {
            try {
                if(object instanceof Property) {
                    return !filter.excludes((Property) object);
                } else {
                    return !filter.excludes((Node) object);
                }
            } catch(RepositoryException e) {
                throw new FunctorException(e);
            }
        }
    }
}