/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import java.util.List;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.ISODateTime;

/**
 * This is the base class for the host pseudo model ($host).
 * Each execution environment (server and client) needs to provide 
 * an implementation appropriate for their context.
 */
public abstract class HostPseudoModel extends PseudoModel {

	/**
	 * Instantiates a HostPseudoModel. 
	 */
	public HostPseudoModel() {
		setClass("hostPseudoModel");
	}

	/**
	 * @exclude from published api.
	 */
	@Override
	public ScriptTable getScriptTable() {
		return HostPseudoModelScript.moScriptTable;
	}

	/**
	 * Returns the name of the client application in which this application is currently running.
	 * For example, in the context of a PDF form viewed in Adobe Reader, this property returns "Reader". 
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.appType</code> scripting property
	 * is to be supported.
	 * @return the name of the client application in which this application is currently running.
	 */
	public String getAppType() {
		throw unimplemented("appType", true);
	}

	/**
	 * Gets the currently active page of a document at run time.
	 * Page values are 0-based, so the first page of a document returns a value of 0.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.currentPage</code> scripting property
	 * is to be supported.
	 * @return the currently active page as a 0-based number
	 */
	public int getCurrentPage() {
		throw unimplemented("currentPage", true);
	}

	/**
	 * Sets the currently active page of a document at run time.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.currentPage</code> scripting property
	 * is to be supported.
	 * @param nPage the 0-based page number to make active
	 */
	public void setCurrentPage(int nPage) {
		throw unimplemented("currentPage", true);
	}

	/**
	 * Returns the current local time expressed as an ISO 8601 string using
	 * the format YYYYMMDDTHHMMSS.
	 * <p/>
	 * This class provides a default implementation for the <code>$host.currentDateTime()</code>
	 * scripting method, and derived classes do not normally need to override this implementation.
	 * 
	 * @return the current local time, expressed as an ISO 8601 string 
	 */
	public String getCurrentDateTime() {
	 	ISODateTime oDateTime = new ISODateTime();
		String sNow = oDateTime.format("YYYYMMDDTHHMMSS");
		return sNow; 
	}
	
	/**
	 * Returns the ordinal number of the current document within the batch.
	 * Hosts which do not support batching will always return 0.
     * The default implementation always returns 0.
     * <p/>
     * Derived classes can override this method to change the behavior of
     * the <code>$host.documentInBatch()</code> scripting method.
	 * @return the ordinal number of the current document within the batch
	 */
	public int getDocumentInBatch() {
		return 0;
	}

	/**
	 * Returns the number of documents in the current batch.
	 * Hosts which do not support batching will always return 1.
	 * The default implementation always returns 1.
	 * <p/>
	 * Derived classes can override this method to change the behavior
	 * of the <code>$host.documentCountInBatch()</code> scripting method.
	 * @return the number of documents in the current batch
	 */
	public int getDocumentCountInBatch() {
		return 1;
	}

	/**
	 * Gets the language of the running host application.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.language</code> scripting property
	 * is to be supported.
	 * @return the language of the running host application
	 */
	public String getLanguage() {
		throw unimplemented("language", true);
	}

	/**
	 * Gets the name of the running host application.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.name</code> scripting property
	 * is to be supported.
	 * @return the name of the running host application
	 */
	public String getName() {
		throw unimplemented("name", true);
	}

	/**
	 * Gets the number of pages in the current document.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.numPages</code> scripting property
	 * is to be supported.
	 * @return the number of pages in the current document
	 */
	public int getNumPages() {
		throw unimplemented("numPages", true);
	}

	/**
	 * Gets the platform of the machine running the script.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.platform</code> scripting property
	 * is to be supported.
	 * @return the platform of the machine running the script
	 */
	public String getPlatform() {
		throw unimplemented("platform", true);
	}

	/**
	 * Gets the title of the document.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.title</code> scripting property
	 * is to be supported.
	 * @return the title of the document
	 */
	public String getTitle() {
		throw unimplemented("title", true);
	}

	/**
	 * Sets the title of the document.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.title</code> scripting property
	 * is to be supported.
	 * @param sTitle the title of the document
	 */
	public void setTitle(String sTitle) {
		throw unimplemented("title", true);
	}

	/**
	 * Gets the packaging of the application that is running the script.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.variation</code> scripting property
	 * is to be supported.
	 * @return the packaging of the application that is running the script
	 */
	public String getVariation() {
		throw unimplemented("variation", true);
	}

	/**
	 * Gets the version number of the current application.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.version</code> scripting property
	 * is to be supported.
	 * @return the version number of the current application
	 */
	public String getVersion() {
		throw unimplemented("version", true);
	}

	/**
	 * Determines whether calculate scripts will execute.
	 * <p/>
	 * The default implementation always returns <code>true</code>.
	 * This method must be overridden in derived classes if {@link #setCalculationsEnabled(boolean)}
	 * is overridden.
	 * @return <code>true</code> if calculate scripts are enabled.
	 */
	public boolean getCalculationsEnabled() {
		return true;
	}

	/**
	 * Determines whether calculate scripts will execute.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.calculationsEnabled</code> scripting property
	 * is to be supported.
	 * @param bEnabled <code>true</code> if calculation scripts are enabled
	 */
	public void setCalculationsEnabled(boolean bEnabled) {
		throw unimplemented("calculationsEnabled", true);
	}

	/**
	 * Determines whether validations will execute.
	 * <p/>
	 * The default implementation always returns <code>true</code>.
	 * This method must be overridden in derived classes if {@link #setValidationsEnabled(boolean)}
	 * is overridden.
	 * @return <code>true</code> if validations are enabled.
	 */
	public boolean getValidationsEnabled() {
		return true;
	}

	/**
	 * Determines whether validations will execute.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.calculationsEnabled</code> scripting property
	 * is to be supported.
	 * @param bEnabled <code>true</code> if validations are enabled
	 */
	public void setValidationsEnabled(boolean bEnabled) {
		throw unimplemented("validationsEnabled", true);
	}

	/**
	 * Causes the system to play a sound.
	 * An implementation for a non-client application may do nothing.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.beep()</code> scripting function
	 * is to be supported.
	 * @param nBeep a system code for the appropriate sound
	 */
	public void beep(int nBeep /* = 0 */) {
		throw unimplemented("beep", false);
	}

	/**
	 * Exports the data from the current form in either XDP or XML format to a file.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.exportData()</code> scripting function
	 * is to be supported.
	 * @param sFilename the location and file name of the file where the data will export.
	 * In a client application, omitting this parameter causes a dialog box to open to let the user
	 * select the file manually.
	 * @param bXDP if <code>true</code>, the data is exported in XDP format; 
	 * if <code>false</code>, the data is exported as plain XML data.
	 */
	public void exportData(String sFilename/* = "" */, boolean bXDP /* = true */) {
		throw unimplemented("exportData", false);
	}

	/**
	 * Imports data to the current form from a specified file.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.importData()</code> scripting function
	 * is to be supported.
	 * @param sFilename the location and name of the file from which the data will be imported.
	 * In a client application, if this parameter is an empty string, a dialog box opens to let the user
	 * select the file manually.
	 * @see #importDataPermsCheck()
	 */
	public void importData(String sFilename /* = "" */) {
		throw unimplemented("importData", false);
	}

	/**
	 * Retrieves the specified URL. It is available only for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.gotoURL()</code> scripting function
	 * is to be supported.
	 * @param sURL a string representing a fully qualified or relative URL. The URL may contain a query string.
	 * @param bNewInstance if <code>true</code>, the resulting pages are appended to the current document.
	 */
	public void gotoURL(String sURL, boolean bNewInstance /* = false */) {
		throw unimplemented("gotoURL", false);
	}

	/**
	 * Displays a dialog box on the screen. It is available only for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.messageBox()</code> scripting function
	 * is to be supported.
	 * @param sMessage the message to display
	 * @param sTitle the title to appear in the title bar of the dialog window
	 * @param nIcon the icon to display in the dialog box
	 * @param nType the buttons to display
	 * @return a value representing the button pressed by the user
	 */
	public int messageBox(String sMessage, String sTitle /* = "" */,
			int nIcon /* = 0 */, int nType /* = 0 */) {
		throw unimplemented("messageBox", false);
	}

	/**
	 * Resets the fields to their default values within the document.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.resetData()</code> scripting function
	 * is to be supported.
	 * @param oNodes a comma-delimited list of SOM expressions of the fields to reset to their default values.
	 * If empty, all fields in the form are reset to their default values.
	 * @see #resetDataPermsCheck(List)
	 */
	public void resetData(List<String> oNodes) {
		throw unimplemented("resetData", false);
	}

	/**
	 * Prints a specific number of pages from a document.
	 * This method is only available for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.print()</code> scripting function
	 * is to be supported.
	 * @param bUI if <code>true</code>, displays a print dialog box and prompts the user for
	 * printing setup information and confirmation of the action
	 * @param nStart the 0-based page number of the start page to print
	 * @param nEnd the 0-based page number of the end page to print
	 * @param bSilent if <code>true</code>, does not display the cancel dialog during the printing process
	 * @param bShrinkToFit if <code>true</code>, shrinks the page (if necessary) to fit within the
	 * imageable area of the printed page 
	 * @param bPrintAsImage if <code>true</code>, prints each page as an image
	 * @param bReverse if <code>true</code>, prints the pages in reverse order
	 * @param bAnnotation if <code>true</code>, prints all annotations
	 */
	public void print(boolean bUI, int nStart, int nEnd, boolean bSilent,
			boolean bShrinkToFit, boolean bPrintAsImage, boolean bReverse,
			boolean bAnnotation) {
		throw unimplemented("print", false);
	}

	/**
	 * Displays a dialog box containing a question and an entry field for the user
	 * to reply to the question.
	 * This method is only available for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.response()</code> scripting function
	 * is to be supported.
	 * @param sQuestion a string representing a question for the user
	 * @param sTitle a string representing the title that appears in the title bar of the dialog box
	 * @param sDefault a string representing the default vlaue for the answer to the question
	 * @param bPassword if <code>true</code>, masks the user's password with asterisks
	 * @return a string representing the user's answer. If the user presses the cancel button on
	 * the dialog box, the answer is <code>null</code>
	 */
	public String response(String sQuestion, String sTitle /* = "" */,
			String sDefault /* = "" */, boolean bPassword /* = false */) {
		throw unimplemented("response", false);
	}

	/**
	 * Returns the form object that currently has the input focus.
	 * The method is only available for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.getFocus()</code> scripting function
	 * is to be supported.
	 * @return the form object that currently has the input focus, or <code>null</code>
	 * if no form object has the input focus
	 */
	public Obj getFocus() {
		throw unimplemented("getFocus", false);
	}

	/**
	 * Sets the focus to the form object specified by the reference syntax expression.
	 * This method is only available for client applications.
	 * When <code>sSOMExpr</code> is null or empty, <code>setFocus</code> performs
	 * a clear focus operation.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.setFocus()</code> scripting function
	 * is to be supported.
	 * @param sSOMExpr a fully qualified reference syntax expression for the form object
	 */
	public void setFocus(String sSOMExpr) {
		throw unimplemented("setFocus", false);
	}

	/**
	 * Sets the focus to the form object specified by the parameter
	 * This method is only available for client applications.
	 * When <code>node</code> is null or empty, <code>setFocus</code> performs
	 * a clear focus operation.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.setFocus()</code> scripting function
	 * is to be supported.
	 * @param node the form node to set focus to
	 */
	public void setFocus(Node node) {
		throw unimplemented("setFocus", false);
	}

	/**
	 * Moves to the next page of a form.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.pageDown()</code> scripting function
	 * is to be supported.
	 */
	public void pageDown() {
		throw unimplemented("pageDown", false);
	}

	/**
	 * Moves to the next previous of a form.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.pageUp()</code> scripting function
	 * is to be supported.
	 */
	public void pageUp() {
		throw unimplemented("pageUp", false);
	}

	/**
	 * Opens the drop-down list specified by the reference syntax expression.
	 * This method is only available for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.openList()</code> scripting function
	 * is to be supported.
	 * @param sSOMExpr a fully qualified reference syntax expression that specifies a drop-down list
	 */
	public void openList(String sSOMExpr) {
		throw unimplemented("openList", false);
	}

	/**
	 * Opens the drop-down list specified by the parameter
	 * This method is only available for client applications.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.openList()</code> scripting function
	 * is to be supported.
	 * @param node the drop-down list form object to open
	 */
	public void openList(Node node) {
		throw unimplemented("openList", false);
	}

	/**
	 * @exclude from public api.
	 */
	public String metaData(int nOutputType) {
		
		// JavaPort: It isn't clear what this is meant to do. The XFA and Acrobat plugin
		//           implementations always return an empty string.		
		return "";
	}
	
	/**
	 * This method is called before any attempt to call the <code>$host.importData()</code>
	 * scripting method. The implementation should check the that all <code>DataModel</code>
	 * nodes (excluding any data description nodes) are not locked.
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.importData()</code> scripting function
	 * is to be supported.
	 * @return <code>true</code> if the call to the <code>importData()</code> scripting
	 * method should be allowed to proceed.
	 * @see #importData(String) 
	 */
	public boolean importDataPermsCheck() {
		throw unimplemented("importDataPermsCheck", false);
	}

	/**
	 * This method is called before any attempt to call the <code>$host.resetData()</code>
	 * scripting method. If <code>nodes</code> is empty, all nodes in <code>FormModel</code> derived from
	 * <code>Container</code> (as well as <code>FormModel</code>) should be checked to ensure that neither
	 * they nor any of their children have an ancestor that is locked. 
	 * If <code>nodes</code> is not empty, the same test is applied to
	 * the nodes identified by the fully qualified reference expressions in <code>nodes</code>
	 * <p/>
	 * This method must be overridden in derived classes if the <code>$host.resetData()</code> scripting function
	 * is to be supported.
	 * @param nodes a list of fully qualified reference expressions of nodes to check
	 * @return <code>true</code> if the call to the <code>resetData()</code> scripting
	 * method should be allowed to proceed.
	 * @see #resetData(List) 
	 */
	public boolean resetDataPermsCheck(List<String> nodes) {
		throw unimplemented("resetDataPermsCheck", false);
	}
	

	private ExFull unimplemented(String sName, boolean bProperty) {
		int nResId = bProperty ? ResId.InvalidPropertyException : ResId.InvalidMethodException;
		MsgFormatPos oMessage = new MsgFormatPos(nResId);
		oMessage.format("$host");
		oMessage.format(sName);
		return new ExFull(oMessage);
	}
}
