package org.json;

import java.util.Iterator;

/*
Copyright (c) 2008 JSON.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

The Software shall be used for Good, not Evil.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

/**
 * This provides static methods to convert an XML text into a JSONArray or
 * JSONObject, and to covert a JSONArray or JSONObject into an XML text using
 * the JsonML transform.
 *
 * @author JSON.org
 * @version 2009-02-11
 * @deprecated Use javax.json or other other json libraries instead
*/
@Deprecated
public class JSONML {

    private static Object parse(XMLTokener x, boolean arrayForm, JSONArray ja) throws JSONException {
        String closeTag = null;
        JSONArray newja = null;
        JSONObject newjo = null;
        String tagName = null;

        while (true) {
            while (x.more()) {
                Object token = x.nextContent();
                if (token == XML.LT) {
                    token = x.nextToken();
                    if (token instanceof Character) {
                        if (token == XML.SLASH) {
                            token = x.nextToken();
                            if (!(token instanceof String)) {
                                throw new JSONException("Expected a closing name instead of '" + token + "'.");
                            }

                            if (x.nextToken() != XML.GT) {
                                throw x.syntaxError("Misshaped close tag");
                            }

                            return token;
                        }

                        if (token != XML.BANG) {
                            if (token != XML.QUEST) {
                                throw x.syntaxError("Misshaped tag");
                            }

                            x.skipPast("?>");
                        } else {
                            char c = x.next();
                            if (c == '-') {
                                if (x.next() == '-') {
                                    x.skipPast("-->");
                                }

                                x.back();
                            } else if (c == '[') {
                                token = x.nextToken();
                                if (!token.equals("CDATA") || x.next() != '[') {
                                    throw x.syntaxError("Expected 'CDATA['");
                                }

                                if (ja != null) {
                                    ja.put(x.nextCDATA());
                                }
                            } else {
                                int i = 1;

                                while (true) {
                                    token = x.nextMeta();
                                    if (token == null) {
                                        throw x.syntaxError("Missing '>' after '<!'.");
                                    }

                                    if (token == XML.LT) {
                                        ++i;
                                    } else if (token == XML.GT) {
                                        --i;
                                    }

                                    if (i <= 0) {
                                        break;
                                    }
                                }
                            }
                        }
                    } else {
                        if (!(token instanceof String)) {
                            throw x.syntaxError("Bad tagName '" + token + "'.");
                        }

                        tagName = (String) token;
                        newja = new JSONArray();
                        newjo = new JSONObject();
                        if (arrayForm) {
                            newja.put(tagName);
                            if (ja != null) {
                                ja.put(newja);
                            }
                        } else {
                            newjo.put("tagName", tagName);
                            if (ja != null) {
                                ja.put(newjo);
                            }
                        }

                        token = null;

                        while (true) {
                            if (token == null) {
                                token = x.nextToken();
                            }

                            if (token == null) {
                                throw x.syntaxError("Misshaped tag");
                            }

                            if (!(token instanceof String)) {
                                if (arrayForm && newjo.length() > 0) {
                                    newja.put(newjo);
                                }

                                if (token == XML.SLASH) {
                                    if (x.nextToken() != XML.GT) {
                                        throw x.syntaxError("Misshaped tag");
                                    }

                                    if (ja == null) {
                                        if (arrayForm) {
                                            return newja;
                                        }

                                        return newjo;
                                    }
                                } else {
                                    if (token != XML.GT) {
                                        throw x.syntaxError("Misshaped tag");
                                    }

                                    closeTag = (String) parse(x, arrayForm, newja);
                                    if (closeTag != null) {
                                        if (!closeTag.equals(tagName)) {
                                            throw x.syntaxError("Mismatched '" + tagName + "' and '" + closeTag + "'");
                                        }

                                        tagName = null;
                                        if (!arrayForm && newja.length() > 0) {
                                            newjo.put("childNodes", newja);
                                        }

                                        if (ja == null) {
                                            if (arrayForm) {
                                                return newja;
                                            }

                                            return newjo;
                                        }
                                    }
                                }
                                break;
                            }

                            String attribute = (String) token;
                            if (!arrayForm && (attribute == "tagName" || attribute == "childNode")) {
                                throw x.syntaxError("Reserved attribute.");
                            }

                            token = x.nextToken();
                            if (token == XML.EQ) {
                                token = x.nextToken();
                                if (!(token instanceof String)) {
                                    throw x.syntaxError("Missing value");
                                }

                                newjo.accumulate(attribute, XML.stringToValue((String) token));
                                token = null;
                            } else {
                                newjo.accumulate(attribute, "");
                            }
                        }
                    }
                } else if (ja != null) {
                    ja.put(token instanceof String ? XML.stringToValue((String) token) : token);
                }
            }

            throw x.syntaxError("Bad XML");
        }
    }

    /**
     * Convert a well-formed (but not necessarily valid) XML string into a
     * JSONArray using the JsonML transform. Each XML tag is represented as
     * a JSONArray in which the first element is the tag name. If the tag has
     * attributes, then the second element will be JSONObject containing the
     * name/value pairs. If the tag contains children, then strings and
     * JSONArrays will represent the child tags.
     * Comments, prologs, DTDs, and <code>&lt;[ [ ]]></code> are ignored.
     * @param string The source string.
     * @return A JSONArray containing the structured data from the XML string.
     * @throws JSONException Thrown on error converting to a JSONArray
     */
    public static JSONArray toJSONArray(String string) throws JSONException {
        return toJSONArray(new XMLTokener(string));
    }

    /**
     * Convert a well-formed (but not necessarily valid) XML string into a
     * JSONArray using the JsonML transform. Each XML tag is represented as
     * a JSONArray in which the first element is the tag name. If the tag has
     * attributes, then the second element will be JSONObject containing the
     * name/value pairs. If the tag contains children, then strings and
     * JSONArrays will represent the child content and tags.
     * Comments, prologs, DTDs, and <code>&lt;[ [ ]]></code> are ignored.
     * @param x An XMLTokener.
     * @return A JSONArray containing the structured data from the XML string.
     * @throws JSONException Thrown on error converting to a JSONArray
     */
    public static JSONArray toJSONArray(XMLTokener x) throws JSONException {
        return (JSONArray) parse(x, true, (JSONArray) null);
    }

    /**
     * Convert a well-formed (but not necessarily valid) XML string into a
     * JSONObject using the JsonML transform. Each XML tag is represented as
     * a JSONObject with a "tagName" property. If the tag has attributes, then
     * the attributes will be in the JSONObject as properties. If the tag
     * contains children, the object will have a "childNodes" property which
     * will be an array of strings and JsonML JSONObjects.

     * Comments, prologs, DTDs, and <code>&lt;[ [ ]]></code> are ignored.
     * @param x An XMLTokener of the XML source text.
     * @return A JSONObject containing the structured data from the XML string.
     * @throws JSONException Thrown on error converting to a JSONObject
     */
    public static JSONObject toJSONObject(XMLTokener x) throws JSONException {
        return (JSONObject) parse(x, false, (JSONArray) null);
    }

    /**
     * Convert a well-formed (but not necessarily valid) XML string into a
     * JSONObject using the JsonML transform. Each XML tag is represented as
     * a JSONObject with a "tagName" property. If the tag has attributes, then
     * the attributes will be in the JSONObject as properties. If the tag
     * contains children, the object will have a "childNodes" property which
     * will be an array of strings and JsonML JSONObjects.

     * Comments, prologs, DTDs, and <code>&lt;[ [ ]]></code> are ignored.
     * @param string The XML source text.
     * @return A JSONObject containing the structured data from the XML string.
     * @throws JSONException Thrown on error converting to a JSONObject
     */
    public static JSONObject toJSONObject(String string) throws JSONException {
        return toJSONObject(new XMLTokener(string));
    }

    /**
     * Reverse the JSONML transformation, making an XML text from a JSONArray.
     * @param ja A JSONArray.
     * @return An XML string.
     * @throws JSONException Thrown on error converting to a string
     */
    public static String toString(JSONArray ja) throws JSONException {
        StringBuffer sb = new StringBuffer();
        String tagName = ja.getString(0);
        XML.noSpace(tagName);
        tagName = XML.escape(tagName);
        sb.append('<');
        sb.append(tagName);
        Object object = ja.opt(1);
        int i;
        if (object instanceof JSONObject) {
            i = 2;
            JSONObject jo = (JSONObject) object;
            Iterator<String> keys = jo.keys();

            while (keys.hasNext()) {
                String key = keys.next().toString();
                XML.noSpace(key);
                String value = jo.optString(key);
                if (value != null) {
                    sb.append(' ');
                    sb.append(XML.escape(key));
                    sb.append('=');
                    sb.append('"');
                    sb.append(XML.escape(value));
                    sb.append('"');
                }
            }
        } else {
            i = 1;
        }

        int length = ja.length();
        if (i >= length) {
            sb.append('/');
            sb.append('>');
        } else {
            sb.append('>');

            do {
                object = ja.get(i);
                ++i;
                if (object != null) {
                    if (object instanceof String) {
                        sb.append(XML.escape(object.toString()));
                    } else if (object instanceof JSONObject) {
                        sb.append(toString((JSONObject) object));
                    } else if (object instanceof JSONArray) {
                        sb.append(toString((JSONArray) object));
                    }
                }
            } while (i < length);

            sb.append('<');
            sb.append('/');
            sb.append(tagName);
            sb.append('>');
        }

        return sb.toString();
    }

    /**
     * Reverse the JSONML transformation, making an XML text from a JSONObject.
     * The JSONObject must contain a "tagName" property. If it has children,
     * then it must have a "childNodes" property containing an array of objects.
     * The other properties are attributes with string values.
     * @param jo A JSONObject.
     * @return An XML string.
     * @throws JSONException Thrown on error converting to a string
     */
    public static String toString(JSONObject jo) throws JSONException {
        StringBuffer sb = new StringBuffer();
        String tagName = jo.optString("tagName");
        if (tagName == null) {
            return XML.escape(jo.toString());
        } else {
            XML.noSpace(tagName);
            tagName = XML.escape(tagName);
            sb.append('<');
            sb.append(tagName);
            Iterator<String> keys = jo.keys();

            while (keys.hasNext()) {
                String key = keys.next().toString();
                if (!key.equals("tagName") && !key.equals("childNodes")) {
                    XML.noSpace(key);
                    String value = jo.optString(key);
                    if (value != null) {
                        sb.append(' ');
                        sb.append(XML.escape(key));
                        sb.append('=');
                        sb.append('"');
                        sb.append(XML.escape(value));
                        sb.append('"');
                    }
                }
            }

            JSONArray ja = jo.optJSONArray("childNodes");
            if (ja == null) {
                sb.append('/');
                sb.append('>');
            } else {
                sb.append('>');
                int length = ja.length();

                for (int i = 0; i < length; ++i) {
                    Object object = ja.get(i);
                    if (object != null) {
                        if (object instanceof String) {
                            sb.append(XML.escape(object.toString()));
                        } else if (object instanceof JSONObject) {
                            sb.append(toString((JSONObject) object));
                        } else if (object instanceof JSONArray) {
                            sb.append(toString((JSONArray) object));
                        } else {
                            sb.append(object.toString());
                        }
                    }
                }

                sb.append('<');
                sb.append('/');
                sb.append(tagName);
                sb.append('>');
            }

            return sb.toString();
        }
    }
}
