/*
 *
 *	File: OTFSWFFont4Description.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.TreeMap;
import java.util.TreeSet;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.SWFFont4Description;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;

class OTFSWFFont4Description extends SWFFont4Description
{
	private final OTFSWFFont3Description defineFont3Desc;
	private final OpenTypeFont theFont;

	OTFSWFFont4Description(OpenTypeFont theFont, boolean wasEmbedded) 
	throws InvalidFontException, UnsupportedFontException
	{
		this.theFont = theFont;
		this.defineFont3Desc = new OTFSWFFont3Description(theFont, wasEmbedded);
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#canDisplay(int)
	 */
	public boolean canDisplay(int c) 
	throws UnsupportedFontException, InvalidFontException 
	{
		try {
			return (theFont.cmap.unicodeChar2glyph(c) != 0);
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#getFirstChar()
	 */
	public int getFirstChar() 
	throws InvalidFontException, UnsupportedFontException 
	{
		return defineFont3Desc.getFirstChar();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#getLastChar()
	 */
	public int getLastChar() 
	throws InvalidFontException, UnsupportedFontException 
	{
		try {
			if (theFont.cmap.symbolSubtableOffset != -1)
				return theFont.cmap.getHighestMappedCode(theFont.cmap.symbolSubtableOffset, false);

			if (theFont.cmap.unicodeSubtableOffset != -1)
				return theFont.cmap.getHighestMappedCode(theFont.cmap.unicodeSubtableOffset, false);
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}

		// Fonts that have only legacy encoded cmaps often set their os2 lastchar
		// to be the last character in some legacy encoding. In these cases, the
		// only way to figure out the last char is to look up the unicode
		// chars until you find one that works, converting all those characters as
		// you go. Since the expected use of this api is to prevent running characters,
		// we aren't saving anything by doing that....so we will just return 0xffff.
		return 0xffff;
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#getPermissions()
	 */
	public Permission getPermissions() 
	throws InvalidFontException, UnsupportedFontException 
	{
		return defineFont3Desc.getPermissions();
	}

	public String getFamily() 
	throws InvalidFontException, UnsupportedFontException
	{
		return defineFont3Desc.getFamily();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#getSubFamily()
	 */
	public String getSubFamily() 
	throws InvalidFontException, UnsupportedFontException 
	{
		return defineFont3Desc.getSubFamily();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#isBold()
	 */
	public boolean isBold() 
	throws InvalidFontException, UnsupportedFontException 
	{
		return defineFont3Desc.isBold();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#isItalic()
	 */
	public boolean isItalic() 
	throws InvalidFontException, UnsupportedFontException 
	{
		return defineFont3Desc.isItalic();
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#streamFontData(java.util.Iterator, java.io.OutputStream)
	 */
	public void streamFontData(Iterator codepoints, OutputStream stream)
	throws InvalidFontException, UnsupportedFontException, IOException 
	{
		Subset subset = new OTSubset(theFont, true);
		TreeSet cps = new TreeSet();
		boolean includeVariationCmap = false;

		if (theFont.cmap != null)
		{
			// add all of the mappings reachable with the input codepoints
			// to cps and allGids.
			// sort first to make cmap smaller
			ArrayList sortedCodepoints = new ArrayList();
			while (codepoints.hasNext())
				sortedCodepoints.add(codepoints.next());
			Collections.sort(sortedCodepoints);
			codepoints = sortedCodepoints.iterator();
			includeVariationCmap = theFont.cmap.gatherPossibleMappings(codepoints, cps, subset);
		}

		TreeMap lookups = null;

		if (theFont.gsub != null)
		{
			GsubHarvester harvester = new GsubHarvester(theFont.gsub, theFont.getNumGlyphs());
			lookups = harvester.gatherPossibleLookups(subset);	
		}

		theFont.subsetAndStreamForSWFEmbedding(subset, stream, lookups, cps, includeVariationCmap);
	}

	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.SWFFont4Description#streamFontData(java.io.OutputStream)
	 */
	public void streamFontData(OutputStream stream)
	throws InvalidFontException, UnsupportedFontException, IOException 
	{
		theFont.streamSFNTForSWFEditting(stream);
	}

}
