/*
* File: CroppedImageInputByteStream.java
*
*************************************************************************
** ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 2012 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
  from Adobe Systems Incorporated.
**************************************************************************/

package com.adobe.internal.io.stream;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
/**
 * This class is used to crop an image input stream to specified bounds.
 * This particularly is helpful for those images which are very huge. Because we never
 * load the complete image in memory at any moment. 
 * There are some methods which are either not supported or not follow the contract of {@link InputByteStream}. 
 * So, clients are recommended to read the javadocs of individual API's of this class before using them.
 * @author hkansal
 *
 */
public class CroppedImageInputByteStream implements InputByteStream{

	private InputStream decodedImage;
	private int oldBytesPerRow, newBytesPerRow;
	private byte[] bufferedCroppedRow;
	private int bufferPos = 0;
	private int rowOffset;
	private long length;
	private long totalBytesRead = 0;
	private boolean shiftBits = false;
	private ByteArrayOutputStream os = null;
	private int references = 1;
	
	/**
	 * Constructor to create an instance of this InputByteStream which reads subimage of dimension (x2-x1)*(y2-y1), of
	 * specified bits per component and number of components, from passed {@link InputStream}
	 * @param bpc bits per component of image
	 * @param noc number of components
	 * @param decodedImage image input stream
	 * @param width image width
	 * @param height image height
	 * @param x1 left
	 * @param y1 bottom
	 * @param x2 right
	 * @param y2 top
	 * @throws IOException
	 */
	public CroppedImageInputByteStream(int bpc, int noc, InputStream decodedImage, int width, int height, int x1,int y1, int x2, int y2) throws IOException{
		this.decodedImage = decodedImage;
		oldBytesPerRow = (int) Math.ceil(width*noc*bpc*1.0/Byte.SIZE);
		newBytesPerRow = (int) Math.ceil((x2-x1)*noc*bpc*1.0/Byte.SIZE);
		os = new ByteArrayOutputStream();
		for(int i=0;i<(height-y2);i++){
			IO.copy(decodedImage, oldBytesPerRow, os);
			os.reset();
		}
		rowOffset = x1*noc*bpc;
		if(rowOffset%Byte.SIZE == 0)// offset doesn't start from within a byte.
			rowOffset = rowOffset/Byte.SIZE;
		else
			shiftBits = true;
		bufferPos = newBytesPerRow;
		length = (y2-y1)*newBytesPerRow;
		
	}
	private void fillRow() throws IOException{
		if(bufferedCroppedRow == null)
			bufferedCroppedRow = new byte[newBytesPerRow];
		IO.copyInternal(decodedImage, oldBytesPerRow, os, false);
		os.flush();
		byte[] row = os.toByteArray();
		
		os.reset();
		if(!shiftBits){// offset doesn't start from within a byte.
			System.arraycopy(row, rowOffset, bufferedCroppedRow, 0, newBytesPerRow);
		}else{// we need some bits (not all) from first byte so we need to shift bits.
			shiftAndWriteBits(row, rowOffset, newBytesPerRow);
		}
		bufferPos = 0;
	}
	
	private void shiftAndWriteBits(byte[] data, int shift, int length) throws IOException{
		int start = shift/Byte.SIZE;
		int offset = shift%8;
		for(int i=0;i<length-1;i++){
			bufferedCroppedRow[i] = (byte) ((data[start]<<(offset)) | (data[++start]&(0xff>>(Byte.SIZE - offset))));
		}
		bufferedCroppedRow[length-1] = (byte) (data[start]<<(offset));
	}
	
	/**
	 * Not supported
	 */
	@Override
	public InputByteStream slice(long begin, long length)
			throws IOException {
		throw new IOException("slice() on CroppedImageInputByteStream not supported.");
	}

	/**
	 * Returns itself.
	 */
	@Override
	public InputByteStream slice() throws IOException {
		this.references++;
		return this;
	}

	@Override
	public int read() throws IOException {
		if(bufferPos >= newBytesPerRow)
			fillRow();
		return bufferedCroppedRow[bufferPos];
	}
	
	@Override
	public int read(byte[] bytes, int position, int length)
			throws IOException {
		int totalRead = 0;
		int bytesToRead = 0;
		if(totalBytesRead >=this.length)
			return -1;// end of stream reached.
		while(totalRead < length && totalBytesRead < this.length){
			if(bufferPos >= newBytesPerRow){
				fillRow();// get the image row.
			}
			bytesToRead = Math.min(length-totalRead, newBytesPerRow-bufferPos);
			System.arraycopy(bufferedCroppedRow, bufferPos, bytes, position + totalRead, bytesToRead);
			bufferPos+=bytesToRead;
			totalRead+=bytesToRead;
			totalBytesRead+=bytesToRead;
		}
		return totalRead;
	}

	@Override
	public int read(byte[] bytes) throws IOException {
		return read(bytes, 0, bytes.length);
	}

	/**
	 * Not supported
	 */
	@Override
	public int unget() throws IOException {
		throw new IOException("unget() on CroppedImageInputByteStream not supported.");
	}

	/**
	 * Not supported
	 */
	@Override
	public InputByteStream seek(long position) throws IOException {
		if(position == getPosition())
			return this;
		throw new IOException("seek() on CroppedImageInputByteStream not supported.");
	}

	@Override
	public long getPosition() throws IOException {
		return totalBytesRead;
	}

	@Override
	public long length() throws IOException {
		return length;
	}

	@Override
	public long bytesAvailable() throws IOException {
		return length - totalBytesRead;
	}

	@Override
	public boolean eof() throws IOException {
		return length == totalBytesRead;
	}

	@Override
	public void close() throws IOException {
		if(--this.references != 0)
			return;
		try{
			if(decodedImage != null)
				decodedImage.close();
		}finally{
			if(os != null)
				os.close();
		}
		bufferedCroppedRow = null;
	}

	/**
	 * Returns an instance of {@link InputStream} which uses this stream only
	 * as back-end. So, closing one of them will close the other also. 	 
	 */
	@Override
	public InputStream toInputStream() throws IOException {
		this.references++;
		return new InputStreamImpl(this);
	}
	
}
