/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.template;

import org.xml.sax.Attributes;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Element;
import com.adobe.xfa.Model;
import com.adobe.xfa.ModelFactory;
import com.adobe.xfa.ModelPeer;
import com.adobe.xfa.Node;
import com.adobe.xfa.STRS;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;


/**
 * A class to represent XFA template model factories.
 * A template model factory is used by an application model
 * to create a template model where appropriate, when loading an XML file.
 */
public final class TemplateModelFactory extends ModelFactory {
	
	private boolean mbApplyFixups;
	private boolean mbFixupRenderCache;
	private TemplateResolver moTemplateResolver;
	
	/**
	 * Instantiates a template model factory.
	 */
	public TemplateModelFactory() {
		super(XFA.TEMPLATETAG, XFA.TEMPLATE, STRS.DOLLARTEMPLATE);
	}

	/**
	 * Indicates whether to apply fixups on the template model after
	 * it's been loaded.
	 * @param bApply apply fixups when true.
	 */
	public void applyFixups(boolean bApply) {
		mbApplyFixups = bApply;
	}
	
	/**
	 * @exclude from published api.
	 */
	public Model createDOM(Element parent) {
		
		Model tm = new TemplateModel(parent.getAppModel(), null);
		
		ModelPeer modelPeer = new ModelPeer(
				parent, null, 
				STRS.XFATEMPLATENS_CURRENT, XFA.TEMPLATE, XFA.TEMPLATE, 
				null, tm);
		
		tm.setXmlPeer(modelPeer);
		tm.connectPeerToDocument();

		return tm;
	}
	
	/**
	 * @exclude from published api.
	 */
	protected Model newModel(AppModel parent, Node prevSibling, String uri, String localName,
			String qName, Attributes a) {

		// JavaPort: Use a different constructor from C++ -- otherwise we initialize our version 
		// number with the default uri instead of the actual uri
		TemplateModel tm = new TemplateModel(parent, prevSibling, uri, qName, localName);
		tm.setXmlPeer(new ModelPeer(
				(Element)parent.getXmlPeer(), null, 
				uri, localName, qName, 
				a, tm));
		tm.setDOMProperties(uri, localName, qName, null);
		
		if (moTemplateResolver != null) {
			tm.setTemplateResolver(moTemplateResolver);
		}
		if (mbApplyFixups)
			tm.applyFixups();
		
		if (mbFixupRenderCache)
			tm.fixupRenderCache();
		return tm; 
	}

	/**
	 * @see ModelFactory#setOption(String, String, boolean)
	 * @exclude
	 */
	public void setOption(String optionName, String optionValue,
			boolean bCritical) {
		// Check for prefix, and remove it.
		String name = optionName;
		int nPackageSeparatorOffset = optionName.indexOf('_');
		if (nPackageSeparatorOffset != -1) {
			if (nPackageSeparatorOffset == 0)
				throw new ExFull(ResId.OptionWrongPackageException, optionName);

			// package specified; verify that it's the right one
			String packageName = optionName.substring(0,
					nPackageSeparatorOffset);
			String oPackageName = packageName.intern();
			if (!isRootName(oPackageName))
				// wrong package
				throw new ExFull(ResId.OptionWrongPackageException, optionName);

			// package is OK; remove it from the option name
			name = optionName.substring(nPackageSeparatorOffset + 1);
		}

		if (name.equals("removeRenderCache")) {
			mbFixupRenderCache = optionValue.equals("1");
		} else {
			super.setOption(optionName, optionValue, bCritical);
		}
	}

	/**
	 * @exclude from published api.
	 */
	public void setTemplateResolver(TemplateResolver oResolver) {
		moTemplateResolver = oResolver;
	}

}
